
/*--------------------------------------------------------------------*/
/*--- Redux: printing output                            rx_print.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Redux, a Valgrind skin for tracing program
   execution.

   Copyright (C) 2003 Nicholas Nethercote
      njn25@cam.ac.uk

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "rx_include.h"

typedef
   enum {
      NodeNone=0, NodeConstString=1, NodeSyscall=2,
      NodeSplit=3, NodeExit=4, NodeSysMem=5, NodeRead=6, NodeCond=7,
      NodeFn=8, NodeAlloc=9
   } NodeStyle;

static Char* dot_nodestyles[] = { 
   /*0*/ "",
   /*1*/ "",
   /*2*/ ",color=black", //",color=red",
   /*3*/ ",style=dotted",
   /*4*/ ",shape=plaintext",
   /*5*/ ",style=dashed",
   /*6*/ ",style=dotted",
   /*7*/ ",style=dotted",
   /*8*/ ",color=blue",
   /*9*/ ",color=black", //"",
};
static Char* aiSee_nodestyles[] = { 
   /*0*/ "",
   /*1*/ " color:lightblue",
   /*2*/ " color:red",
   /*3*/ " borderstyle:dotted",
   /*4*/ " color:lightred",
   /*5*/ " color:yellowgreen",
   /*6*/ " color:lightgrey",
   /*7*/ " color:pink",
   /*8*/ " color:blue",
   /*9*/ " color:brown",
};

typedef
   enum {
      EdgeNone=0,
      EdgeSysMem=1,           // SysMem --> syscall
      EdgeMemArg=2,
      EdgeCond=3, 
      EdgeSyscallConnect=4,   // sequence edges between syscalls
      EdgeCC2Val=5,
      EdgeCondConnect=6,      // sequence edges between conds
      EdgeIncDecChain=7,
   } EdgeStyle;

static Char* dot_edgestyles[] = { 
   /*0*/ "",
   /*1*/ "style=dashed",
   /*2*/ "style=dashed",
   /*3*/ "style=dotted",
   /*4*/ "style=dotted",
   /*5*/ "style=dashed",
   /*6*/ "style=dotted",
   /*7*/ "style=dashed",
};
static Char* aiSee_edgestyles[] = { 
   /*0*/ "",
   /*1*/ "color:yellowgreen",
   /*2*/ "color:lightblue",
   /*3*/ "color:white",
   /*4*/ "color:red",
   /*5*/ "linestyle:dashed",
   /*6*/ "color:pink",
   /*7*/ "linestyle:dotted",
};

/*--------------------------------------------------------------------*/
/*--- Pretty printing of shadow words                              ---*/
/*--------------------------------------------------------------------*/

static Bool sane_kind(ShKind kind)
{
   return (SpError < kind && kind < ShLast);
}

static Char* ConstOrigin_to_string(ConstOrigin origin)
{
   switch (origin) {
   case ConstError:   VG_(skin_panic)("ConstError!");
   case ConstStartup: return "s";
   case ConstCode:    return "c";
   /*case ConstMmap:    return "mm";*/
   case ConstWiden:   return "w";
   case ConstZero:    return "z";
   case ConstLea1:    return "L1";
   case ConstLea2:    return "L2";
   default:           VG_(skin_panic)("Unexpected ConstOrigin");
   }
}

static Char* AllocKind_to_string(AllocKind akind)
{
   switch (akind) {
   case AllocError:    VG_(skin_panic)("AllocError!");
   case AllocMalloc:   return "malloc";
   case AllocNew:      return "new";
   case AllocVecNew:   return "new[]";
   case AllocMemAlign: return "memalign";
   case AllocCalloc:   return "calloc";
   case AllocRealloc:  return "realloc";
   default:            VG_(skin_panic)("Unexpected AllocKind");
   }
}

static Char* size_to_suffix(RxSize size)
{
   switch (size) {
   case 0: return "?"; 
   case 1: return "b";
   case 2: return "w";
   case 4: return "L";
   default: VG_(skin_panic)("Unexpected size");
   }
}

static UInt char_as_string(UInt k, Char* buf)
{
   switch (k) {
   case 32 ... 33:   
   case 35 ... 91:  
   case 93 ... 127: return VG_(sprintf)(buf, "%c", k); // normal char
   case  7:         return VG_(sprintf)(buf, "\\\\a"); // bell
   case  8:         return VG_(sprintf)(buf, "\\\\b"); // backspace
   case  9:         return VG_(sprintf)(buf, "\\\\t"); // horizontal tab
   case 10:         return VG_(sprintf)(buf, "\\\\n"); // newline
   case 11:         return VG_(sprintf)(buf, "\\\\v"); // vertical tab
   case 12:         return VG_(sprintf)(buf, "\\\\f"); // form feed
   case 13:         return VG_(sprintf)(buf, "\\\\r"); // carriage return
   case 34:         return VG_(sprintf)(buf, "\\\"");  // double quote
   case 92:         return VG_(sprintf)(buf, "\\\\");  // backslash
   default:         return VG_(sprintf)(buf, "0x%02x", k);  // 0x__
   }
}

static __inline__
UInt pr_val(UInt k, UInt size)
{
   if (1 == size) {
      Char buf[] = { '\'', '\0', '\0', '\0', '\0', '\0', '\0' };
      char_as_string(k, buf+1);
      if ('0' == buf[1])
         return VG_(printf)(buf+1);    // 0x__ one, no quotes needed
      else {
         Int i;
         for (i = 1; '\0' != buf[i]; i++) { }
         buf[i] = '\'';
         return VG_(printf)(buf);
      }
   } 
   else if (size == 2 || k > 0x8000000)
      return VG_(printf)("%p%s", k, size_to_suffix(size));
   else
      return VG_(printf)("%u%s", k, size_to_suffix(size));
}

void RX_(up_ShW)(ShW* sh)
{
   VG_(printf)(".kind:        %u\n", sh->kind);
   VG_(printf)(".extra:       %u\n", sh->extra);
   VG_(printf)(".argc:        %u\n", sh->argc);
   VG_(printf)(".size         %u\n", sh->size);
   VG_(printf)(".n_parents:   %u\n", sh->n_parents);
   VG_(printf)(".rewritten:   %u\n", sh->rewritten);
   VG_(printf)(".graphed:     %u\n", sh->graphed);
   VG_(printf)(".val:         %u\n", sh->val);
#ifdef RX_SLICING
   VG_(printf)(".instr_addr:  %p\n", sh->instr_addr);
#endif
   // print addresses of first three args, if present;  no more than that in
   // case argc is bad.
   if (sh->argc >= 1) VG_(printf)(".argv[0]:     %p,%d,%d\n", sh->argv[0],sh->argv[0]->size,sh->argv[0]->kind);
   if (sh->argc >= 2) VG_(printf)(".argv[1]:     %p,%d,%d\n", sh->argv[1],sh->argv[1]->size,sh->argv[1]->kind);
   if (sh->argc >= 3) VG_(printf)(".argv[2]:     %p\n", sh->argv[2]);
}

Char* name_of_ShW(ShW* sh)
{
#define err "This should never be seen -- indicates a bug!"
   
   switch (sh->kind) {
   case SpError:      return err;      // should be never printed
   case SpStartup:    return err;      // only for laziness
   case SpHeap:       return "heap??";
   case SpZeroHeap:   return "calloc.0L";
   case SpBrk:        return "brk";
   case SpUnknownMem: return "M??";
   case SpUnknownReg: return "R??";
   case SpSyscallReg: return "syscallReg!!";
   case SpSignalReg:  return "sigR??";
   case SpPthOpReg:   return "pthOpReg!!";
   case SpClReqReg:   return "clReqR??";
   case SpRep:        return "REP";
   case SpEspEbp:     return "%%e[sb]p";

   case ShConst:      return err;      // ConstOrigin used instead
   case ShAdd:        return "+";
   case ShSub:        return "-";
   case ShMul:        return "*";
   case ShMul64High:  return "*h";
   case ShMul64:      return "*l";
   case ShDiv64:      return "div";
   case ShMod64:      return "mod";
   case ShAnd:        return "&";
   case ShOr:         return "|";
   case ShXor:        return "^";
   case ShShl:        return "shl";
   case ShShr:        return "shr";
   case ShSar:        return "sar";
   case ShRol:        return "rol";
   case ShRor:        return "ror";
   case ShInc:        return "inc";
   case ShDec:        return "dec";
   case ShNot:        return "~";
   case ShNeg:        return "neg";
   case ShLea1Add:    return "l+";
   case ShLea2Add:    return "lea2";
   case ShCond:       return "j";      // prefix
   case ShCC2Val:     return "cc2val"; // used by pp_:  prefix for dot_
   case ShWiden:      return "widen";
   case ShShld:       return "shld";
   case ShShrd:       return "shrd";
   case ShSyscall:    return err;      // syscall name used instead
   case ShSysMem:     return err;      // mem output block offset used instead
   case ShSplit4B:    return "sp4B";
   case ShSplit2W:    return "sp2W";
   case ShSplit2B:    return "sp2B";
   case ShReadB:      return "B";      // used by pp_:  prefix for dot_
   case ShReadW:      return "W";      // used by pp_:  prefix for dot_
   case ShExit:       return "_exit";
   case ShChunk:      return "chunk";
   case ShString:     return "str";
   case ShInd:        return err;      // should not be printed
   case ShAlloc:      return err;      // AllocKind used instead
   default:           VG_(printf)("bad kind = %d\n", sh->kind);
                      VG_(skin_panic)("name_of_ShW: bad kind");
   }
}

// Forward declaration
static void pp_ShW_x(ShW* sh, UInt depth);

static
void pp_UnOp(Char* name, ShW* sh, UInt depth)
{
   VG_(printf)("%s(", name);
   pp_ShW_x(sh->argv[0], depth);
   VG_(printf)(")");
}

static
void pp_BinOp(Char* name, ShW* sh, UInt depth)
{
   VG_(printf)("(");
   pp_ShW_x(sh->argv[0], depth);
   VG_(printf)(" %s ", name);
   pp_ShW_x(sh->argv[1], depth);
   VG_(printf)(")");
}

static
void pp_TrinOp(Char* name, ShW* sh, UInt depth)
{
   VG_(printf)("(");
   pp_ShW_x(sh->argv[0], depth);
   VG_(printf)(" %s ", name);
   pp_ShW_x(sh->argv[1], depth);
   VG_(printf)("::", name);
   pp_ShW_x(sh->argv[2], depth);
   VG_(printf)(")");
}

static
void pp_QuartOp(Char* name, ShW* sh, UInt depth, Bool more_args)
{
   VG_(printf)("%s(", name);
   pp_ShW_x(sh->argv[0], depth);
   VG_(printf)(", ", name);
   pp_ShW_x(sh->argv[1], depth);
   VG_(printf)(", ", name);
   pp_ShW_x(sh->argv[2], depth);
   VG_(printf)(", ", name);
   pp_ShW_x(sh->argv[3], depth);
   if (more_args)
      VG_(printf)(", ...");
   VG_(printf)(")");
}

static
void pp_Op(ShW* sh, UInt depth)
{
   Char* s  = name_of_ShW(sh);

   switch ( sh->argc ) {
   case 0:  VG_(printf)(s);                        break;
   case 1:  pp_UnOp    (s, sh, depth);         break;
   case 2:  pp_BinOp   (s, sh, depth);         break;
   case 3:  pp_TrinOp  (s, sh, depth);         break;
   case 4:  pp_QuartOp (s, sh, depth, False);  break;
   default: pp_QuartOp (s, sh, depth, True);   break;
   }
}

// For printing unary ops that are often strung in long chains, eg. ++
static void pp_RepUnOp(Char* s, ShKind kind, ShW* sh, UInt depth)
{
   UInt n = 1;
   while (sh->kind == kind) {
      sh = sh->argv[0];
      n++;
   }
   if (1 == n)
      VG_(printf)("%s(", s);
   else
      VG_(printf)("%s[%u](", s, n);

   pp_ShW_x(sh, depth);
   VG_(printf)(")");
}

static void pp_ShW_x(ShW* sh, UInt depth)
{
   Char* s;
   
   sk_assert(sane_kind(sh->kind));

   if (depth >= RX_(clo_print_depth)) {
      VG_(printf)("...");
      return;
   }
   depth++;

   s = name_of_ShW(sh);
   
   switch (sh->kind) {

   case SpStartup:
      VG_(skin_panic)("pp'ing SpStartup node!");

   case ShConst:
      VG_(printf)("%s.", ConstOrigin_to_string(sh->extra));
      pr_val(sh->val, sh->size);
      break;

   case ShInc:
   case ShDec:
      pp_RepUnOp(s, ShDec, sh->argv[0], depth);
      break;

   case ShCond:
      VG_(printf)("%s%s(", s, VG_(name_UCondcode)(sh->extra));
      pp_ShW_x(sh->argv[0], depth);
      VG_(printf)(")");
      break;

   case ShSyscall: {
      UInt         syscallno = sh->extra;
      SyscallInfo* syscall   = & RX_(syscall_info)[syscallno];
      Int i;
      VG_(printf)("%s(", syscall->name);

      // Print args (but not memargs -- too hard in text)
      if (syscall->argc > 0) {
         for (i = 0; i < syscall->argc-1; i++) {
            VG_(printf)("%s=", syscall->argv[i]);
            pp_ShW_x(sh->argv[i], depth);
            VG_(printf)(", ");
         }
         VG_(printf)("%s=", syscall->argv[i]);
         pp_ShW_x(sh->argv[i], depth);  // last one without comma
         VG_(printf)(")");
      }
      break;
   }

   case ShSysMem:
      VG_(printf)("(+%u)(%p)", sh->extra, sh->val);
      break;

   case ShAlloc:
      VG_(printf)("%s.", AllocKind_to_string(sh->extra));
      pr_val(sh->val, sh->size);
      break;

   // All non-special cases
   default: 
      pp_Op(sh, depth);
      break;
   }
}

void RX_(pp_ShW)(ShW* sh)
{
   pp_ShW_x(sh, 0);
}

void RX_(ppp_ShW)(Char* s, ShW* sh)
{
   VG_(printf)("%s: ", s);
   RX_(pp_ShW)(sh);
   VG_(printf)("\n");
}
   
/*--------------------------------------------------------------------*/
/*--- Pre-pass for counting/resetting n_parents                    ---*/
/*--------------------------------------------------------------------*/

static PCount inc_PCount(PCount c)
{
   switch (c) {
   case PZero: return POne;
   case POne : 
   case PMany: return PMany;
   default:    VG_(skin_panic)("Unknown PCount value");
   }
}

static void pre_ShW(ShW* sh, Bool reset)
{
   Int i;
   
   sk_assert(sane_kind(sh->kind));

   if (!reset) {
      // Don't do more than once, but note if multiple parents
      sh->n_parents = inc_PCount(sh->n_parents);
      if (sh->n_parents == PMany) return;
   } else {
      // Don't do more than once, zero the .graphed field
      if ( ! sh->graphed) return;
      sh->graphed = False;
   }

   switch (sh->kind) {
   case SpStartup:
      VG_(skin_panic)("pre-processing SpStartup node!");

   // All non-special cases
   default:
      for (i = 0; i < sh->argc; i++)
         pre_ShW(sh->argv[i], reset);
      break;
   }
}

/*--------------------------------------------------------------------*/
/*--- Rewriting the graph                                          ---*/
/*--------------------------------------------------------------------*/

// Forward declaration
static void rew_ShW(ShW** sh_ptr, ShW* sh, ShW* parent);

// Replaces a binop with z.0L, eg. for xor(X,X)
static void rew_zero_shadow_binop(ShW* sh)
{
   sh->kind    = ShConst;
   sh->extra   = ConstZero;
   sh->val     = 0;
   sh->argc    = 0;
   sh->argv[0] = RX_(specials)[SpError];
   sh->argv[1] = RX_(specials)[SpError];
}

// Bypasses a boring op, eg. widen.  `arg' is the # of the interesting arg
static void rew_bypass(ShW** sh_ptr, ShW* sh, UInt arg)
{
   sh->rewritten = False;    // in case the node has >1 dependents
   * sh_ptr  = sh->argv[arg];
}

__attribute__((unused))
static void rew_convert_to_ShConst(ShW* sh, UInt extra, UInt val)
{
   Int i;
   
   sh->kind = ShConst;
   sh->extra = extra;
   for (i = 0; i < sh->argc; i++) {
      sh->argv[i] = RX_(specials)[SpError];
   }
   sh->val  = val;
   sh->argc = 0;
}
   
static void rew_convert_to_ShInd(ShW* sh, ShW* arg)
{
   Int i;
   
   sk_assert(sh->argc > 0);
   sh->kind    = ShInd;
   sh->extra   = 0;
   sh->argv[0] = arg;
   for (i = 1; i < sh->argc; i++) {
      sh->argv[i] = RX_(specials)[SpError];
   }
   sh->argc    = 1;
   sh->size    = Sz0;
   sh->val     = 0;
}

// inlining helps reduce stack consumption due to recursion
static __inline__ void rew_Op(ShW* sh)
{
   Int i;
   for (i = 0; i < sh->argc; i++)
      rew_ShW(& sh->argv[i], sh->argv[i], sh);
}

// XXX: some of these rewrites affect the PCount... does that matter?
static void rew_ShW(ShW** sh_ptr, ShW* sh, ShW* parent)
{
top:
   sk_assert(sane_kind(sh->kind));
   sk_assert(sh->n_parents != PZero);

   // Don't rewrite more than once
//   RX_(ppp_ShW)("rewriting", sh);
   if (sh->rewritten) { /*VG_(printf)("(already rewritten)");*/ return; }
   sh->rewritten = True;

   switch (sh->kind) {
   case SpStartup:
      VG_(skin_panic)("rewriting SpStartup node!");

   // xor(X,X), sub(X,X) ==> 0
   //
   // Nb: leaves X's dangling (usually single nodes, but could be any size)
   #if 0
   // NOW DOING THE ELISION AT NODE-BUILDING TIME
   case ShSub:
   case ShXor:
      if (sh->argv[0] == sh->argv[1]) {
         rew_zero_shadow_binop(sh);
      } else {
         rew_Op(sh);
      }
      break;
   #endif

   // div|mod(X, Y, sar(Y,31)==0L)   ==>   div|mod(X, Y)
   #if 0
   case ShDiv64:
   case ShMod64:
      if (ShSar   == sh->argv[2]->kind &&
          0       == sh->argv[2]->val  &&
          ShConst == sh->argv[2]->argv[1]->kind &&
          31      == sh->argv[2]->argv[1]->val  &&
          sh->argv[1] == sh->argv[2]->argv[0])
      {
         ShW* tmp;
         tmp = sh->argv[0];
         sh->argv[0] = sh->argv[1];
         sh->argv[1] = tmp;
         sh->argv[2] = RX_(specials)[SpError];
         sh->argc = 2;
         rew_Op(sh);
      } else {
         rew_Op(sh);
      }
      break;
   #endif


   // lea1(X,Y) ==> add(X,Y)
#if 0
   case ShLea1Add:
      sh->rewritten = False;       // because we redo it immediately
      sh->kind      = ShAdd;
      // avoid unnecessary frames on Valgrind's small stack
      goto top;   //rew_ShW(sh_ptr, sh, parent);
      break;
#endif

   // skip widenings
   case ShWiden:
      rew_bypass(sh_ptr, sh, 0);

      // avoid unnecessary frames on Valgrind's small stack
      parent = sh;
      sh = *sh_ptr;
      goto top;   //rew_ShW(sh_ptr, * sh_ptr, sh);
      break;

   // Bn( (m.L)(X) ) ==> m.B(X.B)   (cut out the Bn middle man)
   //
   //
   // XXX: is this being used, if it's being done at split time?
   //
   // 
   case ShReadB:
      if (ShSysMem == sh->argv[0]->kind) {
         sh->kind      = ShSysMem;
         sh->argv[0]   = sh->argv[0]->argv[0];
         // Nb: sh->size is already Sz1, what we want
      }
      rew_Op(sh);
      break;

   case ShSplit4B:
      // sp4(cc2val(X), B1(Z), B2(Z), B2(Z))  ==> cc2val(X):[01]L
      // where Z.val == 0
      if (ShCC2Val == sh->argv[0]->kind &&
          ShReadB  == sh->argv[1]->kind &&
          ShReadB  == sh->argv[2]->kind &&
          ShReadB  == sh->argv[3]->kind &&
          sh->argv[1]->argv[0] == sh->argv[2]->argv[0] &&
          sh->argv[2]->argv[0] == sh->argv[3]->argv[0])
      {
         rew_bypass(sh_ptr, sh, 0);
         // avoid unnecessary frames on Valgrind's small stack
         parent = sh;
         sh = *sh_ptr;
         goto top;   //rew_ShW(sh_ptr, * sh_ptr, sh);

      // sp4(c,c2,c3,c4) ==> const.(c|c2|c3|c4)
#if 0
      // XXX: do I want this?
      } else if (ShConst == sh->argv[0]->kind &&
                 ShConst == sh->argv[1]->kind &&
                 ShConst == sh->argv[2]->kind &&
                 ShConst == sh->argv[3]->kind &&
                 sh->argv[0]->extra == sh->argv[1]->extra &&
                 sh->argv[1]->extra == sh->argv[2]->extra &&
                 sh->argv[2]->extra == sh->argv[3]->extra)
      {
         // sh->val unchanged
         rew_convert_to_ShConst(sh, sh->argv[0]->extra, sh->val);
         rew_Op(sh);
#endif

      // sp4(B0(X),B1(X),B2(X),B3(X)) ==> ind(X)
      // (can't just replace sp4(...) with X because if something else
      //  points directly to the original X, there will be two Xs printed)
      } else if (ShReadB == sh->argv[0]->kind &&
                 ShReadB == sh->argv[1]->kind &&
                 ShReadB == sh->argv[2]->kind &&
                 ShReadB == sh->argv[3]->kind &&
                 sh->argv[0]->argv[0] == sh->argv[1]->argv[0] &&
                 sh->argv[1]->argv[0] == sh->argv[2]->argv[0] &&
                 sh->argv[2]->argv[0] == sh->argv[3]->argv[0])
      {
         rew_convert_to_ShInd(sh, sh->argv[0]->argv[0]);
         rew_Op(sh);
      } else {
         rew_Op(sh);
      }
      break;

   case ShSplit2W:
      // sp2W(c.w,c2.w) ==> const.((c|c2).L)
      // XXX: do I want this?
#if 0
      if (ShConst == sh->argv[0]->kind &&
          ShConst == sh->argv[1]->kind &&
          sh->argv[0]->extra == sh->argv[1]->extra)
      {
         // sh->val unchanged
         rew_convert_to_ShConst(sh, sh->argv[0]->extra, sh->val);
         rew_Op(sh);
#endif
      // sp2W(W0(X),W1(X)) ==> ind(X)
      /*} else*/ if (ShReadW == sh->argv[0]->kind &&
                 ShReadW == sh->argv[1]->kind &&
                 sh->argv[0]->argv[0] == sh->argv[1]->argv[0])
      {
         rew_convert_to_ShInd(sh, sh->argv[0]->argv[0]);
         rew_Op(sh);

      } else {
         rew_Op(sh);
      }
      break;

   case ShSplit2B:
      // sp2B(c.B,c2.B) ==> const.((c|c2).W)
      // XXX: do I want this?
#if 0
      if (ShConst == sh->argv[0]->kind &&
          ShConst == sh->argv[1]->kind &&
          sh->argv[0]->extra == sh->argv[1]->extra)
      {
         // sh->val unchanged
         rew_convert_to_ShConst(sh, sh->argv[0]->extra, sh->val);
         rew_Op(sh);
#endif
      /*} else*/ {
         rew_Op(sh);
      }
      break;

   // exit(+(X, c.0L)) ==> exit(X)
   case ShExit: {
      ShW* arg = sh->argv[0];
      if (ShLea1Add == arg->kind           &&
          ShConst   == arg->argv[1]->kind  &&
          ConstCode == arg->argv[1]->extra &&
          0         == arg->argv[1]->val) 
      {
         sh->argv[0] = sh->argv[0]->argv[0];
      }
      rew_Op(sh);
      break;
   }

   // All non-special cases
   default: 
      rew_Op(sh); 
      break;
   }
}


/*--------------------------------------------------------------------*/
/*--- Graph printing of shadow words                               ---*/
/*--------------------------------------------------------------------*/

// Note: although the data structure directionality comes from dependencies
// (ie. an operator points to its operands), we print the arrows out the
// other way, as data flow, because that's more intuitive, and because 'dot'
// only allows downwards arrows, that means 'exit' can be at the bottom,
// too.

// Forward declaration
static void gr_ShW_x(ShW* sh, UInt depth);

static __inline__
UInt gr_const(ShW* sh)
{
   UInt len = 0;
   sk_assert(ShConst == sh->kind);

   len += VG_(printf)("%s.", ConstOrigin_to_string(sh->extra));
   len += pr_val(sh->val, sh->size);
   return len;
}

// Inline all special nodes.
// Inline all unshared constants, and shared constants if --inline=all.
static __inline__ Bool gr_arg_is_inlineable(ShW* sh)
{
   sk_assert( 0 <= RX_(clo_inline) && RX_(clo_inline <= 2) );

   if ( 0 == RX_(clo_inline) ) {
      return False;

   } else if ( RX_(is_special_node)(sh->kind) ) {
      return True;

   } else if ( ShConst == sh->kind ) {
      if ( 1 == RX_(clo_inline) ) {
         return (POne == sh->n_parents);
      } else {
         return True;
      }
   } else {
      return False;
   }
}

static __inline__
UInt gr_arg(ShW* sh)
{
   if ( ! gr_arg_is_inlineable(sh) ) {
      return VG_(printf)("_");
   } else if (ShConst == sh->kind) {
      return gr_const(sh);
   } else if ( RX_(is_special_node)(sh->kind) ) {
      return VG_(printf)( name_of_ShW(sh) );
   } else {
      VG_(skin_panic)("problem in gr_arg()");
   }
}

static void gr_node_start(ShW* sh)
{
   switch (RX_(clo_format)) {
   case RxDot:
      VG_(printf)("\tn%x [label=\"", sh);
      break;

   case RxAiSee:
      VG_(printf)("\tnode: {title:\"n%x\" label:\"", sh);
      break;

   default: VG_(skin_panic)("Unknown format\n");
   }
}

static void gr_node_end(NodeStyle nodestyle)
{
   switch (RX_(clo_format)) {
   case RxDot:   VG_(printf)("\"%s]\n", dot_nodestyles[nodestyle]);   break;
   case RxAiSee: VG_(printf)("\"%s}\n", aiSee_nodestyles[nodestyle]); break;
   default: VG_(skin_panic)("Unknown format\n");
   }
}

// NULL sep means "don't print the sep or the value".
static void gr_node(Char* s, Char* sep, NodeStyle nodestyle,
                    Int argc, ShW** argv, ShW* sh)
{
   Int  i;
   UInt len = 0;
   Bool some_args_inlined = False;

   gr_node_start(sh);

   // Operator string
   len += VG_(printf)(s);

   // Determine if any nodes are inlined
   for (i = 0; i < argc; i++) {
      if ( gr_arg_is_inlineable(argv[i]) ) {
         some_args_inlined = True;
         break;
      }
   }

   // Print any inlined constants within the node;  last one without a
   // comma.
   if (some_args_inlined && 0 != RX_(clo_inline) && 0 != argc) {
      len += VG_(printf)("(");
      for (i = 0; i < argc-1; i++) {
         len += gr_arg(argv[i]);
         len += VG_(printf)(", ");
      }
      len += gr_arg(argv[i]);
      len += VG_(printf)(")");
   }

   if (NULL != sep) {
      // separator: break into two lines if it's too long, else use 'sep'
      VG_(printf)( len > 10 ? (Char*)"\\n" : sep );

      // Value
      pr_val(sh->val, sh->size);
   }

   // Rest
   gr_node_end(nodestyle);
}

static void gr_edge(ShW* parent, ShW* child, EdgeStyle edgestyle)
{
   // Must bypass ShInd nodes
   if (ShInd == child->kind) {
      child = child->argv[0];
   }
   
   switch (RX_(clo_format)) {
   case RxDot:   
      VG_(printf)("\tn%x -> n%x [%s]\n",
                  child, parent, dot_edgestyles[edgestyle]);
      break;

   case RxAiSee: 
      VG_(printf)("\tedge: {sourcename:\"n%x\" targetname:\"n%x\" %s}\n",
                  child, parent, aiSee_edgestyles[edgestyle]);
      break;

   default: VG_(skin_panic)("Unknown format\n");
   }
}

static void gr_incdec_chain_edge(ShW* parent, ShW* child, UInt n)
{
   switch (RX_(clo_format)) {
   case RxDot:   
      VG_(printf)("\tn%x -> n%x [label=\"[%u]\",%s]\n",
                  child, parent, n, dot_edgestyles[EdgeIncDecChain]);
      break;

   case RxAiSee: 
      VG_(printf)("\tedge: {sourcename:\"n%x\" targetname:\"n%x\""
                  "label:\"[%u]\" %s}\n", child, parent, n,
                  aiSee_edgestyles[EdgeIncDecChain]);
      break;

   default: VG_(skin_panic)("Unknown format\n");
   }
}

static
void gr_Op(Char* s, Char* sep, NodeStyle nodestyle, EdgeStyle edgestyle,
           Int argc, ShW** argv, ShW* sh, UInt depth)
{
   Int i;
   sk_assert(argc >= 0);

   gr_node(s, sep, nodestyle, argc, argv, sh);  // operator
   for (i = 0; i < argc; i++) {
      if ( ! gr_arg_is_inlineable(argv[i]) ) {
         gr_edge(sh, argv[i], edgestyle);   // rand i --> rator
         gr_ShW_x(argv[i], depth);       // operand i
      }
   }
}

// For printing unary ops that are often strung in long chains, eg. ++, --
// Chains are printed like X --> ++ -[n]-> ++ --> Y where 'n' ops are
// elided.  Note that we don't elide nodes with >1 parent.
static 
void gr_RepUnOp(Char* s, ShW* sh, UInt depth)
{
   UInt   n     = 1;
   ShW*   first = sh;
   ShKind kind  = sh->kind;

   // We want 'sh' to end up pointing at the op node which will form the
   // head of the chain (be it the last op node, or the first with multiple
   // parents).  Note that the first node is allowed to have multiple
   // parents, so we don't stop on that case.
   while (sh->argv[0]->kind == kind && 
          (sh->n_parents < PMany || sh == first) ) 
   {
      sh = sh->argv[0];
      n++;
   }

   if (n < 3) {                                  // 1 or 2: print as normal  
      gr_Op(s, " : ", NodeNone, EdgeNone, 1, first->argv, first, depth);    
   } else {
      gr_node(s, " : ", NodeNone, 1, first->argv, first);  // first node
      // Number of elided nodes is n-2
      gr_incdec_chain_edge(first, sh, n-2);
      gr_ShW_x(sh, depth+1);        // last node
   }
}

static __inline__
void gr_special_node(Char* s, ShW* sh)
{
   gr_node_start(sh);
   VG_(printf)(s);
   gr_node_end(NodeNone);
}

static
void gr_string(Char* s, ShW* sh, UInt depth)
{
   UInt first_extra;
   UInt i;
   Char buf[10];
                  
   if (RX_(clo_inline) <= 1) {
      gr_Op(s, NULL, NodeNone, EdgeNone, sh->argc, sh->argv, sh, depth); 
      
   } else {
      gr_node_start(sh);
      // start of proper string gathering
      if (ShConst == sh->argv[0]->kind) {
         first_extra = sh->argv[0]->extra;
         VG_(printf)("(%p)\\n%s.\\\"", sh->val,
                     ConstOrigin_to_string(first_extra));

         for (i = 0; i < sh->argc; i++) {
            if (first_extra != sh->argv[i]->extra) {
               first_extra = sh->argv[i]->extra;
               VG_(printf)("\\\"\\n"
                           "%s.\\\"", ConstOrigin_to_string(first_extra));
            }
            char_as_string(sh->argv[i]->val, buf);
            VG_(printf)(buf);
         }
         VG_(printf)("\\\"");
         gr_node_end(NodeNone);

      } else if (ShSysMem == sh->argv[0]->kind) {
         ShW* syscall_node = sh->argv[0]->argv[0];
         VG_(printf)("(+%u).\\\"", sh->argv[0]->extra);
         for (i = 0; i < sh->argc; i++) {
            sk_assert(ShSysMem == sh->argv[i]->kind);
            sk_assert(syscall_node == sh->argv[i]->argv[0]);
            char_as_string(sh->argv[i]->val, buf);
            VG_(printf)(buf);
         }
         VG_(printf)("\\\"");
         gr_node_end(NodeSysMem);

         gr_edge(sh, syscall_node, EdgeSysMem);   // print edge to syscall
         gr_ShW_x(syscall_node, depth);               // print syscall

      } else {
         VG_(skin_panic)("malformed ShString node");
      }
   }
}

static Char xbuf[64];

static void gr_ShW_x(ShW* sh, UInt depth)
{
   Char* s;

   if (!RX_(sane_ShW)(sh, /*building*/False)) {
      RX_(up_ShW)(sh);
      RX_(pp_ShW)(sh);
      VG_(skin_panic)("graphing insane ShW");
   }

   // Don't write more than once
   if (sh->graphed) return;
   sh->graphed = True;
   
   // Depth limit
   if (depth >= RX_(clo_print_depth)) {
      gr_node("...", " : ", NodeNone, 0, NULL, sh);
      return;
   }
   depth++;

   s = name_of_ShW(sh);
   
   switch (sh->kind) {
   case ShConst:
      gr_node(ConstOrigin_to_string(sh->extra), ".", NodeNone, 0, NULL, sh);
      break;

   case ShInc:
   case ShDec:
      gr_RepUnOp(s, sh, depth);
      break;
      
   case ShSplit2B:
   case ShSplit2W:    
   case ShSplit4B:
      gr_Op(s, " : ", NodeSplit, EdgeNone, sh->argc, sh->argv, sh, depth);
      break;

   case ShExit:       
      gr_Op(s, " : ", NodeSyscall, EdgeNone, sh->argc, sh->argv, sh, depth);
      break;
      
   case ShSysMem:
      VG_(sprintf)(xbuf, "(+%u)", sh->extra);
      gr_Op(xbuf, " : ", NodeSysMem, EdgeSysMem, sh->argc, sh->argv, sh, depth);
      break;

   case ShReadB:
   case ShReadW:
      VG_(sprintf)(xbuf, "%s%d", s, sh->extra);
      gr_Op(xbuf, " : ", NodeRead, EdgeNone, sh->argc, sh->argv, sh, depth);
      break;

   case ShCC2Val:
      VG_(sprintf)(xbuf, "%s%s", s, VG_(name_UCondcode)(sh->extra));
      gr_Op(xbuf, " : ", NodeNone, EdgeCC2Val, sh->argc, sh->argv, sh, depth);
      break;

   case ShCond: {
      ShW* test = sh->argv[0];
                   
      VG_(sprintf)(xbuf, "%s%s", s, VG_(name_UCondcode)(sh->extra));
//      gr_Op(xbuf, "style=dotted", "style=dotted", 1, sh); break;

      // This elides the SUB/AND/whatever node...
      gr_node(xbuf, " : ", NodeCond, 0, NULL, sh);
      gr_edge(sh, test->argv[0], EdgeCond);
      gr_ShW_x(test->argv[0], depth);
      gr_edge(sh, test->argv[1], EdgeCond);
      gr_ShW_x(test->argv[1], depth);
      break;
   }

   case ShSyscall: {
      SyscallInfo* syscall = & RX_(syscall_info)[sh->extra];

      gr_Op(syscall->name, " : ", NodeSyscall, EdgeNone, sh->argc, 
            sh->argv, sh, depth);
      break;
   }

   case ShFnOut:
      gr_Op(s, " : ", NodeSysMem, EdgeSysMem, sh->argc, sh->argv, sh, depth);
      break;

   case ShChunk:
      VG_(sprintf)(xbuf, "(%p)\\n%s", sh->val, s);
      gr_Op(xbuf, /*sep*/NULL, NodeNone, EdgeNone, sh->argc, sh->argv, sh, depth);
      break;

   case ShString:
      gr_string(s, sh, depth);
      break;

   case ShInd:
      gr_ShW_x(sh->argv[0], depth);
      break;

   case ShAlloc:
      gr_node(AllocKind_to_string(sh->extra), ".", NodeAlloc, 0, NULL, sh);
      break;


   // All other cases
   default: 
      if ( RX_(is_special_node)(sh->kind) ) {
         sk_assert(SpError != sh->kind);
         gr_special_node(s, sh);
      } else {
         gr_Op(s, " : ", NodeNone, EdgeNone, sh->argc, sh->argv, sh, depth); 
      }
      break;
   }
}

/*--------------------------------------------------------------------*/
/*--- Top level entry                                              ---*/
/*--------------------------------------------------------------------*/

static void gr_ShW(ShW* sh)
{
   gr_ShW_x(sh, 0);
}

// Print graph header ('ordering=in' puts node inputs in the given order)
static void gr_header(void)
{
   switch (RX_(clo_format)) {
   case RxDot:   
      VG_(printf)("digraph G {\n"
                  "\tgraph [ordering=in%s]\n"  
                  "\tnode  [shape=box, height=0.1, color=grey]\n", 
                  ( RX_(clo_a4) ? ",size=\"7.5,10.5\"" : "" )
                  );
      break;

   case RxAiSee:
      VG_(printf)("graph: {\n"
                  "\tlayout_algorithm: minbackward\n"
                  "\tmanhattan_edges: yes\n"
                  "\tsplines: no\n"
                  "\tport_sharing: no\n"
                  "\tarrowmode: free\n"
                  "\tdisplay_edge_labels: yes\n"
                  "\tcolorentry 43: 255 228 181   // moccasin\n"
                  "\tcolor: 43\n"
                  );
      break;

   default: VG_(skin_panic)("Unknown format\n");
   }
}

static void gr_footer(void)
{
   VG_(printf)("}\n");
}

void RX_(gr_var)(Addr a)
{
   ShW* sh = RX_(get_mem_shadow)(a);
   pre_ShW(sh, False);

   // need to have a dummy node as parent
   if ( RX_(clo_rewrite) )
      rew_ShW( & sh, sh, /*parent*/RX_(specials)[SpError]);
   gr_header();
   gr_ShW(sh);
   gr_footer();
   pre_ShW(sh, True);
}

/*--------------------------------------------------------------------*/
/*--- Dumping for slicing                                          ---*/
/*--------------------------------------------------------------------*/

#ifdef RX_SLICING

static UInt consts_dmped = 0;
static UInt ops_dmped    = 0;

// Forward declaration
static void dmp_ShW(ShW* sh, Bool reset);

static void dmp_header(void)
{
   Int i;
   // "cmd:" line
   VG_(printf)("cmd: ");
   for (i = 0; i < VG_(client_argc); i++) {
       VG_(printf)(" %s", VG_(client_argv)[i]);
   }  
   // "events:" line
   VG_(printf)("\nevents: consts ops\n");
}

static void dmp_footer(void)
{
   VG_(printf)("summary: %d %d\n", consts_dmped, ops_dmped);
}

static void dmp_node(ShW* sh)
{
   Bool ok = True;
   Char filename[100];
   Char fnname  [100];
   static Char prev_filename[100];
   static Char prev_fnname  [100];
   UInt linenum;


   //XXX hmm... not sure about this... it was violated by the stack machine's slice...
   

   //sk_assert(sh->instr_addr != 0xdeadbeef);

   ok = ok && VG_(get_filename_linenum) ( sh->instr_addr, filename, 100,
                                          & linenum );
   ok = ok && VG_(get_fnname)           ( sh->instr_addr, fnname,   100 );

   if (ok) {
      if (0 != VG_(strcmp)(prev_filename, filename))
         VG_(printf)("fl=%s\n", filename);
      if (0 != VG_(strcmp)(prev_fnname, fnname))
         VG_(printf)("fn=%s\n", fnname);
      if (ShConst == sh->kind) {
         VG_(printf)("%d 1 . # ", linenum);
         consts_dmped++;
      } else {
         VG_(printf)("%d . 1 # ", linenum);
         ops_dmped++;
      }
      RX_(pp_ShW)(sh);
      VG_(printf)("\n");
   }
   VG_(strcpy)(prev_filename, filename);
   VG_(strcpy)(prev_fnname,   fnname);
}

static __inline__ void dmp_Op(ShW* sh, Bool reset)
{
   Int i;
   if (!reset)
      dmp_node(sh);
   for (i = 0; i < sh->argc; i++)
      dmp_ShW(sh->argv[i], reset);
}

static void dmp_ShW(ShW* sh, Bool reset)
{
   sk_assert(sane_kind(sh->kind));

   // Don't do more than once;  unsets if reset == False
   if (sh->graphed == !reset) return;
   sh->graphed = !reset;

   switch (sh->kind) {

   default:
      dmp_Op(sh, reset); 
      break;
   }
}

void RX_(dmp_ShW)(ShW* sh)
{
   dmp_header();
   dmp_ShW(sh, False);
   dmp_ShW(sh, True);
   dmp_footer();
}

void RX_(dmp_var)(Addr a)
{
   RX_(dmp_ShW)( RX_(get_mem_shadow)(a) );
}
#endif

void RX_(gr_graph) ( UInt conds_n, ShW* conds_made[] )
{
   Int i;
   SyscallMade *made, *prev;

   // Prepare syscall nodes (includes exit value)
   for (made = RX_(syscalls_made); NULL != made; made = made->next) {
      pre_ShW(made->node, False);
   }
   if ( RX_(clo_rewrite) ) {
      for (made = RX_(syscalls_made); NULL != made; made = made->next) {
         rew_ShW( & made->node, made->node,
                  /*parent*/RX_(specials)[SpError] );
      }
   }

   // Prepare cond nodes (if wanted)
   for (i = 0; i < conds_n; i++) {
      pre_ShW(conds_made[i], False);
   }
   if ( RX_(clo_rewrite) ) {
      for (i = 0; i < conds_n; i++) {
         rew_ShW( & conds_made[i], conds_made[i], 
                  /*parent*/RX_(specials)[SpError] );
      }
   }

   // Print header
   gr_header();
   
   // Print syscall nodes (and f-nodes that fold syscall nodes),
   // connecting them with serialisation edges
   for (i = 0, prev = NULL, made = RX_(syscalls_made);
        NULL != made;
        i++, prev = made, made = made->next) 
   {
      VG_(printf)("/* r%u */\n", i);
      gr_ShW(made->node);     // syscall
      if (NULL != prev) {     // serialisation edge to previous syscall
         gr_edge(made->node, prev->node, EdgeSyscallConnect);
      }
   }

   // Print cond nodes (if wanted)
   for (i = 0; i < conds_n; i++) {
      VG_(printf)("/* c%u */\n", i);
      gr_ShW(conds_made[i]);      // cond
      if (i != 0) {                       // temporal edge to previous cond
         gr_edge(conds_made[i], conds_made[i-1], EdgeCondConnect);
      }
   }

   // Print footer
   gr_footer();
}

/*--------------------------------------------------------------------*/
/*--- end                                               rx_print.c ---*/
/*--------------------------------------------------------------------*/

