#define _CFFI_

/* We try to define Py_LIMITED_API before including Python.h.

   Mess: we can only define it if Py_DEBUG, Py_TRACE_REFS and
   Py_REF_DEBUG are not defined.  This is a best-effort approximation:
   we can learn about Py_DEBUG from pyconfig.h, but it is unclear if
   the same works for the other two macros.  Py_DEBUG implies them,
   but not the other way around.

   The implementation is messy (issue #350): on Windows, with _MSC_VER,
   we have to define Py_LIMITED_API even before including pyconfig.h.
   In that case, we guess what pyconfig.h will do to the macros above,
   and check our guess after the #include.

   Note that on Windows, with CPython 3.x, you need >= 3.5 and virtualenv
   version >= 16.0.0.  With older versions of either, you don't get a
   copy of PYTHON3.DLL in the virtualenv.  We can't check the version of
   CPython *before* we even include pyconfig.h.  ffi.set_source() puts
   a ``#define _CFFI_NO_LIMITED_API'' at the start of this file if it is
   running on Windows < 3.5, as an attempt at fixing it, but that's
   arguably wrong because it may not be the target version of Python.
   Still better than nothing I guess.  As another workaround, you can
   remove the definition of Py_LIMITED_API here.

   See also 'py_limited_api' in cffi/setuptools_ext.py.
*/
#if !defined(_CFFI_USE_EMBEDDING) && !defined(Py_LIMITED_API)
#  ifdef _MSC_VER
#    if !defined(_DEBUG) && !defined(Py_DEBUG) && !defined(Py_TRACE_REFS) && !defined(Py_REF_DEBUG) && !defined(_CFFI_NO_LIMITED_API)
#      define Py_LIMITED_API
#    endif
#    include <pyconfig.h>
     /* sanity-check: Py_LIMITED_API will cause crashes if any of these
        are also defined.  Normally, the Python file PC/pyconfig.h does not
        cause any of these to be defined, with the exception that _DEBUG
        causes Py_DEBUG.  Double-check that. */
#    ifdef Py_LIMITED_API
#      if defined(Py_DEBUG)
#        error "pyconfig.h unexpectedly defines Py_DEBUG, but Py_LIMITED_API is set"
#      endif
#      if defined(Py_TRACE_REFS)
#        error "pyconfig.h unexpectedly defines Py_TRACE_REFS, but Py_LIMITED_API is set"
#      endif
#      if defined(Py_REF_DEBUG)
#        error "pyconfig.h unexpectedly defines Py_REF_DEBUG, but Py_LIMITED_API is set"
#      endif
#    endif
#  else
#    include <pyconfig.h>
#    if !defined(Py_DEBUG) && !defined(Py_TRACE_REFS) && !defined(Py_REF_DEBUG) && !defined(_CFFI_NO_LIMITED_API)
#      define Py_LIMITED_API
#    endif
#  endif
#endif

#include <Python.h>
#ifdef __cplusplus
extern "C" {
#endif
#include <stddef.h>

/* This part is from file 'cffi/parse_c_type.h'.  It is copied at the
   beginning of C sources generated by CFFI's ffi.set_source(). */

typedef void *_cffi_opcode_t;

#define _CFFI_OP(opcode, arg)   (_cffi_opcode_t)(opcode | (((uintptr_t)(arg)) << 8))
#define _CFFI_GETOP(cffi_opcode)    ((unsigned char)(uintptr_t)cffi_opcode)
#define _CFFI_GETARG(cffi_opcode)   (((intptr_t)cffi_opcode) >> 8)

#define _CFFI_OP_PRIMITIVE       1
#define _CFFI_OP_POINTER         3
#define _CFFI_OP_ARRAY           5
#define _CFFI_OP_OPEN_ARRAY      7
#define _CFFI_OP_STRUCT_UNION    9
#define _CFFI_OP_ENUM           11
#define _CFFI_OP_FUNCTION       13
#define _CFFI_OP_FUNCTION_END   15
#define _CFFI_OP_NOOP           17
#define _CFFI_OP_BITFIELD       19
#define _CFFI_OP_TYPENAME       21
#define _CFFI_OP_CPYTHON_BLTN_V 23   // varargs
#define _CFFI_OP_CPYTHON_BLTN_N 25   // noargs
#define _CFFI_OP_CPYTHON_BLTN_O 27   // O  (i.e. a single arg)
#define _CFFI_OP_CONSTANT       29
#define _CFFI_OP_CONSTANT_INT   31
#define _CFFI_OP_GLOBAL_VAR     33
#define _CFFI_OP_DLOPEN_FUNC    35
#define _CFFI_OP_DLOPEN_CONST   37
#define _CFFI_OP_GLOBAL_VAR_F   39
#define _CFFI_OP_EXTERN_PYTHON  41

#define _CFFI_PRIM_VOID          0
#define _CFFI_PRIM_BOOL          1
#define _CFFI_PRIM_CHAR          2
#define _CFFI_PRIM_SCHAR         3
#define _CFFI_PRIM_UCHAR         4
#define _CFFI_PRIM_SHORT         5
#define _CFFI_PRIM_USHORT        6
#define _CFFI_PRIM_INT           7
#define _CFFI_PRIM_UINT          8
#define _CFFI_PRIM_LONG          9
#define _CFFI_PRIM_ULONG        10
#define _CFFI_PRIM_LONGLONG     11
#define _CFFI_PRIM_ULONGLONG    12
#define _CFFI_PRIM_FLOAT        13
#define _CFFI_PRIM_DOUBLE       14
#define _CFFI_PRIM_LONGDOUBLE   15

#define _CFFI_PRIM_WCHAR        16
#define _CFFI_PRIM_INT8         17
#define _CFFI_PRIM_UINT8        18
#define _CFFI_PRIM_INT16        19
#define _CFFI_PRIM_UINT16       20
#define _CFFI_PRIM_INT32        21
#define _CFFI_PRIM_UINT32       22
#define _CFFI_PRIM_INT64        23
#define _CFFI_PRIM_UINT64       24
#define _CFFI_PRIM_INTPTR       25
#define _CFFI_PRIM_UINTPTR      26
#define _CFFI_PRIM_PTRDIFF      27
#define _CFFI_PRIM_SIZE         28
#define _CFFI_PRIM_SSIZE        29
#define _CFFI_PRIM_INT_LEAST8   30
#define _CFFI_PRIM_UINT_LEAST8  31
#define _CFFI_PRIM_INT_LEAST16  32
#define _CFFI_PRIM_UINT_LEAST16 33
#define _CFFI_PRIM_INT_LEAST32  34
#define _CFFI_PRIM_UINT_LEAST32 35
#define _CFFI_PRIM_INT_LEAST64  36
#define _CFFI_PRIM_UINT_LEAST64 37
#define _CFFI_PRIM_INT_FAST8    38
#define _CFFI_PRIM_UINT_FAST8   39
#define _CFFI_PRIM_INT_FAST16   40
#define _CFFI_PRIM_UINT_FAST16  41
#define _CFFI_PRIM_INT_FAST32   42
#define _CFFI_PRIM_UINT_FAST32  43
#define _CFFI_PRIM_INT_FAST64   44
#define _CFFI_PRIM_UINT_FAST64  45
#define _CFFI_PRIM_INTMAX       46
#define _CFFI_PRIM_UINTMAX      47
#define _CFFI_PRIM_FLOATCOMPLEX 48
#define _CFFI_PRIM_DOUBLECOMPLEX 49
#define _CFFI_PRIM_CHAR16       50
#define _CFFI_PRIM_CHAR32       51

#define _CFFI__NUM_PRIM         52
#define _CFFI__UNKNOWN_PRIM           (-1)
#define _CFFI__UNKNOWN_FLOAT_PRIM     (-2)
#define _CFFI__UNKNOWN_LONG_DOUBLE    (-3)

#define _CFFI__IO_FILE_STRUCT         (-1)


struct _cffi_global_s {
    const char *name;
    void *address;
    _cffi_opcode_t type_op;
    void *size_or_direct_fn;  // OP_GLOBAL_VAR: size, or 0 if unknown
                              // OP_CPYTHON_BLTN_*: addr of direct function
};

struct _cffi_getconst_s {
    unsigned long long value;
    const struct _cffi_type_context_s *ctx;
    int gindex;
};

struct _cffi_struct_union_s {
    const char *name;
    int type_index;          // -> _cffi_types, on a OP_STRUCT_UNION
    int flags;               // _CFFI_F_* flags below
    size_t size;
    int alignment;
    int first_field_index;   // -> _cffi_fields array
    int num_fields;
};
#define _CFFI_F_UNION         0x01   // is a union, not a struct
#define _CFFI_F_CHECK_FIELDS  0x02   // complain if fields are not in the
                                     // "standard layout" or if some are missing
#define _CFFI_F_PACKED        0x04   // for CHECK_FIELDS, assume a packed struct
#define _CFFI_F_EXTERNAL      0x08   // in some other ffi.include()
#define _CFFI_F_OPAQUE        0x10   // opaque

struct _cffi_field_s {
    const char *name;
    size_t field_offset;
    size_t field_size;
    _cffi_opcode_t field_type_op;
};

struct _cffi_enum_s {
    const char *name;
    int type_index;          // -> _cffi_types, on a OP_ENUM
    int type_prim;           // _CFFI_PRIM_xxx
    const char *enumerators; // comma-delimited string
};

struct _cffi_typename_s {
    const char *name;
    int type_index;   /* if opaque, points to a possibly artificial
                         OP_STRUCT which is itself opaque */
};

struct _cffi_type_context_s {
    _cffi_opcode_t *types;
    const struct _cffi_global_s *globals;
    const struct _cffi_field_s *fields;
    const struct _cffi_struct_union_s *struct_unions;
    const struct _cffi_enum_s *enums;
    const struct _cffi_typename_s *typenames;
    int num_globals;
    int num_struct_unions;
    int num_enums;
    int num_typenames;
    const char *const *includes;
    int num_types;
    int flags;      /* future extension */
};

struct _cffi_parse_info_s {
    const struct _cffi_type_context_s *ctx;
    _cffi_opcode_t *output;
    unsigned int output_size;
    size_t error_location;
    const char *error_message;
};

struct _cffi_externpy_s {
    const char *name;
    size_t size_of_result;
    void *reserved1, *reserved2;
};

#ifdef _CFFI_INTERNAL
static int parse_c_type(struct _cffi_parse_info_s *info, const char *input);
static int search_in_globals(const struct _cffi_type_context_s *ctx,
                             const char *search, size_t search_len);
static int search_in_struct_unions(const struct _cffi_type_context_s *ctx,
                                   const char *search, size_t search_len);
#endif

/* this block of #ifs should be kept exactly identical between
   c/_cffi_backend.c, cffi/vengine_cpy.py, cffi/vengine_gen.py
   and cffi/_cffi_include.h */
#if defined(_MSC_VER)
# include <malloc.h>   /* for alloca() */
# if _MSC_VER < 1600   /* MSVC < 2010 */
   typedef __int8 int8_t;
   typedef __int16 int16_t;
   typedef __int32 int32_t;
   typedef __int64 int64_t;
   typedef unsigned __int8 uint8_t;
   typedef unsigned __int16 uint16_t;
   typedef unsigned __int32 uint32_t;
   typedef unsigned __int64 uint64_t;
   typedef __int8 int_least8_t;
   typedef __int16 int_least16_t;
   typedef __int32 int_least32_t;
   typedef __int64 int_least64_t;
   typedef unsigned __int8 uint_least8_t;
   typedef unsigned __int16 uint_least16_t;
   typedef unsigned __int32 uint_least32_t;
   typedef unsigned __int64 uint_least64_t;
   typedef __int8 int_fast8_t;
   typedef __int16 int_fast16_t;
   typedef __int32 int_fast32_t;
   typedef __int64 int_fast64_t;
   typedef unsigned __int8 uint_fast8_t;
   typedef unsigned __int16 uint_fast16_t;
   typedef unsigned __int32 uint_fast32_t;
   typedef unsigned __int64 uint_fast64_t;
   typedef __int64 intmax_t;
   typedef unsigned __int64 uintmax_t;
# else
#  include <stdint.h>
# endif
# if _MSC_VER < 1800   /* MSVC < 2013 */
#  ifndef __cplusplus
    typedef unsigned char _Bool;
#  endif
# endif
#else
# include <stdint.h>
# if (defined (__SVR4) && defined (__sun)) || defined(_AIX) || defined(__hpux)
#  include <alloca.h>
# endif
#endif

#ifdef __GNUC__
# define _CFFI_UNUSED_FN  __attribute__((unused))
#else
# define _CFFI_UNUSED_FN  /* nothing */
#endif

#ifdef __cplusplus
# ifndef _Bool
   typedef bool _Bool;   /* semi-hackish: C++ has no _Bool; bool is builtin */
# endif
#endif

/**********  CPython-specific section  **********/
#ifndef PYPY_VERSION


#if PY_MAJOR_VERSION >= 3
# define PyInt_FromLong PyLong_FromLong
#endif

#define _cffi_from_c_double PyFloat_FromDouble
#define _cffi_from_c_float PyFloat_FromDouble
#define _cffi_from_c_long PyInt_FromLong
#define _cffi_from_c_ulong PyLong_FromUnsignedLong
#define _cffi_from_c_longlong PyLong_FromLongLong
#define _cffi_from_c_ulonglong PyLong_FromUnsignedLongLong
#define _cffi_from_c__Bool PyBool_FromLong

#define _cffi_to_c_double PyFloat_AsDouble
#define _cffi_to_c_float PyFloat_AsDouble

#define _cffi_from_c_int(x, type)                                        \
    (((type)-1) > 0 ? /* unsigned */                                     \
        (sizeof(type) < sizeof(long) ?                                   \
            PyInt_FromLong((long)x) :                                    \
         sizeof(type) == sizeof(long) ?                                  \
            PyLong_FromUnsignedLong((unsigned long)x) :                  \
            PyLong_FromUnsignedLongLong((unsigned long long)x)) :        \
        (sizeof(type) <= sizeof(long) ?                                  \
            PyInt_FromLong((long)x) :                                    \
            PyLong_FromLongLong((long long)x)))

#define _cffi_to_c_int(o, type)                                          \
    ((type)(                                                             \
     sizeof(type) == 1 ? (((type)-1) > 0 ? (type)_cffi_to_c_u8(o)        \
                                         : (type)_cffi_to_c_i8(o)) :     \
     sizeof(type) == 2 ? (((type)-1) > 0 ? (type)_cffi_to_c_u16(o)       \
                                         : (type)_cffi_to_c_i16(o)) :    \
     sizeof(type) == 4 ? (((type)-1) > 0 ? (type)_cffi_to_c_u32(o)       \
                                         : (type)_cffi_to_c_i32(o)) :    \
     sizeof(type) == 8 ? (((type)-1) > 0 ? (type)_cffi_to_c_u64(o)       \
                                         : (type)_cffi_to_c_i64(o)) :    \
     (Py_FatalError("unsupported size for type " #type), (type)0)))

#define _cffi_to_c_i8                                                    \
                 ((int(*)(PyObject *))_cffi_exports[1])
#define _cffi_to_c_u8                                                    \
                 ((int(*)(PyObject *))_cffi_exports[2])
#define _cffi_to_c_i16                                                   \
                 ((int(*)(PyObject *))_cffi_exports[3])
#define _cffi_to_c_u16                                                   \
                 ((int(*)(PyObject *))_cffi_exports[4])
#define _cffi_to_c_i32                                                   \
                 ((int(*)(PyObject *))_cffi_exports[5])
#define _cffi_to_c_u32                                                   \
                 ((unsigned int(*)(PyObject *))_cffi_exports[6])
#define _cffi_to_c_i64                                                   \
                 ((long long(*)(PyObject *))_cffi_exports[7])
#define _cffi_to_c_u64                                                   \
                 ((unsigned long long(*)(PyObject *))_cffi_exports[8])
#define _cffi_to_c_char                                                  \
                 ((int(*)(PyObject *))_cffi_exports[9])
#define _cffi_from_c_pointer                                             \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[10])
#define _cffi_to_c_pointer                                               \
    ((char *(*)(PyObject *, struct _cffi_ctypedescr *))_cffi_exports[11])
#define _cffi_get_struct_layout                                          \
    not used any more
#define _cffi_restore_errno                                              \
    ((void(*)(void))_cffi_exports[13])
#define _cffi_save_errno                                                 \
    ((void(*)(void))_cffi_exports[14])
#define _cffi_from_c_char                                                \
    ((PyObject *(*)(char))_cffi_exports[15])
#define _cffi_from_c_deref                                               \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[16])
#define _cffi_to_c                                                       \
    ((int(*)(char *, struct _cffi_ctypedescr *, PyObject *))_cffi_exports[17])
#define _cffi_from_c_struct                                              \
    ((PyObject *(*)(char *, struct _cffi_ctypedescr *))_cffi_exports[18])
#define _cffi_to_c_wchar_t                                               \
    ((_cffi_wchar_t(*)(PyObject *))_cffi_exports[19])
#define _cffi_from_c_wchar_t                                             \
    ((PyObject *(*)(_cffi_wchar_t))_cffi_exports[20])
#define _cffi_to_c_long_double                                           \
    ((long double(*)(PyObject *))_cffi_exports[21])
#define _cffi_to_c__Bool                                                 \
    ((_Bool(*)(PyObject *))_cffi_exports[22])
#define _cffi_prepare_pointer_call_argument                              \
    ((Py_ssize_t(*)(struct _cffi_ctypedescr *,                           \
                    PyObject *, char **))_cffi_exports[23])
#define _cffi_convert_array_from_object                                  \
    ((int(*)(char *, struct _cffi_ctypedescr *, PyObject *))_cffi_exports[24])
#define _CFFI_CPIDX  25
#define _cffi_call_python                                                \
    ((void(*)(struct _cffi_externpy_s *, char *))_cffi_exports[_CFFI_CPIDX])
#define _cffi_to_c_wchar3216_t                                           \
    ((int(*)(PyObject *))_cffi_exports[26])
#define _cffi_from_c_wchar3216_t                                         \
    ((PyObject *(*)(int))_cffi_exports[27])
#define _CFFI_NUM_EXPORTS 28

struct _cffi_ctypedescr;

static void *_cffi_exports[_CFFI_NUM_EXPORTS];

#define _cffi_type(index)   (                           \
    assert((((uintptr_t)_cffi_types[index]) & 1) == 0), \
    (struct _cffi_ctypedescr *)_cffi_types[index])

static PyObject *_cffi_init(const char *module_name, Py_ssize_t version,
                            const struct _cffi_type_context_s *ctx)
{
    PyObject *module, *o_arg, *new_module;
    void *raw[] = {
        (void *)module_name,
        (void *)version,
        (void *)_cffi_exports,
        (void *)ctx,
    };

    module = PyImport_ImportModule("_cffi_backend");
    if (module == NULL)
        goto failure;

    o_arg = PyLong_FromVoidPtr((void *)raw);
    if (o_arg == NULL)
        goto failure;

    new_module = PyObject_CallMethod(
        module, (char *)"_init_cffi_1_0_external_module", (char *)"O", o_arg);

    Py_DECREF(o_arg);
    Py_DECREF(module);
    return new_module;

  failure:
    Py_XDECREF(module);
    return NULL;
}


#ifdef HAVE_WCHAR_H
typedef wchar_t _cffi_wchar_t;
#else
typedef uint16_t _cffi_wchar_t;   /* same random pick as _cffi_backend.c */
#endif

_CFFI_UNUSED_FN static uint16_t _cffi_to_c_char16_t(PyObject *o)
{
    if (sizeof(_cffi_wchar_t) == 2)
        return (uint16_t)_cffi_to_c_wchar_t(o);
    else
        return (uint16_t)_cffi_to_c_wchar3216_t(o);
}

_CFFI_UNUSED_FN static PyObject *_cffi_from_c_char16_t(uint16_t x)
{
    if (sizeof(_cffi_wchar_t) == 2)
        return _cffi_from_c_wchar_t((_cffi_wchar_t)x);
    else
        return _cffi_from_c_wchar3216_t((int)x);
}

_CFFI_UNUSED_FN static int _cffi_to_c_char32_t(PyObject *o)
{
    if (sizeof(_cffi_wchar_t) == 4)
        return (int)_cffi_to_c_wchar_t(o);
    else
        return (int)_cffi_to_c_wchar3216_t(o);
}

_CFFI_UNUSED_FN static PyObject *_cffi_from_c_char32_t(unsigned int x)
{
    if (sizeof(_cffi_wchar_t) == 4)
        return _cffi_from_c_wchar_t((_cffi_wchar_t)x);
    else
        return _cffi_from_c_wchar3216_t((int)x);
}

union _cffi_union_alignment_u {
    unsigned char m_char;
    unsigned short m_short;
    unsigned int m_int;
    unsigned long m_long;
    unsigned long long m_longlong;
    float m_float;
    double m_double;
    long double m_longdouble;
};

struct _cffi_freeme_s {
    struct _cffi_freeme_s *next;
    union _cffi_union_alignment_u alignment;
};

_CFFI_UNUSED_FN static int
_cffi_convert_array_argument(struct _cffi_ctypedescr *ctptr, PyObject *arg,
                             char **output_data, Py_ssize_t datasize,
                             struct _cffi_freeme_s **freeme)
{
    char *p;
    if (datasize < 0)
        return -1;

    p = *output_data;
    if (p == NULL) {
        struct _cffi_freeme_s *fp = (struct _cffi_freeme_s *)PyObject_Malloc(
            offsetof(struct _cffi_freeme_s, alignment) + (size_t)datasize);
        if (fp == NULL)
            return -1;
        fp->next = *freeme;
        *freeme = fp;
        p = *output_data = (char *)&fp->alignment;
    }
    memset((void *)p, 0, (size_t)datasize);
    return _cffi_convert_array_from_object(p, ctptr, arg);
}

_CFFI_UNUSED_FN static void
_cffi_free_array_arguments(struct _cffi_freeme_s *freeme)
{
    do {
        void *p = (void *)freeme;
        freeme = freeme->next;
        PyObject_Free(p);
    } while (freeme != NULL);
}

/**********  end CPython-specific section  **********/
#else
_CFFI_UNUSED_FN
static void (*_cffi_call_python_org)(struct _cffi_externpy_s *, char *);
# define _cffi_call_python  _cffi_call_python_org
#endif


#define _cffi_array_len(array)   (sizeof(array) / sizeof((array)[0]))

#define _cffi_prim_int(size, sign)                                      \
    ((size) == 1 ? ((sign) ? _CFFI_PRIM_INT8  : _CFFI_PRIM_UINT8)  :    \
     (size) == 2 ? ((sign) ? _CFFI_PRIM_INT16 : _CFFI_PRIM_UINT16) :    \
     (size) == 4 ? ((sign) ? _CFFI_PRIM_INT32 : _CFFI_PRIM_UINT32) :    \
     (size) == 8 ? ((sign) ? _CFFI_PRIM_INT64 : _CFFI_PRIM_UINT64) :    \
     _CFFI__UNKNOWN_PRIM)

#define _cffi_prim_float(size)                                          \
    ((size) == sizeof(float) ? _CFFI_PRIM_FLOAT :                       \
     (size) == sizeof(double) ? _CFFI_PRIM_DOUBLE :                     \
     (size) == sizeof(long double) ? _CFFI__UNKNOWN_LONG_DOUBLE :       \
     _CFFI__UNKNOWN_FLOAT_PRIM)

#define _cffi_check_int(got, got_nonpos, expected)      \
    ((got_nonpos) == (expected <= 0) &&                 \
     (got) == (unsigned long long)expected)

#ifdef MS_WIN32
# define _cffi_stdcall  __stdcall
#else
# define _cffi_stdcall  /* nothing */
#endif

#ifdef __cplusplus
}
#endif

/************************************************************/


/* define our OpenSSL API compatibility level to 1.0.1. Any symbols older than
   that will raise an error during compilation. We can raise this number again
   after we drop 1.0.2 support in the distant future.  */
#define OPENSSL_API_COMPAT 0x10001000L

#include <openssl/opensslv.h>


#if defined(LIBRESSL_VERSION_NUMBER)
#define CRYPTOGRAPHY_IS_LIBRESSL 1
#else
#define CRYPTOGRAPHY_IS_LIBRESSL 0
#endif

/*
    LibreSSL removed e_os2.h from the public headers so we'll only include it
    if we're using vanilla OpenSSL.
*/
#if !CRYPTOGRAPHY_IS_LIBRESSL
#include <openssl/e_os2.h>
#endif
#if defined(_WIN32)
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <Wincrypt.h>
#include <Winsock2.h>
#endif

#define CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER     (OPENSSL_VERSION_NUMBER >= 0x1010006f && !CRYPTOGRAPHY_IS_LIBRESSL)

#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_110J     (OPENSSL_VERSION_NUMBER < 0x101000af || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_111     (OPENSSL_VERSION_NUMBER < 0x10101000 || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B     (OPENSSL_VERSION_NUMBER < 0x10101020 || CRYPTOGRAPHY_IS_LIBRESSL)
#define CRYPTOGRAPHY_OPENSSL_LESS_THAN_111D     (OPENSSL_VERSION_NUMBER < 0x10101040 || CRYPTOGRAPHY_IS_LIBRESSL)
#if (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111D && !CRYPTOGRAPHY_IS_LIBRESSL &&     !defined(OPENSSL_NO_ENGINE)) || defined(USE_OSRANDOM_RNG_FOR_TESTING)
#define CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE 1
#else
#define CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE 0
#endif


#include <openssl/aes.h>


#include <openssl/asn1.h>


#include <openssl/bn.h>


#include <openssl/bio.h>


#if !defined(OPENSSL_NO_CMAC)
#include <openssl/cmac.h>
#endif


#include <openssl/conf.h>


#include <openssl/crypto.h>


#if CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER && !defined(OPENSSL_NO_CT)
#include <openssl/ct.h>

typedef STACK_OF(SCT) Cryptography_STACK_OF_SCT;
#endif


#include <openssl/dh.h>


#include <openssl/dsa.h>


#include <openssl/ec.h>
#include <openssl/obj_mac.h>


#include <openssl/ecdh.h>


#include <openssl/ecdsa.h>


#include <openssl/engine.h>


#include <openssl/err.h>


#include <openssl/evp.h>


#include <openssl/crypto.h>


#include <openssl/hmac.h>


#include <openssl/obj_mac.h>


#include <openssl/objects.h>


#include <openssl/ocsp.h>


#include <openssl/opensslv.h>

#ifndef OPENSSL_NO_ENGINE
/* OpenSSL has ENGINE support so include all of this. */
#ifdef _WIN32
  #include <Wincrypt.h>
#else
  #include <fcntl.h>
  #include <unistd.h>
   /* for defined(BSD) */
  #ifndef __MVS__
    #include <sys/param.h>
  #endif

  #ifdef BSD
    /* for SYS_getentropy */
    #include <sys/syscall.h>
  #endif

  #ifdef __APPLE__
    #include <sys/random.h>
    /* To support weak linking we need to declare this as a weak import even if
     * it's not present in sys/random (e.g. macOS < 10.12). */
    extern int getentropy(void *buffer, size_t size) __attribute((weak_import));
  #endif

  #ifdef __linux__
    /* for SYS_getrandom */
    #include <sys/syscall.h>
    #ifndef GRND_NONBLOCK
      #define GRND_NONBLOCK 0x0001
    #endif /* GRND_NONBLOCK */

    #ifndef SYS_getrandom
      /* We only bother to define the constants for platforms where we ship
       * wheels, since that's the predominant way you get a situation where
       * you don't have SYS_getrandom at compile time but do have the syscall
       * at runtime */
      #if defined(__x86_64__)
        #define SYS_getrandom 318
      #elif defined(__i386__)
        #define SYS_getrandom 355
      #elif defined(__aarch64__)
        #define SYS_getrandom 278
      #endif
    #endif
  #endif /* __linux__ */
#endif /* _WIN32 */

#define CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM 1
#define CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY 2
#define CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM 3
#define CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM 4

#ifndef CRYPTOGRAPHY_OSRANDOM_ENGINE
  #if defined(_WIN32)
    /* Windows */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM
  #elif defined(BSD) && defined(SYS_getentropy)
    /* OpenBSD 5.6+ & macOS with SYS_getentropy defined, although < 10.12 will fallback
     * to urandom */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY
  #elif defined(__linux__) && defined(SYS_getrandom)
    /* Linux 3.17+ */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM
  #else
    /* Keep this as last entry, fall back to /dev/urandom */
    #define CRYPTOGRAPHY_OSRANDOM_ENGINE CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM
  #endif
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE */

/* Fallbacks need /dev/urandom helper functions. */
#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM || \
     CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM || \
     (CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY && \
     defined(__APPLE__))
  #define CRYPTOGRAPHY_OSRANDOM_NEEDS_DEV_URANDOM 1
#endif

enum {
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED = -2,
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT,
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK,
    CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS
};

enum {
    CRYPTOGRAPHY_OSRANDOM_GETENTROPY_NOT_INIT,
    CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK,
    CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS
};

/* engine ctrl */
#define CRYPTOGRAPHY_OSRANDOM_GET_IMPLEMENTATION ENGINE_CMD_BASE

/* error reporting */
static void ERR_load_Cryptography_OSRandom_strings(void);
static void ERR_Cryptography_OSRandom_error(int function, int reason,
                                            char *file, int line);

#define CRYPTOGRAPHY_OSRANDOM_F_INIT 100
#define CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES 101
#define CRYPTOGRAPHY_OSRANDOM_F_FINISH 102
#define CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_FD 300
#define CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_READ 301

#define CRYPTOGRAPHY_OSRANDOM_R_CRYPTACQUIRECONTEXT 100
#define CRYPTOGRAPHY_OSRANDOM_R_CRYPTGENRANDOM 101
#define CRYPTOGRAPHY_OSRANDOM_R_CRYPTRELEASECONTEXT 102

#define CRYPTOGRAPHY_OSRANDOM_R_GETENTROPY_FAILED 200

#define CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_OPEN_FAILED 300
#define CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_READ_FAILED 301

#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED 400
#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED_UNEXPECTED 402
#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_FAILED 403
#define CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_NOT_INIT 404
#endif


#include <openssl/pem.h>


#include <openssl/pkcs12.h>


#include <openssl/rand.h>


#include <openssl/rsa.h>


#include <openssl/ssl.h>

typedef STACK_OF(SSL_CIPHER) Cryptography_STACK_OF_SSL_CIPHER;


#include <openssl/ssl.h>

/*
 * This is part of a work-around for the difficulty cffi has in dealing with
 * `STACK_OF(foo)` as the name of a type.  We invent a new, simpler name that
 * will be an alias for this type and use the alias throughout.  This works
 * together with another opaque typedef for the same name in the TYPES section.
 * Note that the result is an opaque type.
 */
typedef STACK_OF(X509) Cryptography_STACK_OF_X509;
typedef STACK_OF(X509_CRL) Cryptography_STACK_OF_X509_CRL;
typedef STACK_OF(X509_REVOKED) Cryptography_STACK_OF_X509_REVOKED;


#include <openssl/x509.h>

/*
 * See the comment above Cryptography_STACK_OF_X509 in x509.py
 */
typedef STACK_OF(X509_NAME) Cryptography_STACK_OF_X509_NAME;
typedef STACK_OF(X509_NAME_ENTRY) Cryptography_STACK_OF_X509_NAME_ENTRY;


#include <openssl/x509v3.h>

/*
 * This is part of a work-around for the difficulty cffi has in dealing with
 * `LHASH_OF(foo)` as the name of a type.  We invent a new, simpler name that
 * will be an alias for this type and use the alias throughout.  This works
 * together with another opaque typedef for the same name in the TYPES section.
 * Note that the result is an opaque type.
 */
typedef LHASH_OF(CONF_VALUE) Cryptography_LHASH_OF_CONF_VALUE;

typedef STACK_OF(ACCESS_DESCRIPTION) Cryptography_STACK_OF_ACCESS_DESCRIPTION;
typedef STACK_OF(DIST_POINT) Cryptography_STACK_OF_DIST_POINT;
typedef STACK_OF(POLICYQUALINFO) Cryptography_STACK_OF_POLICYQUALINFO;
typedef STACK_OF(POLICYINFO) Cryptography_STACK_OF_POLICYINFO;
typedef STACK_OF(ASN1_INTEGER) Cryptography_STACK_OF_ASN1_INTEGER;
typedef STACK_OF(GENERAL_SUBTREE) Cryptography_STACK_OF_GENERAL_SUBTREE;


#include <openssl/x509_vfy.h>

/*
 * This is part of a work-around for the difficulty cffi has in dealing with
 * `STACK_OF(foo)` as the name of a type.  We invent a new, simpler name that
 * will be an alias for this type and use the alias throughout.  This works
 * together with another opaque typedef for the same name in the TYPES section.
 * Note that the result is an opaque type.
 */
typedef STACK_OF(ASN1_OBJECT) Cryptography_STACK_OF_ASN1_OBJECT;
typedef STACK_OF(X509_OBJECT) Cryptography_STACK_OF_X509_OBJECT;


#include <openssl/pkcs7.h>


#include <string.h>
















/* In 1.1.0 SSLeay has finally been retired. We bidirectionally define the
   values so you can use either one. This is so we can use the new function
   names no matter what OpenSSL we're running on, but users on older pyOpenSSL
   releases won't see issues if they're running OpenSSL 1.1.0 */
#if !defined(SSLEAY_VERSION)
# define SSLeay                  OpenSSL_version_num
# define SSLeay_version          OpenSSL_version
# define SSLEAY_VERSION_NUMBER   OPENSSL_VERSION_NUMBER
# define SSLEAY_VERSION          OPENSSL_VERSION
# define SSLEAY_CFLAGS           OPENSSL_CFLAGS
# define SSLEAY_BUILT_ON         OPENSSL_BUILT_ON
# define SSLEAY_PLATFORM         OPENSSL_PLATFORM
# define SSLEAY_DIR              OPENSSL_DIR
#endif
#if !defined(OPENSSL_VERSION)
# define OpenSSL_version_num     SSLeay
# define OpenSSL_version         SSLeay_version
# define OPENSSL_VERSION         SSLEAY_VERSION
# define OPENSSL_CFLAGS          SSLEAY_CFLAGS
# define OPENSSL_BUILT_ON        SSLEAY_BUILT_ON
# define OPENSSL_PLATFORM        SSLEAY_PLATFORM
# define OPENSSL_DIR             SSLEAY_DIR
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_OPENSSL_CLEANUP = 0;

void (*OPENSSL_cleanup)(void) = NULL;

/* This function has a significantly different signature pre-1.1.0. since it is
 * for testing only, we don't bother to expose it on older OpenSSLs.
 */
static const long Cryptography_HAS_MEM_FUNCTIONS = 0;
int (*Cryptography_CRYPTO_set_mem_functions)(
    void *(*)(size_t, const char *, int),
    void *(*)(void *, size_t, const char *, int),
    void (*)(void *, const char *, int)) = NULL;

#else
static const long Cryptography_HAS_OPENSSL_CLEANUP = 1;
static const long Cryptography_HAS_MEM_FUNCTIONS = 1;

int Cryptography_CRYPTO_set_mem_functions(
    void *(*m)(size_t, const char *, int),
    void *(*r)(void *, size_t, const char *, int),
    void (*f)(void *, const char *, int)
) {
    return CRYPTO_set_mem_functions(m, r, f);
}
#endif

void *Cryptography_malloc_wrapper(size_t size, const char *path, int line) {
    return malloc(size);
}

void *Cryptography_realloc_wrapper(void *ptr, size_t size, const char *path,
                                   int line) {
    return realloc(ptr, size);
}

void Cryptography_free_wrapper(void *ptr, const char *path, int line) {
    free(ptr);
}


#if CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER && !defined(OPENSSL_NO_CT)
static const long Cryptography_HAS_SCT = 1;
#else
static const long Cryptography_HAS_SCT = 0;

typedef enum {
    SCT_VERSION_NOT_SET,
    SCT_VERSION_V1
} sct_version_t;
typedef enum {
    CT_LOG_ENTRY_TYPE_NOT_SET,
    CT_LOG_ENTRY_TYPE_X509,
    CT_LOG_ENTRY_TYPE_PRECERT
} ct_log_entry_type_t;
typedef enum {
    SCT_SOURCE_UNKNOWN,
    SCT_SOURCE_TLS_EXTENSION,
    SCT_SOURCE_X509V3_EXTENSION,
    SCT_SOURCE_OCSP_STAPLED_RESPONSE
} sct_source_t;

/* OpenSSL compiled with `no-ct` still defines the `SCT` struct. */
#if !defined(OPENSSL_NO_CT)
typedef void SCT;
#endif

typedef void Cryptography_STACK_OF_SCT;

sct_version_t (*SCT_get_version)(const SCT *) = NULL;
ct_log_entry_type_t (*SCT_get_log_entry_type)(const SCT *) = NULL;
size_t (*SCT_get0_log_id)(const SCT *, unsigned char **) = NULL;
size_t (*SCT_get0_signature)(const SCT *, unsigned char **) = NULL;
uint64_t (*SCT_get_timestamp)(const SCT *) = NULL;

int (*SCT_set_source)(SCT *, sct_source_t) = NULL;

Cryptography_STACK_OF_SCT *(*sk_SCT_new_null)(void) = NULL;
void (*sk_SCT_free)(Cryptography_STACK_OF_SCT *) = NULL;
int (*sk_SCT_num)(const Cryptography_STACK_OF_SCT *) = NULL;
SCT *(*sk_SCT_value)(const Cryptography_STACK_OF_SCT *, int) = NULL;
int (*sk_SCT_push)(Cryptography_STACK_OF_SCT *, SCT *) = NULL;

void (*SCT_LIST_free)(Cryptography_STACK_OF_SCT *) = NULL;
SCT *(*SCT_new)(void) = NULL;
int (*SCT_set1_log_id)(SCT *, unsigned char *, size_t) = NULL;
void (*SCT_set_timestamp)(SCT *, uint64_t) = NULL;
int (*SCT_set_version)(SCT *, sct_version_t) = NULL;
int (*SCT_set_log_entry_type)(SCT *, ct_log_entry_type_t) = NULL;
#endif


#if CRYPTOGRAPHY_IS_LIBRESSL
#ifndef DH_CHECK_Q_NOT_PRIME
#define DH_CHECK_Q_NOT_PRIME            0x10
#endif

#ifndef DH_CHECK_INVALID_Q_VALUE
#define DH_CHECK_INVALID_Q_VALUE        0x20
#endif

#ifndef DH_CHECK_INVALID_J_VALUE
#define DH_CHECK_INVALID_J_VALUE        0x40
#endif

/* DH_check implementation taken from OpenSSL 1.1.0pre6 */

/*-
 * Check that p is a safe prime and
 * if g is 2, 3 or 5, check that it is a suitable generator
 * where
 * for 2, p mod 24 == 11
 * for 3, p mod 12 == 5
 * for 5, p mod 10 == 3 or 7
 * should hold.
 */

int Cryptography_DH_check(const DH *dh, int *ret)
{
    int ok = 0, r;
    BN_CTX *ctx = NULL;
    BN_ULONG l;
    BIGNUM *t1 = NULL, *t2 = NULL;

    *ret = 0;
    ctx = BN_CTX_new();
    if (ctx == NULL)
        goto err;
    BN_CTX_start(ctx);
    t1 = BN_CTX_get(ctx);
    if (t1 == NULL)
        goto err;
    t2 = BN_CTX_get(ctx);
    if (t2 == NULL)
        goto err;

    if (dh->q) {
        if (BN_cmp(dh->g, BN_value_one()) <= 0)
            *ret |= DH_NOT_SUITABLE_GENERATOR;
        else if (BN_cmp(dh->g, dh->p) >= 0)
            *ret |= DH_NOT_SUITABLE_GENERATOR;
        else {
            /* Check g^q == 1 mod p */
            if (!BN_mod_exp(t1, dh->g, dh->q, dh->p, ctx))
                goto err;
            if (!BN_is_one(t1))
                *ret |= DH_NOT_SUITABLE_GENERATOR;
        }
        r = BN_is_prime_ex(dh->q, BN_prime_checks, ctx, NULL);
        if (r < 0)
            goto err;
        if (!r)
            *ret |= DH_CHECK_Q_NOT_PRIME;
        /* Check p == 1 mod q  i.e. q divides p - 1 */
        if (!BN_div(t1, t2, dh->p, dh->q, ctx))
            goto err;
        if (!BN_is_one(t2))
            *ret |= DH_CHECK_INVALID_Q_VALUE;
        if (dh->j && BN_cmp(dh->j, t1))
            *ret |= DH_CHECK_INVALID_J_VALUE;

    } else if (BN_is_word(dh->g, DH_GENERATOR_2)) {
        l = BN_mod_word(dh->p, 24);
        if (l == (BN_ULONG)-1)
            goto err;
        if (l != 11)
            *ret |= DH_NOT_SUITABLE_GENERATOR;
    } else if (BN_is_word(dh->g, DH_GENERATOR_5)) {
        l = BN_mod_word(dh->p, 10);
        if (l == (BN_ULONG)-1)
            goto err;
        if ((l != 3) && (l != 7))
            *ret |= DH_NOT_SUITABLE_GENERATOR;
    } else
        *ret |= DH_UNABLE_TO_CHECK_GENERATOR;

    r = BN_is_prime_ex(dh->p, BN_prime_checks, ctx, NULL);
    if (r < 0)
        goto err;
    if (!r)
        *ret |= DH_CHECK_P_NOT_PRIME;
    else if (!dh->q) {
        if (!BN_rshift1(t1, dh->p))
            goto err;
        r = BN_is_prime_ex(t1, BN_prime_checks, ctx, NULL);
        if (r < 0)
            goto err;
        if (!r)
            *ret |= DH_CHECK_P_NOT_SAFE_PRIME;
    }
    ok = 1;
 err:
    if (ctx != NULL) {
        BN_CTX_end(ctx);
        BN_CTX_free(ctx);
    }
    return (ok);
}
#else
int Cryptography_DH_check(const DH *dh, int *ret) {
    return DH_check(dh, ret);
}
#endif

/* These functions were added in OpenSSL 1.1.0f commit d0c50e80a8 */
/* Define our own to simplify support across all versions. */
#if defined(EVP_PKEY_DHX) && EVP_PKEY_DHX != -1
DH *Cryptography_d2i_DHxparams_bio(BIO *bp, DH **x) {
    return ASN1_d2i_bio_of(DH, DH_new, d2i_DHxparams, bp, x);
}
int Cryptography_i2d_DHxparams_bio(BIO *bp, DH *x) {
    return ASN1_i2d_bio_of_const(DH, i2d_DHxparams, bp, x);
}
#else
DH *(*Cryptography_d2i_DHxparams_bio)(BIO *bp, DH **x) = NULL;
int (*Cryptography_i2d_DHxparams_bio)(BIO *bp, DH *x) = NULL;
#endif




#if defined(OPENSSL_NO_EC2M)
static const long Cryptography_HAS_EC2M = 0;

int (*EC_POINT_set_affine_coordinates_GF2m)(const EC_GROUP *, EC_POINT *,
    const BIGNUM *, const BIGNUM *, BN_CTX *) = NULL;

int (*EC_POINT_get_affine_coordinates_GF2m)(const EC_GROUP *,
    const EC_POINT *, BIGNUM *, BIGNUM *, BN_CTX *) = NULL;

int (*EC_POINT_set_compressed_coordinates_GF2m)(const EC_GROUP *, EC_POINT *,
    const BIGNUM *, int, BN_CTX *) = NULL;
#else
static const long Cryptography_HAS_EC2M = 1;
#endif






#ifdef OPENSSL_NO_ENGINE
static const long Cryptography_HAS_ENGINE = 0;

ENGINE *(*ENGINE_by_id)(const char *) = NULL;
int (*ENGINE_init)(ENGINE *) = NULL;
int (*ENGINE_finish)(ENGINE *) = NULL;
ENGINE *(*ENGINE_get_default_RAND)(void) = NULL;
int (*ENGINE_set_default_RAND)(ENGINE *) = NULL;
void (*ENGINE_unregister_RAND)(ENGINE *) = NULL;
int (*ENGINE_ctrl_cmd)(ENGINE *, const char *, long, void *,
                       void (*)(void), int) = NULL;

int (*ENGINE_free)(ENGINE *) = NULL;
const char *(*ENGINE_get_id)(const ENGINE *) = NULL;
const char *(*ENGINE_get_name)(const ENGINE *) = NULL;

int (*ENGINE_ctrl_cmd_string)(ENGINE *, const char *, const char *,
                              int) = NULL;
void (*ENGINE_load_builtin_engines)(void) = NULL;
EVP_PKEY *(*ENGINE_load_private_key)(ENGINE *, const char *, UI_METHOD *,
                                     void *) = NULL;
EVP_PKEY *(*ENGINE_load_public_key)(ENGINE *, const char *,
                                    UI_METHOD *, void *) = NULL;

#else
static const long Cryptography_HAS_ENGINE = 1;
#endif




#ifdef EVP_PKEY_DHX
const long Cryptography_HAS_EVP_PKEY_DHX = 1;
#else
const long Cryptography_HAS_EVP_PKEY_DHX = 0;
const long EVP_PKEY_DHX = -1;
#endif

int Cryptography_EVP_PKEY_id(const EVP_PKEY *key) {
    return EVP_PKEY_id(key);
}
EVP_MD_CTX *Cryptography_EVP_MD_CTX_new(void) {
    return EVP_MD_CTX_new();
}
void Cryptography_EVP_MD_CTX_free(EVP_MD_CTX *md) {
    EVP_MD_CTX_free(md);
}

#if CRYPTOGRAPHY_IS_LIBRESSL || defined(OPENSSL_NO_SCRYPT)
static const long Cryptography_HAS_SCRYPT = 0;
int (*EVP_PBE_scrypt)(const char *, size_t, const unsigned char *, size_t,
                      uint64_t, uint64_t, uint64_t, uint64_t, unsigned char *,
                      size_t) = NULL;
#else
static const long Cryptography_HAS_SCRYPT = 1;
#endif

#if !CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint = 1;
#else
static const long Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint = 0;
size_t (*EVP_PKEY_get1_tls_encodedpoint)(EVP_PKEY *, unsigned char **) = NULL;
int (*EVP_PKEY_set1_tls_encodedpoint)(EVP_PKEY *, const unsigned char *,
                                      size_t) = NULL;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_111
static const long Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY = 0;
static const long Cryptography_HAS_RAW_KEY = 0;
static const long Cryptography_HAS_EVP_DIGESTFINAL_XOF = 0;
int (*EVP_DigestFinalXOF)(EVP_MD_CTX *, unsigned char *, size_t) = NULL;
int (*EVP_DigestSign)(EVP_MD_CTX *, unsigned char *, size_t *,
                      const unsigned char *tbs, size_t) = NULL;
int (*EVP_DigestVerify)(EVP_MD_CTX *, const unsigned char *, size_t,
                        const unsigned char *, size_t) = NULL;
EVP_PKEY *(*EVP_PKEY_new_raw_private_key)(int, ENGINE *, const unsigned char *,
                                       size_t) = NULL;
EVP_PKEY *(*EVP_PKEY_new_raw_public_key)(int, ENGINE *, const unsigned char *,
                                      size_t) = NULL;
int (*EVP_PKEY_get_raw_private_key)(const EVP_PKEY *, unsigned char *,
                                    size_t *) = NULL;
int (*EVP_PKEY_get_raw_public_key)(const EVP_PKEY *, unsigned char *,
                                   size_t *) = NULL;
#else
static const long Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY = 1;
static const long Cryptography_HAS_RAW_KEY = 1;
static const long Cryptography_HAS_EVP_DIGESTFINAL_XOF = 1;
#endif

/* OpenSSL 1.1.0+ does this define for us, but if not present we'll do it */
#if !defined(EVP_CTRL_AEAD_SET_IVLEN)
# define EVP_CTRL_AEAD_SET_IVLEN EVP_CTRL_GCM_SET_IVLEN
#endif
#if !defined(EVP_CTRL_AEAD_GET_TAG)
# define EVP_CTRL_AEAD_GET_TAG EVP_CTRL_GCM_GET_TAG
#endif
#if !defined(EVP_CTRL_AEAD_SET_TAG)
# define EVP_CTRL_AEAD_SET_TAG EVP_CTRL_GCM_SET_TAG
#endif

/* This is tied to X25519 support so we reuse the Cryptography_HAS_X25519
   conditional to remove it. OpenSSL 1.1.0 didn't have this define, but
   1.1.1 will when it is released. We can remove this in the distant
   future when we drop 1.1.0 support. */
#ifndef EVP_PKEY_X25519
#define EVP_PKEY_X25519 NID_X25519
#endif

/* This is tied to X448 support so we reuse the Cryptography_HAS_X448
   conditional to remove it. OpenSSL 1.1.1 adds this define.  We can remove
   this in the distant future when we drop 1.1.0 support. */
#ifndef EVP_PKEY_X448
#define EVP_PKEY_X448 NID_X448
#endif

/* This is tied to ED25519 support so we reuse the Cryptography_HAS_ED25519
   conditional to remove it. */
#ifndef EVP_PKEY_ED25519
#define EVP_PKEY_ED25519 NID_ED25519
#endif

/* This is tied to ED448 support so we reuse the Cryptography_HAS_ED448
   conditional to remove it. */
#ifndef EVP_PKEY_ED448
#define EVP_PKEY_ED448 NID_ED448
#endif

/* This is tied to poly1305 support so we reuse the Cryptography_HAS_POLY1305
   conditional to remove it. */
#ifndef EVP_PKEY_POLY1305
#define EVP_PKEY_POLY1305 NID_poly1305
#endif


#if CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_FIPS = 0;
int (*FIPS_mode_set)(int) = NULL;
int (*FIPS_mode)(void) = NULL;
#else
static const long Cryptography_HAS_FIPS = 1;
#endif




#ifndef NID_ED25519
static const long Cryptography_HAS_ED25519 = 0;
static const int NID_ED25519 = 0;
#else
static const long Cryptography_HAS_ED25519 = 1;
#endif
#ifndef NID_ED448
static const long Cryptography_HAS_ED448 = 0;
static const int NID_ED448 = 0;
#else
static const long Cryptography_HAS_ED448 = 1;
#endif
#ifndef NID_poly1305
static const long Cryptography_HAS_POLY1305 = 0;
static const int NID_poly1305 = 0;
#else
static const long Cryptography_HAS_POLY1305 = 1;
#endif




#if (     !CRYPTOGRAPHY_IS_LIBRESSL &&     CRYPTOGRAPHY_OPENSSL_LESS_THAN_110J     )
/* These structs come from ocsp_lcl.h and are needed to de-opaque the struct
   for the getters in OpenSSL 1.1.0 through 1.1.0i */
struct ocsp_responder_id_st {
    int type;
    union {
        X509_NAME *byName;
        ASN1_OCTET_STRING *byKey;
    } value;
};
struct ocsp_response_data_st {
    ASN1_INTEGER *version;
    OCSP_RESPID responderId;
    ASN1_GENERALIZEDTIME *producedAt;
    STACK_OF(OCSP_SINGLERESP) *responses;
    STACK_OF(X509_EXTENSION) *responseExtensions;
};
struct ocsp_basic_response_st {
    OCSP_RESPDATA tbsResponseData;
    X509_ALGOR signatureAlgorithm;
    ASN1_BIT_STRING *signature;
    STACK_OF(X509) *certs;
};
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL
/* These functions are all taken from ocsp_cl.c in OpenSSL 1.1.0 */
const OCSP_CERTID *OCSP_SINGLERESP_get0_id(const OCSP_SINGLERESP *single)
{
    return single->certId;
}
const Cryptography_STACK_OF_X509 *OCSP_resp_get0_certs(
    const OCSP_BASICRESP *bs)
{
    return bs->certs;
}
int OCSP_resp_get0_id(const OCSP_BASICRESP *bs,
                      const ASN1_OCTET_STRING **pid,
                      const X509_NAME **pname)
{
    const OCSP_RESPID *rid = bs->tbsResponseData->responderId;

    if (rid->type == V_OCSP_RESPID_NAME) {
        *pname = rid->value.byName;
        *pid = NULL;
    } else if (rid->type == V_OCSP_RESPID_KEY) {
        *pid = rid->value.byKey;
        *pname = NULL;
    } else {
        return 0;
    }
    return 1;
}
const ASN1_GENERALIZEDTIME *OCSP_resp_get0_produced_at(
    const OCSP_BASICRESP* bs)
{
    return bs->tbsResponseData->producedAt;
}
const ASN1_OCTET_STRING *OCSP_resp_get0_signature(const OCSP_BASICRESP *bs)
{
    return bs->signature;
}
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_110J
const X509_ALGOR *OCSP_resp_get0_tbs_sigalg(const OCSP_BASICRESP *bs)
{
#if CRYPTOGRAPHY_IS_LIBRESSL
    return bs->signatureAlgorithm;
#else
    return &bs->signatureAlgorithm;
#endif
}

const OCSP_RESPDATA *OCSP_resp_get0_respdata(const OCSP_BASICRESP *bs)
{
#if CRYPTOGRAPHY_IS_LIBRESSL
    return bs->tbsResponseData;
#else
    return &bs->tbsResponseData;
#endif
}
#endif



/* osurandom engine
 *
 * Windows         CryptGenRandom()
 * macOS >= 10.12  getentropy()
 * OpenBSD 5.6+    getentropy()
 * other BSD       getentropy() if SYS_getentropy is defined
 * Linux 3.17+     getrandom() with fallback to /dev/urandom
 * other           /dev/urandom with cached fd
 *
 * The /dev/urandom, getrandom and getentropy code is derived from Python's
 * Python/random.c, written by Antoine Pitrou and Victor Stinner.
 *
 * Copyright 2001-2016 Python Software Foundation; All Rights Reserved.
 */

#ifdef __linux__
#include <poll.h>
#endif

#if CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE
/* OpenSSL has ENGINE support and is older than 1.1.1d (the first version that
 * properly implements fork safety in its RNG) so build the engine. */
static const char *Cryptography_osrandom_engine_id = "osrandom";

/****************************************************************************
 * Windows
 */
#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM
static const char *Cryptography_osrandom_engine_name = "osrandom_engine CryptGenRandom()";
static HCRYPTPROV hCryptProv = 0;

static int osrandom_init(ENGINE *e) {
    if (hCryptProv != 0) {
        return 1;
    }
    if (CryptAcquireContext(&hCryptProv, NULL, NULL,
                            PROV_RSA_FULL, CRYPT_VERIFYCONTEXT)) {
        return 1;
    } else {
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_INIT,
            CRYPTOGRAPHY_OSRANDOM_R_CRYPTACQUIRECONTEXT,
            __FILE__, __LINE__
        );
        return 0;
    }
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    if (hCryptProv == 0) {
        return 0;
    }

    if (!CryptGenRandom(hCryptProv, (DWORD)size, buffer)) {
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
            CRYPTOGRAPHY_OSRANDOM_R_CRYPTGENRANDOM,
            __FILE__, __LINE__
        );
        return 0;
    }
    return 1;
}

static int osrandom_finish(ENGINE *e) {
    if (CryptReleaseContext(hCryptProv, 0)) {
        hCryptProv = 0;
        return 1;
    } else {
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_FINISH,
            CRYPTOGRAPHY_OSRANDOM_R_CRYPTRELEASECONTEXT,
            __FILE__, __LINE__
        );
        return 0;
    }
}

static int osrandom_rand_status(void) {
    return hCryptProv != 0;
}

static const char *osurandom_get_implementation(void) {
    return "CryptGenRandom";
}

#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_CRYPTGENRANDOM */

/****************************************************************************
 * /dev/urandom helpers for all non-BSD Unix platforms
 */
#ifdef CRYPTOGRAPHY_OSRANDOM_NEEDS_DEV_URANDOM

static struct {
    int fd;
    dev_t st_dev;
    ino_t st_ino;
} urandom_cache = { -1 };

static int open_cloexec(const char *path) {
    int open_flags = O_RDONLY;
#ifdef O_CLOEXEC
    open_flags |= O_CLOEXEC;
#endif

    int fd = open(path, open_flags);
    if (fd == -1) {
        return -1;
    }

#ifndef O_CLOEXEC
    int flags = fcntl(fd, F_GETFD);
    if (flags == -1) {
        return -1;
    }
    if (fcntl(fd, F_SETFD, flags | FD_CLOEXEC) == -1) {
        return -1;
    }
#endif
    return fd;
}

#ifdef __linux__
/* On Linux, we open("/dev/random") and use poll() to wait until it's readable
 * before we read from /dev/urandom, this ensures that we don't read from
 * /dev/urandom before the kernel CSPRNG is initialized. This isn't necessary on
 * other platforms because they don't have the same _bug_ as Linux does with
 * /dev/urandom and early boot. */
static int wait_on_devrandom(void) {
    struct pollfd pfd = {};
    int ret = 0;
    int random_fd = open_cloexec("/dev/random");
    if (random_fd < 0) {
        return -1;
    }
    pfd.fd = random_fd;
    pfd.events = POLLIN;
    pfd.revents = 0;
    do {
        ret = poll(&pfd, 1, -1);
    } while (ret < 0 && (errno == EINTR || errno == EAGAIN));
    close(random_fd);
    return ret;
}
#endif

/* return -1 on error */
static int dev_urandom_fd(void) {
    int fd = -1;
    struct stat st;

    /* Check that fd still points to the correct device */
    if (urandom_cache.fd >= 0) {
        if (fstat(urandom_cache.fd, &st)
                || st.st_dev != urandom_cache.st_dev
                || st.st_ino != urandom_cache.st_ino) {
            /* Somebody replaced our FD. Invalidate our cache but don't
             * close the fd. */
            urandom_cache.fd = -1;
        }
    }
    if (urandom_cache.fd < 0) {
#ifdef __linux__
        if (wait_on_devrandom() < 0) {
            goto error;
        }
#endif

        fd = open_cloexec("/dev/urandom");
        if (fd < 0) {
            goto error;
        }
        if (fstat(fd, &st)) {
            goto error;
        }
        /* Another thread initialized the fd */
        if (urandom_cache.fd >= 0) {
            close(fd);
            return urandom_cache.fd;
        }
        urandom_cache.st_dev = st.st_dev;
        urandom_cache.st_ino = st.st_ino;
        urandom_cache.fd = fd;
    }
    return urandom_cache.fd;

  error:
    if (fd != -1) {
        close(fd);
    }
    ERR_Cryptography_OSRandom_error(
        CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_FD,
        CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_OPEN_FAILED,
        __FILE__, __LINE__
    );
    return -1;
}

static int dev_urandom_read(unsigned char *buffer, int size) {
    int fd;
    int n;

    fd = dev_urandom_fd();
    if (fd < 0) {
        return 0;
    }

    while (size > 0) {
        do {
            n = (int)read(fd, buffer, (size_t)size);
        } while (n < 0 && errno == EINTR);

        if (n <= 0) {
            ERR_Cryptography_OSRandom_error(
                CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_READ,
                CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_READ_FAILED,
                __FILE__, __LINE__
            );
            return 0;
        }
        buffer += n;
        size -= n;
    }
    return 1;
}

static void dev_urandom_close(void) {
    if (urandom_cache.fd >= 0) {
        int fd;
        struct stat st;

        if (fstat(urandom_cache.fd, &st)
                && st.st_dev == urandom_cache.st_dev
                && st.st_ino == urandom_cache.st_ino) {
            fd = urandom_cache.fd;
            urandom_cache.fd = -1;
            close(fd);
        }
    }
}
#endif /* CRYPTOGRAPHY_OSRANDOM_NEEDS_DEV_URANDOM */

/****************************************************************************
 * BSD getentropy
 */
#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY
static const char *Cryptography_osrandom_engine_name = "osrandom_engine getentropy()";

static int getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_NOT_INIT;

static int osrandom_init(ENGINE *e) {
#if !defined(__APPLE__)
    getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS;
#else
    if (__builtin_available(macOS 10.12, *)) {
        getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS;
    } else {
        getentropy_works = CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK;
        int fd = dev_urandom_fd();
        if (fd < 0) {
            return 0;
        }
    }
#endif
    return 1;
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    int len;
    int res;

    switch(getentropy_works) {
#if defined(__APPLE__)
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK:
        return dev_urandom_read(buffer, size);
#endif
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS:
        while (size > 0) {
            /* OpenBSD and macOS restrict maximum buffer size to 256. */
            len = size > 256 ? 256 : size;
/* on mac, availability is already checked using `__builtin_available` above */
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wunguarded-availability"
            res = getentropy(buffer, (size_t)len);
#pragma clang diagnostic pop
            if (res < 0) {
                ERR_Cryptography_OSRandom_error(
                    CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
                    CRYPTOGRAPHY_OSRANDOM_R_GETENTROPY_FAILED,
                    __FILE__, __LINE__
                );
                return 0;
            }
            buffer += len;
            size -= len;
        }
        return 1;
    }
    __builtin_unreachable();
}

static int osrandom_finish(ENGINE *e) {
    return 1;
}

static int osrandom_rand_status(void) {
    return 1;
}

static const char *osurandom_get_implementation(void) {
    switch(getentropy_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_FALLBACK:
        return "/dev/urandom";
    case CRYPTOGRAPHY_OSRANDOM_GETENTROPY_WORKS:
        return "getentropy";
    }
    __builtin_unreachable();
}
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_GETENTROPY */

/****************************************************************************
 * Linux getrandom engine with fallback to dev_urandom
 */

#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM
static const char *Cryptography_osrandom_engine_name = "osrandom_engine getrandom()";

static int getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT;

static int osrandom_init(ENGINE *e) {
    /* We try to detect working getrandom until we succeed. */
    if (getrandom_works != CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS) {
        long n;
        char dest[1];
        /* if the kernel CSPRNG is not initialized this will block */
        n = syscall(SYS_getrandom, dest, sizeof(dest), 0);
        if (n == sizeof(dest)) {
            getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS;
        } else {
            int e = errno;
            switch(e) {
            case ENOSYS:
                /* Fallback: Kernel does not support the syscall. */
                getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK;
                break;
            case EPERM:
                /* Fallback: seccomp prevents syscall */
                getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK;
                break;
            default:
                /* EINTR cannot occur for buflen < 256. */
                ERR_Cryptography_OSRandom_error(
                    CRYPTOGRAPHY_OSRANDOM_F_INIT,
                    CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED_UNEXPECTED,
                    "errno", e
                );
                getrandom_works = CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED;
                break;
            }
        }
    }

    /* fallback to dev urandom */
    if (getrandom_works == CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK) {
        int fd = dev_urandom_fd();
        if (fd < 0) {
            return 0;
        }
    }
    return 1;
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    long n;

    switch(getrandom_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED:
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
            CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED,
            __FILE__, __LINE__
        );
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT:
        ERR_Cryptography_OSRandom_error(
            CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
            CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_NOT_INIT,
            __FILE__, __LINE__
        );
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK:
        return dev_urandom_read(buffer, size);
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS:
        while (size > 0) {
            do {
                n = syscall(SYS_getrandom, buffer, size, 0);
            } while (n < 0 && errno == EINTR);

            if (n <= 0) {
                ERR_Cryptography_OSRandom_error(
                    CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES,
                    CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_FAILED,
                    __FILE__, __LINE__
                );
                return 0;
            }
            buffer += n;
            size -= (int)n;
        }
        return 1;
    }
    __builtin_unreachable();
}

static int osrandom_finish(ENGINE *e) {
    dev_urandom_close();
    return 1;
}

static int osrandom_rand_status(void) {
    switch(getrandom_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED:
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT:
        return 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK:
        return urandom_cache.fd >= 0;
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS:
        return 1;
    }
    __builtin_unreachable();
}

static const char *osurandom_get_implementation(void) {
    switch(getrandom_works) {
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_INIT_FAILED:
        return "<failed>";
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_NOT_INIT:
        return "<not initialized>";
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_FALLBACK:
        return "/dev/urandom";
    case CRYPTOGRAPHY_OSRANDOM_GETRANDOM_WORKS:
        return "getrandom";
    }
    __builtin_unreachable();
}
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_GETRANDOM */

/****************************************************************************
 * dev_urandom engine for all remaining platforms
 */

#if CRYPTOGRAPHY_OSRANDOM_ENGINE == CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM
static const char *Cryptography_osrandom_engine_name = "osrandom_engine /dev/urandom";

static int osrandom_init(ENGINE *e) {
    int fd = dev_urandom_fd();
    if (fd < 0) {
        return 0;
    }
    return 1;
}

static int osrandom_rand_bytes(unsigned char *buffer, int size) {
    return dev_urandom_read(buffer, size);
}

static int osrandom_finish(ENGINE *e) {
    dev_urandom_close();
    return 1;
}

static int osrandom_rand_status(void) {
    return urandom_cache.fd >= 0;
}

static const char *osurandom_get_implementation(void) {
    return "/dev/urandom";
}
#endif /* CRYPTOGRAPHY_OSRANDOM_ENGINE_DEV_URANDOM */

/****************************************************************************
 * ENGINE boiler plate
 */

/* This replicates the behavior of the OpenSSL FIPS RNG, which returns a
   -1 in the event that there is an error when calling RAND_pseudo_bytes. */
static int osrandom_pseudo_rand_bytes(unsigned char *buffer, int size) {
    int res = osrandom_rand_bytes(buffer, size);
    if (res == 0) {
        return -1;
    } else {
        return res;
    }
}

static RAND_METHOD osrandom_rand = {
    NULL,
    osrandom_rand_bytes,
    NULL,
    NULL,
    osrandom_pseudo_rand_bytes,
    osrandom_rand_status,
};

static const ENGINE_CMD_DEFN osrandom_cmd_defns[] = {
    {CRYPTOGRAPHY_OSRANDOM_GET_IMPLEMENTATION,
     "get_implementation",
     "Get CPRNG implementation.",
     ENGINE_CMD_FLAG_NO_INPUT},
     {0, NULL, NULL, 0}
};

static int osrandom_ctrl(ENGINE *e, int cmd, long i, void *p, void (*f) (void)) {
    const char *name;
    size_t len;

    switch (cmd) {
    case CRYPTOGRAPHY_OSRANDOM_GET_IMPLEMENTATION:
        /* i: buffer size, p: char* buffer */
        name = osurandom_get_implementation();
        len = strlen(name);
        if ((p == NULL) && (i == 0)) {
            /* return required buffer len */
            return (int)len;
        }
        if ((p == NULL) || i < 0 || ((size_t)i <= len)) {
            /* no buffer or buffer too small */
            ENGINEerr(ENGINE_F_ENGINE_CTRL, ENGINE_R_INVALID_ARGUMENT);
            return 0;
        }
        strcpy((char *)p, name);
        return (int)len;
    default:
        ENGINEerr(ENGINE_F_ENGINE_CTRL, ENGINE_R_CTRL_COMMAND_NOT_IMPLEMENTED);
        return 0;
    }
}

/* error reporting */
#define ERR_FUNC(func) ERR_PACK(0, func, 0)
#define ERR_REASON(reason) ERR_PACK(0, 0, reason)

static ERR_STRING_DATA CRYPTOGRAPHY_OSRANDOM_lib_name[] = {
    {0, "osrandom_engine"},
    {0, NULL}
};

static ERR_STRING_DATA CRYPTOGRAPHY_OSRANDOM_str_funcs[] = {
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_INIT),
     "osrandom_init"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_RAND_BYTES),
     "osrandom_rand_bytes"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_FINISH),
     "osrandom_finish"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_FD),
     "dev_urandom_fd"},
    {ERR_FUNC(CRYPTOGRAPHY_OSRANDOM_F_DEV_URANDOM_READ),
     "dev_urandom_read"},
    {0, NULL}
};

static ERR_STRING_DATA CRYPTOGRAPHY_OSRANDOM_str_reasons[] = {
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_CRYPTACQUIRECONTEXT),
     "CryptAcquireContext() failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_CRYPTGENRANDOM),
     "CryptGenRandom() failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_CRYPTRELEASECONTEXT),
     "CryptReleaseContext() failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETENTROPY_FAILED),
     "getentropy() failed"},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_OPEN_FAILED),
     "open('/dev/urandom') failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_DEV_URANDOM_READ_FAILED),
     "Reading from /dev/urandom fd failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED),
     "getrandom() initialization failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_INIT_FAILED_UNEXPECTED),
     "getrandom() initialization failed with unexpected errno."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_FAILED),
     "getrandom() syscall failed."},
    {ERR_REASON(CRYPTOGRAPHY_OSRANDOM_R_GETRANDOM_NOT_INIT),
     "getrandom() engine was not properly initialized."},
    {0, NULL}
};

static int Cryptography_OSRandom_lib_error_code = 0;

static void ERR_load_Cryptography_OSRandom_strings(void)
{
    if (Cryptography_OSRandom_lib_error_code == 0) {
        Cryptography_OSRandom_lib_error_code = ERR_get_next_error_library();
        ERR_load_strings(Cryptography_OSRandom_lib_error_code,
                         CRYPTOGRAPHY_OSRANDOM_lib_name);
        ERR_load_strings(Cryptography_OSRandom_lib_error_code,
                         CRYPTOGRAPHY_OSRANDOM_str_funcs);
        ERR_load_strings(Cryptography_OSRandom_lib_error_code,
                         CRYPTOGRAPHY_OSRANDOM_str_reasons);
    }
}

static void ERR_Cryptography_OSRandom_error(int function, int reason,
                                            char *file, int line)
{
    ERR_PUT_error(Cryptography_OSRandom_lib_error_code, function, reason,
                  file, line);
}

/* Returns 1 if successfully added, 2 if engine has previously been added,
   and 0 for error. */
int Cryptography_add_osrandom_engine(void) {
    ENGINE *e;

    ERR_load_Cryptography_OSRandom_strings();

    e = ENGINE_by_id(Cryptography_osrandom_engine_id);
    if (e != NULL) {
        ENGINE_free(e);
        return 2;
    } else {
        ERR_clear_error();
    }

    e = ENGINE_new();
    if (e == NULL) {
        return 0;
    }
    if (!ENGINE_set_id(e, Cryptography_osrandom_engine_id) ||
            !ENGINE_set_name(e, Cryptography_osrandom_engine_name) ||
            !ENGINE_set_RAND(e, &osrandom_rand) ||
            !ENGINE_set_init_function(e, osrandom_init) ||
            !ENGINE_set_finish_function(e, osrandom_finish) ||
            !ENGINE_set_cmd_defns(e, osrandom_cmd_defns) ||
            !ENGINE_set_ctrl_function(e, osrandom_ctrl)) {
        ENGINE_free(e);
        return 0;
    }
    if (!ENGINE_add(e)) {
        ENGINE_free(e);
        return 0;
    }
    if (!ENGINE_free(e)) {
        return 0;
    }

    return 1;
}

#else
/* If OpenSSL has no ENGINE support then we don't want
 * to compile the osrandom engine, but we do need some
 * placeholders */
static const char *Cryptography_osrandom_engine_id = "no-engine-support";
static const char *Cryptography_osrandom_engine_name = "osrandom_engine disabled";

int Cryptography_add_osrandom_engine(void) {
    return 0;
}

#endif


#if !defined(EVP_PKEY_DHX) || EVP_PKEY_DHX == -1
int (*PEM_write_bio_DHxparams)(BIO *, DH *) = NULL;
#endif






#if !CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_RSA_OAEP_MD = 1;
static const long Cryptography_HAS_RSA_OAEP_LABEL = 1;
#else
static const long Cryptography_HAS_RSA_OAEP_MD = 0;
static const long Cryptography_HAS_RSA_OAEP_LABEL = 0;
int (*EVP_PKEY_CTX_set_rsa_oaep_md)(EVP_PKEY_CTX *, EVP_MD *) = NULL;
int (*EVP_PKEY_CTX_set0_rsa_oaep_label)(EVP_PKEY_CTX *, unsigned char *,
                                        int) = NULL;
#endif


// This symbol is being preserved because removing it will break users with
// pyOpenSSL < 19.1 and pip < 20.x. We need to leave this in place until those
// users have upgraded. PersistentlyDeprecated2020
static const long Cryptography_HAS_TLSEXT_HOSTNAME = 1;

#if CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_VERIFIED_CHAIN = 0;
Cryptography_STACK_OF_X509 *(*SSL_get0_verified_chain)(const SSL *) = NULL;
#else
static const long Cryptography_HAS_VERIFIED_CHAIN = 1;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_111
static const long Cryptography_HAS_KEYLOG = 0;
void (*SSL_CTX_set_keylog_callback)(SSL_CTX *,
                                    void (*) (const SSL *, const char *)
                                    ) = NULL;
void (*(*SSL_CTX_get_keylog_callback)(SSL_CTX *))(
                                                  const SSL *,
                                                  const char *
                                                  ) = NULL;
#else
static const long Cryptography_HAS_KEYLOG = 1;
#endif

static const long Cryptography_HAS_SECURE_RENEGOTIATION = 1;

#ifdef OPENSSL_NO_SSL3_METHOD
static const long Cryptography_HAS_SSL3_METHOD = 0;
SSL_METHOD* (*SSLv3_method)(void) = NULL;
SSL_METHOD* (*SSLv3_client_method)(void) = NULL;
SSL_METHOD* (*SSLv3_server_method)(void) = NULL;
#else
static const long Cryptography_HAS_SSL3_METHOD = 1;
#endif

static const long Cryptography_HAS_RELEASE_BUFFERS = 1;
static const long Cryptography_HAS_OP_NO_COMPRESSION = 1;
static const long Cryptography_HAS_TLSv1_1 = 1;
static const long Cryptography_HAS_TLSv1_2 = 1;
static const long Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING = 1;
static const long Cryptography_HAS_SSL_OP_NO_TICKET = 1;
static const long Cryptography_HAS_SSL_SET_SSL_CTX = 1;
static const long Cryptography_HAS_NEXTPROTONEG = 0;
static const long Cryptography_HAS_ALPN = 1;

#if CRYPTOGRAPHY_IS_LIBRESSL
void (*SSL_CTX_set_cert_cb)(SSL_CTX *, int (*)(SSL *, void *), void *) = NULL;
void (*SSL_set_cert_cb)(SSL *, int (*)(SSL *, void *), void *) = NULL;
static const long Cryptography_HAS_SET_CERT_CB = 0;
#else
static const long Cryptography_HAS_SET_CERT_CB = 1;
#endif

static const long Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS = 1;

/* in OpenSSL 1.1.0 the SSL_ST values were renamed to TLS_ST and several were
   removed */
#if CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_SSL_ST = 1;
#else
static const long Cryptography_HAS_SSL_ST = 0;
static const long SSL_ST_BEFORE = 0;
static const long SSL_ST_OK = 0;
static const long SSL_ST_INIT = 0;
static const long SSL_ST_RENEGOTIATE = 0;
#endif
#if !CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_TLS_ST = 1;
#else
static const long Cryptography_HAS_TLS_ST = 0;
static const long TLS_ST_BEFORE = 0;
static const long TLS_ST_OK = 0;
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL
static const long SSL_OP_NO_DTLSv1 = 0;
static const long SSL_OP_NO_DTLSv1_2 = 0;
long (*DTLS_set_link_mtu)(SSL *, long) = NULL;
long (*DTLS_get_link_min_mtu)(SSL *) = NULL;
#endif

static const long Cryptography_HAS_DTLS = 1;
/* Wrap DTLSv1_get_timeout to avoid cffi to handle a 'struct timeval'. */
long Cryptography_DTLSv1_get_timeout(SSL *ssl, time_t *ptv_sec,
                                     long *ptv_usec) {
    struct timeval tv = { 0 };
    long r = DTLSv1_get_timeout(ssl, &tv);

    if (r == 1) {
        if (ptv_sec) {
            *ptv_sec = tv.tv_sec;
        }

        if (ptv_usec) {
            *ptv_usec = tv.tv_usec;
        }
    }

    return r;
}

#if CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_SIGALGS = 0;
const int (*SSL_get_sigalgs)(SSL *, int, int *, int *, int *, unsigned char *,
                             unsigned char *) = NULL;
const long (*SSL_CTX_set1_sigalgs_list)(SSL_CTX *, const char *) = NULL;
#else
static const long Cryptography_HAS_SIGALGS = 1;
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL || defined(OPENSSL_NO_PSK)
static const long Cryptography_HAS_PSK = 0;
int (*SSL_CTX_use_psk_identity_hint)(SSL_CTX *, const char *) = NULL;
void (*SSL_CTX_set_psk_server_callback)(SSL_CTX *,
                                        unsigned int (*)(
                                            SSL *,
                                            const char *,
                                            unsigned char *,
                                            unsigned int
                                        )) = NULL;
void (*SSL_CTX_set_psk_client_callback)(SSL_CTX *,
                                        unsigned int (*)(
                                            SSL *,
                                            const char *,
                                            char *,
                                            unsigned int,
                                            unsigned char *,
                                            unsigned int
                                        )) = NULL;
#else
static const long Cryptography_HAS_PSK = 1;
#endif

#if !CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_CUSTOM_EXT = 1;
#else
static const long Cryptography_HAS_CUSTOM_EXT = 0;
typedef int (*custom_ext_add_cb)(SSL *, unsigned int,
                                 const unsigned char **,
                                 size_t *, int *,
                                 void *);
typedef void (*custom_ext_free_cb)(SSL *, unsigned int,
                                   const unsigned char *,
                                   void *);
typedef int (*custom_ext_parse_cb)(SSL *, unsigned int,
                                   const unsigned char *,
                                   size_t, int *,
                                   void *);
int (*SSL_CTX_add_client_custom_ext)(SSL_CTX *, unsigned int,
                                     custom_ext_add_cb,
                                     custom_ext_free_cb, void *,
                                     custom_ext_parse_cb,
                                     void *) = NULL;
int (*SSL_CTX_add_server_custom_ext)(SSL_CTX *, unsigned int,
                                     custom_ext_add_cb,
                                     custom_ext_free_cb, void *,
                                     custom_ext_parse_cb,
                                     void *) = NULL;
int (*SSL_extension_supported)(unsigned int) = NULL;
#endif

#ifndef OPENSSL_NO_SRTP
static const long Cryptography_HAS_SRTP = 1;
#else
static const long Cryptography_HAS_SRTP = 0;
int (*SSL_CTX_set_tlsext_use_srtp)(SSL_CTX *, const char *) = NULL;
int (*SSL_set_tlsext_use_srtp)(SSL *, const char *) = NULL;
SRTP_PROTECTION_PROFILE * (*SSL_get_selected_srtp_profile)(SSL *) = NULL;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_111
static const long Cryptography_HAS_TLSv1_3 = 0;
static const long SSL_OP_NO_TLSv1_3 = 0;
static const long SSL_VERIFY_POST_HANDSHAKE = 0;
int (*SSL_CTX_set_ciphersuites)(SSL_CTX *, const char *) = NULL;
int (*SSL_verify_client_post_handshake)(SSL *) = NULL;
void (*SSL_CTX_set_post_handshake_auth)(SSL_CTX *, int) = NULL;
void (*SSL_set_post_handshake_auth)(SSL *, int) = NULL;
uint32_t (*SSL_SESSION_get_max_early_data)(const SSL_SESSION *) = NULL;
int (*SSL_write_early_data)(SSL *, const void *, size_t, size_t *) = NULL;
int (*SSL_read_early_data)(SSL *, void *, size_t, size_t *) = NULL;
int (*SSL_CTX_set_max_early_data)(SSL_CTX *, uint32_t) = NULL;
#else
static const long Cryptography_HAS_TLSv1_3 = 1;
#endif

#if CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 && !CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_GET_PROTO_VERSION = 0;

long (*SSL_CTX_get_min_proto_version)(SSL_CTX *) = NULL;
long (*SSL_CTX_get_max_proto_version)(SSL_CTX *) = NULL;
long (*SSL_get_min_proto_version)(SSL *) = NULL;
long (*SSL_get_max_proto_version)(SSL *) = NULL;
#else
static const long Cryptography_HAS_GET_PROTO_VERSION = 1;
#endif


#if CRYPTOGRAPHY_IS_LIBRESSL
int i2d_re_X509_tbs(X509 *x, unsigned char **pp)
{
    /* in 1.0.2+ this function also sets x->cert_info->enc.modified = 1
       but older OpenSSLs don't have the enc ASN1_ENCODING member in the
       X509 struct.  Setting modified to 1 marks the encoding
       (x->cert_info->enc.enc) as invalid, but since the entire struct isn't
       present we don't care. */
    return i2d_X509_CINF(x->cert_info, pp);
}
#endif

/* Being kept around for pyOpenSSL */
X509_REVOKED *Cryptography_X509_REVOKED_dup(X509_REVOKED *rev) {
    return X509_REVOKED_dup(rev);
}
/* Added in 1.1.0 but we need it in all versions now due to the great
   opaquing. */
#if CRYPTOGRAPHY_IS_LIBRESSL
int i2d_re_X509_REQ_tbs(X509_REQ *req, unsigned char **pp)
{
    req->req_info->enc.modified = 1;
    return i2d_X509_REQ_INFO(req->req_info, pp);
}
int i2d_re_X509_CRL_tbs(X509_CRL *crl, unsigned char **pp) {
    crl->crl->enc.modified = 1;
    return i2d_X509_CRL_INFO(crl->crl, pp);
}
#endif






#if !CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_102_VERIFICATION = 1;
#else
static const long Cryptography_HAS_102_VERIFICATION = 0;
static const long X509_V_ERR_SUITE_B_INVALID_VERSION = 0;
static const long X509_V_ERR_SUITE_B_INVALID_ALGORITHM = 0;
static const long X509_V_ERR_SUITE_B_INVALID_CURVE = 0;
static const long X509_V_ERR_SUITE_B_INVALID_SIGNATURE_ALGORITHM = 0;
static const long X509_V_ERR_SUITE_B_LOS_NOT_ALLOWED = 0;
static const long X509_V_ERR_SUITE_B_CANNOT_SIGN_P_384_WITH_P_256 = 0;
static const long X509_V_FLAG_SUITEB_128_LOS_ONLY = 0;
static const long X509_V_FLAG_SUITEB_192_LOS = 0;
static const long X509_V_FLAG_SUITEB_128_LOS = 0;
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_110_VERIFICATION_PARAMS = 0;
#ifndef X509_CHECK_FLAG_NEVER_CHECK_SUBJECT
static const long X509_CHECK_FLAG_NEVER_CHECK_SUBJECT = 0;
#endif
#else
static const long Cryptography_HAS_110_VERIFICATION_PARAMS = 1;
#endif

#if CRYPTOGRAPHY_IS_LIBRESSL
static const long Cryptography_HAS_X509_STORE_CTX_GET_ISSUER = 0;
typedef void *X509_STORE_CTX_get_issuer_fn;
X509_STORE_CTX_get_issuer_fn (*X509_STORE_get_get_issuer)(X509_STORE *) = NULL;
void (*X509_STORE_set_get_issuer)(X509_STORE *,
                                  X509_STORE_CTX_get_issuer_fn) = NULL;
#else
static const long Cryptography_HAS_X509_STORE_CTX_GET_ISSUER = 1;
#endif



typedef struct {
    char *password;
    int length;
    int called;
    int error;
    int maxsize;
} CRYPTOGRAPHY_PASSWORD_DATA;

int Cryptography_pem_password_cb(char *buf, int size,
                                  int rwflag, void *userdata) {
    /* The password cb is only invoked if OpenSSL decides the private
       key is encrypted. So this path only occurs if it needs a password */
    CRYPTOGRAPHY_PASSWORD_DATA *st = (CRYPTOGRAPHY_PASSWORD_DATA *)userdata;
    st->called += 1;
    st->maxsize = size;
    if (st->length == 0) {
        st->error = -1;
        return 0;
    } else if (st->length < size) {
        memcpy(buf, st->password, st->length);
        return st->length;
    } else {
        st->error = -2;
        return 0;
    }
}

#define CRYPTOGRAPHY_PACKAGE_VERSION "3.3.1"

/************************************************************/

static void *_cffi_types[] = {
/*  0 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1056), // ACCESS_DESCRIPTION *()(Cryptography_STACK_OF_ACCESS_DESCRIPTION *, int)
/*  1 */ _CFFI_OP(_CFFI_OP_POINTER, 3034), // Cryptography_STACK_OF_ACCESS_DESCRIPTION *
/*  2 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7), // int
/*  3 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/*  4 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1056), // ACCESS_DESCRIPTION *()(void)
/*  5 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/*  6 */ _CFFI_OP(_CFFI_OP_FUNCTION, 757), // ASN1_ENUMERATED *()(void)
/*  7 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/*  8 */ _CFFI_OP(_CFFI_OP_FUNCTION, 9), // ASN1_GENERALIZEDTIME *()(ASN1_GENERALIZEDTIME *, time_t)
/*  9 */ _CFFI_OP(_CFFI_OP_POINTER, 3010), // ASN1_GENERALIZEDTIME *
/* 10 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, _cffi_prim_int(sizeof(time_t), (
           ((time_t)-1) | 0 /* check that time_t is an integer type */
         ) <= 0)), // time_t
/* 11 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 12 */ _CFFI_OP(_CFFI_OP_FUNCTION, 9), // ASN1_GENERALIZEDTIME *()(ASN1_OCTET_STRING *, ASN1_GENERALIZEDTIME * *)
/* 13 */ _CFFI_OP(_CFFI_OP_POINTER, 3017), // ASN1_OCTET_STRING *
/* 14 */ _CFFI_OP(_CFFI_OP_POINTER, 9), // ASN1_GENERALIZEDTIME * *
/* 15 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 16 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3009), // ASN1_GENERALIZEDTIME const *()(OCSP_BASICRESP const *)
/* 17 */ _CFFI_OP(_CFFI_OP_POINTER, 3078), // OCSP_BASICRESP const *
/* 18 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 19 */ _CFFI_OP(_CFFI_OP_FUNCTION, 21), // ASN1_INTEGER *()(BIGNUM *, ASN1_INTEGER *)
/* 20 */ _CFFI_OP(_CFFI_OP_POINTER, 3021), // BIGNUM *
/* 21 */ _CFFI_OP(_CFFI_OP_POINTER, 3012), // ASN1_INTEGER *
/* 22 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 23 */ _CFFI_OP(_CFFI_OP_FUNCTION, 21), // ASN1_INTEGER *()(Cryptography_STACK_OF_ASN1_INTEGER *, int)
/* 24 */ _CFFI_OP(_CFFI_OP_POINTER, 3035), // Cryptography_STACK_OF_ASN1_INTEGER *
/* 25 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 26 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 27 */ _CFFI_OP(_CFFI_OP_FUNCTION, 21), // ASN1_INTEGER *()(X509 *)
/* 28 */ _CFFI_OP(_CFFI_OP_POINTER, 3119), // X509 *
/* 29 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 30 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3011), // ASN1_INTEGER const *()(X509_REVOKED const *)
/* 31 */ _CFFI_OP(_CFFI_OP_POINTER, 3131), // X509_REVOKED const *
/* 32 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 33 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3013), // ASN1_NULL *()(void)
/* 34 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 35 */ _CFFI_OP(_CFFI_OP_FUNCTION, 607), // ASN1_OBJECT *()(Cryptography_STACK_OF_ASN1_OBJECT *, int)
/* 36 */ _CFFI_OP(_CFFI_OP_POINTER, 3036), // Cryptography_STACK_OF_ASN1_OBJECT *
/* 37 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 38 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 39 */ _CFFI_OP(_CFFI_OP_FUNCTION, 607), // ASN1_OBJECT *()(X509_EXTENSION *)
/* 40 */ _CFFI_OP(_CFFI_OP_POINTER, 3125), // X509_EXTENSION *
/* 41 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 42 */ _CFFI_OP(_CFFI_OP_FUNCTION, 607), // ASN1_OBJECT *()(X509_NAME_ENTRY *)
/* 43 */ _CFFI_OP(_CFFI_OP_POINTER, 3128), // X509_NAME_ENTRY *
/* 44 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 45 */ _CFFI_OP(_CFFI_OP_FUNCTION, 607), // ASN1_OBJECT *()(char const *, int)
/* 46 */ _CFFI_OP(_CFFI_OP_POINTER, 3135), // char const *
/* 47 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 48 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 49 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(ASN1_OCTET_STRING *, long)
/* 50 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 51 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9), // long
/* 52 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 53 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(X509 *)
/* 54 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 55 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 56 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(X509_CRL *)
/* 57 */ _CFFI_OP(_CFFI_OP_POINTER, 3124), // X509_CRL *
/* 58 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 59 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(X509_EXTENSION *)
/* 60 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 61 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 62 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(X509_NAME_ENTRY *)
/* 63 */ _CFFI_OP(_CFFI_OP_NOOP, 43),
/* 64 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 65 */ _CFFI_OP(_CFFI_OP_FUNCTION, 13), // ASN1_OCTET_STRING *()(void)
/* 66 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 67 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3016), // ASN1_OCTET_STRING const *()(OCSP_BASICRESP const *)
/* 68 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 69 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 70 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3016), // ASN1_OCTET_STRING const *()(X509_REVOKED const *)
/* 71 */ _CFFI_OP(_CFFI_OP_NOOP, 31),
/* 72 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 73 */ _CFFI_OP(_CFFI_OP_FUNCTION, 801), // ASN1_TYPE *()(ASN1_TYPE * *, unsigned char const * *, long)
/* 74 */ _CFFI_OP(_CFFI_OP_POINTER, 801), // ASN1_TYPE * *
/* 75 */ _CFFI_OP(_CFFI_OP_POINTER, 103), // unsigned char const * *
/* 76 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 77 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 78 */ _CFFI_OP(_CFFI_OP_FUNCTION, 801), // ASN1_TYPE *()(X509_ATTRIBUTE *, int)
/* 79 */ _CFFI_OP(_CFFI_OP_POINTER, 3123), // X509_ATTRIBUTE *
/* 80 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 81 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 82 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2557), // AUTHORITY_KEYID *()(void)
/* 83 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 84 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2560), // BASIC_CONSTRAINTS *()(void)
/* 85 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 86 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(ASN1_INTEGER *, BIGNUM *)
/* 87 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 88 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 89 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 90 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(BIGNUM *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 91 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 92 */ _CFFI_OP(_CFFI_OP_POINTER, 3021), // BIGNUM const *
/* 93 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 94 */ _CFFI_OP(_CFFI_OP_POINTER, 3024), // BN_CTX *
/* 95 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 96 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(BIGNUM const *)
/* 97 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 98 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 99 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(BN_CTX *)
/* 100 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 101 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 102 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(unsigned char const *, int, BIGNUM *)
/* 103 */ _CFFI_OP(_CFFI_OP_POINTER, 3140), // unsigned char const *
/* 104 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 105 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 106 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 107 */ _CFFI_OP(_CFFI_OP_FUNCTION, 20), // BIGNUM *()(void)
/* 108 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 109 */ _CFFI_OP(_CFFI_OP_FUNCTION, 92), // BIGNUM const *()(EC_KEY const *)
/* 110 */ _CFFI_OP(_CFFI_OP_POINTER, 3055), // EC_KEY const *
/* 111 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 112 */ _CFFI_OP(_CFFI_OP_FUNCTION, 92), // BIGNUM const *()(void)
/* 113 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 114 */ _CFFI_OP(_CFFI_OP_FUNCTION, 186), // BIO *()(BIO_METHOD *)
/* 115 */ _CFFI_OP(_CFFI_OP_POINTER, 3023), // BIO_METHOD *
/* 116 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 117 */ _CFFI_OP(_CFFI_OP_FUNCTION, 186), // BIO *()(char const *, char const *)
/* 118 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 119 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 120 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 121 */ _CFFI_OP(_CFFI_OP_FUNCTION, 186), // BIO *()(int, int)
/* 122 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 123 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 124 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 125 */ _CFFI_OP(_CFFI_OP_FUNCTION, 186), // BIO *()(void const *, int)
/* 126 */ _CFFI_OP(_CFFI_OP_POINTER, 3142), // void const *
/* 127 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 128 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 129 */ _CFFI_OP(_CFFI_OP_FUNCTION, 115), // BIO_METHOD *()(void)
/* 130 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 131 */ _CFFI_OP(_CFFI_OP_FUNCTION, 94), // BN_CTX *()(void)
/* 132 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 133 */ _CFFI_OP(_CFFI_OP_FUNCTION, 830), // BN_MONT_CTX *()(void)
/* 134 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 135 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1031), // CMAC_CTX *()(void)
/* 136 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 137 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1), // Cryptography_STACK_OF_ACCESS_DESCRIPTION *()(void)
/* 138 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 139 */ _CFFI_OP(_CFFI_OP_FUNCTION, 24), // Cryptography_STACK_OF_ASN1_INTEGER *()(void)
/* 140 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 141 */ _CFFI_OP(_CFFI_OP_FUNCTION, 36), // Cryptography_STACK_OF_ASN1_OBJECT *()(void)
/* 142 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 143 */ _CFFI_OP(_CFFI_OP_FUNCTION, 204), // Cryptography_STACK_OF_DIST_POINT *()(void)
/* 144 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 145 */ _CFFI_OP(_CFFI_OP_FUNCTION, 329), // Cryptography_STACK_OF_GENERAL_SUBTREE *()(void)
/* 146 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 147 */ _CFFI_OP(_CFFI_OP_FUNCTION, 449), // Cryptography_STACK_OF_POLICYINFO *()(void)
/* 148 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 149 */ _CFFI_OP(_CFFI_OP_FUNCTION, 455), // Cryptography_STACK_OF_POLICYQUALINFO *()(void)
/* 150 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 151 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1101), // Cryptography_STACK_OF_SCT *()(void)
/* 152 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 153 */ _CFFI_OP(_CFFI_OP_FUNCTION, 413), // Cryptography_STACK_OF_X509 *()(OCSP_BASICRESP const *)
/* 154 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 155 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 156 */ _CFFI_OP(_CFFI_OP_FUNCTION, 413), // Cryptography_STACK_OF_X509 *()(SSL const *)
/* 157 */ _CFFI_OP(_CFFI_OP_POINTER, 3111), // SSL const *
/* 158 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 159 */ _CFFI_OP(_CFFI_OP_FUNCTION, 413), // Cryptography_STACK_OF_X509 *()(X509_STORE_CTX *)
/* 160 */ _CFFI_OP(_CFFI_OP_POINTER, 3133), // X509_STORE_CTX *
/* 161 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 162 */ _CFFI_OP(_CFFI_OP_FUNCTION, 413), // Cryptography_STACK_OF_X509 *()(void)
/* 163 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 164 */ _CFFI_OP(_CFFI_OP_FUNCTION, 565), // Cryptography_STACK_OF_X509_CRL *()(void)
/* 165 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 166 */ _CFFI_OP(_CFFI_OP_FUNCTION, 633), // Cryptography_STACK_OF_X509_NAME *()(SSL const *)
/* 167 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 168 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 169 */ _CFFI_OP(_CFFI_OP_FUNCTION, 633), // Cryptography_STACK_OF_X509_NAME *()(char const *)
/* 170 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 171 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 172 */ _CFFI_OP(_CFFI_OP_FUNCTION, 633), // Cryptography_STACK_OF_X509_NAME *()(void)
/* 173 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 174 */ _CFFI_OP(_CFFI_OP_FUNCTION, 175), // Cryptography_STACK_OF_X509_NAME_ENTRY *()(Cryptography_STACK_OF_X509_NAME_ENTRY *)
/* 175 */ _CFFI_OP(_CFFI_OP_POINTER, 3046), // Cryptography_STACK_OF_X509_NAME_ENTRY *
/* 176 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 177 */ _CFFI_OP(_CFFI_OP_FUNCTION, 175), // Cryptography_STACK_OF_X509_NAME_ENTRY *()(void)
/* 178 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 179 */ _CFFI_OP(_CFFI_OP_FUNCTION, 666), // Cryptography_STACK_OF_X509_OBJECT *()(X509_STORE *)
/* 180 */ _CFFI_OP(_CFFI_OP_POINTER, 3132), // X509_STORE *
/* 181 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 182 */ _CFFI_OP(_CFFI_OP_FUNCTION, 682), // Cryptography_STACK_OF_X509_REVOKED *()(X509_CRL *)
/* 183 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 184 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 185 */ _CFFI_OP(_CFFI_OP_FUNCTION, 196), // DH *()(BIO *, DH * *)
/* 186 */ _CFFI_OP(_CFFI_OP_POINTER, 3022), // BIO *
/* 187 */ _CFFI_OP(_CFFI_OP_POINTER, 196), // DH * *
/* 188 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 189 */ _CFFI_OP(_CFFI_OP_FUNCTION, 196), // DH *()(BIO *, DH * *, int(*)(char *, int, int, void *), void *)
/* 190 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 191 */ _CFFI_OP(_CFFI_OP_NOOP, 187),
/* 192 */ _CFFI_OP(_CFFI_OP_POINTER, 2160), // int(*)(char *, int, int, void *)
/* 193 */ _CFFI_OP(_CFFI_OP_POINTER, 3142), // void *
/* 194 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 195 */ _CFFI_OP(_CFFI_OP_FUNCTION, 196), // DH *()(DH *)
/* 196 */ _CFFI_OP(_CFFI_OP_POINTER, 3049), // DH *
/* 197 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 198 */ _CFFI_OP(_CFFI_OP_FUNCTION, 196), // DH *()(EVP_PKEY *)
/* 199 */ _CFFI_OP(_CFFI_OP_POINTER, 3066), // EVP_PKEY *
/* 200 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 201 */ _CFFI_OP(_CFFI_OP_FUNCTION, 196), // DH *()(void)
/* 202 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 203 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1077), // DIST_POINT *()(Cryptography_STACK_OF_DIST_POINT *, int)
/* 204 */ _CFFI_OP(_CFFI_OP_POINTER, 3037), // Cryptography_STACK_OF_DIST_POINT *
/* 205 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 206 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 207 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1077), // DIST_POINT *()(void)
/* 208 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 209 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2644), // DIST_POINT_NAME *()(void)
/* 210 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 211 */ _CFFI_OP(_CFFI_OP_FUNCTION, 212), // DSA *()(DSA *)
/* 212 */ _CFFI_OP(_CFFI_OP_POINTER, 3052), // DSA *
/* 213 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 214 */ _CFFI_OP(_CFFI_OP_FUNCTION, 212), // DSA *()(EVP_PKEY *)
/* 215 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 216 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 217 */ _CFFI_OP(_CFFI_OP_FUNCTION, 212), // DSA *()(void)
/* 218 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 219 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2661), // EC_GROUP *()(int)
/* 220 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 221 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 222 */ _CFFI_OP(_CFFI_OP_FUNCTION, 238), // EC_GROUP const *()(EC_KEY const *)
/* 223 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 224 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 225 */ _CFFI_OP(_CFFI_OP_FUNCTION, 893), // EC_KEY *()(BIO *, EC_KEY * *)
/* 226 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 227 */ _CFFI_OP(_CFFI_OP_POINTER, 893), // EC_KEY * *
/* 228 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 229 */ _CFFI_OP(_CFFI_OP_FUNCTION, 893), // EC_KEY *()(EVP_PKEY *)
/* 230 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 231 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 232 */ _CFFI_OP(_CFFI_OP_FUNCTION, 893), // EC_KEY *()(int)
/* 233 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 234 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 235 */ _CFFI_OP(_CFFI_OP_FUNCTION, 893), // EC_KEY *()(void)
/* 236 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 237 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1298), // EC_METHOD const *()(EC_GROUP const *)
/* 238 */ _CFFI_OP(_CFFI_OP_POINTER, 3054), // EC_GROUP const *
/* 239 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 240 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1207), // EC_POINT *()(EC_GROUP const *)
/* 241 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 242 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 243 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1207), // EC_POINT *()(EC_POINT const *, EC_GROUP const *)
/* 244 */ _CFFI_OP(_CFFI_OP_POINTER, 3057), // EC_POINT const *
/* 245 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 246 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 247 */ _CFFI_OP(_CFFI_OP_FUNCTION, 244), // EC_POINT const *()(EC_GROUP const *)
/* 248 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 249 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 250 */ _CFFI_OP(_CFFI_OP_FUNCTION, 244), // EC_POINT const *()(EC_KEY const *)
/* 251 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 252 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 253 */ _CFFI_OP(_CFFI_OP_FUNCTION, 282), // ENGINE *()(char const *)
/* 254 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 255 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 256 */ _CFFI_OP(_CFFI_OP_FUNCTION, 282), // ENGINE *()(void)
/* 257 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 258 */ _CFFI_OP(_CFFI_OP_FUNCTION, 885), // EVP_CIPHER const *()(EVP_CIPHER_CTX const *)
/* 259 */ _CFFI_OP(_CFFI_OP_POINTER, 3063), // EVP_CIPHER_CTX const *
/* 260 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 261 */ _CFFI_OP(_CFFI_OP_FUNCTION, 885), // EVP_CIPHER const *()(char const *)
/* 262 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 263 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 264 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1318), // EVP_CIPHER_CTX *()(void)
/* 265 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 266 */ _CFFI_OP(_CFFI_OP_FUNCTION, 354), // EVP_MD const *()(char const *)
/* 267 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 268 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 269 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1351), // EVP_MD_CTX *()(void)
/* 270 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 271 */ _CFFI_OP(_CFFI_OP_FUNCTION, 199), // EVP_PKEY *()(BIO *, EVP_PKEY * *)
/* 272 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 273 */ _CFFI_OP(_CFFI_OP_POINTER, 199), // EVP_PKEY * *
/* 274 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 275 */ _CFFI_OP(_CFFI_OP_FUNCTION, 199), // EVP_PKEY *()(BIO *, EVP_PKEY * *, int(*)(char *, int, int, void *), void *)
/* 276 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 277 */ _CFFI_OP(_CFFI_OP_NOOP, 273),
/* 278 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 279 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 280 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 281 */ _CFFI_OP(_CFFI_OP_FUNCTION, 199), // EVP_PKEY *()(ENGINE *, char const *, UI_METHOD *, void *)
/* 282 */ _CFFI_OP(_CFFI_OP_POINTER, 3061), // ENGINE *
/* 283 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 284 */ _CFFI_OP(_CFFI_OP_POINTER, 3117), // UI_METHOD *
/* 285 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 286 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 287 */ _CFFI_OP(_CFFI_OP_FUNCTION, 199), // EVP_PKEY *()(NETSCAPE_SPKI *)
/* 288 */ _CFFI_OP(_CFFI_OP_POINTER, 3075), // NETSCAPE_SPKI *
/* 289 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 290 */ _CFFI_OP(_CFFI_OP_FUNCTION, 199), // EVP_PKEY *()(X509 *)
/* 291 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 292 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 293 */ _CFFI_OP(_CFFI_OP_FUNCTION, 199), // EVP_PKEY *()(X509_REQ *)
/* 294 */ _CFFI_OP(_CFFI_OP_POINTER, 3130), // X509_REQ *
/* 295 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 296 */ _CFFI_OP(_CFFI_OP_FUNCTION, 199), // EVP_PKEY *()(int, ENGINE *, unsigned char const *, size_t)
/* 297 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 298 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 299 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 300 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28), // size_t
/* 301 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 302 */ _CFFI_OP(_CFFI_OP_FUNCTION, 199), // EVP_PKEY *()(void)
/* 303 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 304 */ _CFFI_OP(_CFFI_OP_FUNCTION, 309), // EVP_PKEY_CTX *()(EVP_PKEY *, ENGINE *)
/* 305 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 306 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 307 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 308 */ _CFFI_OP(_CFFI_OP_FUNCTION, 309), // EVP_PKEY_CTX *()(EVP_PKEY_CTX *)
/* 309 */ _CFFI_OP(_CFFI_OP_POINTER, 3067), // EVP_PKEY_CTX *
/* 310 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 311 */ _CFFI_OP(_CFFI_OP_FUNCTION, 309), // EVP_PKEY_CTX *()(int, ENGINE *)
/* 312 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 313 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 314 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 315 */ _CFFI_OP(_CFFI_OP_FUNCTION, 928), // GENERAL_NAME *()(GENERAL_NAMES *, int)
/* 316 */ _CFFI_OP(_CFFI_OP_POINTER, 3070), // GENERAL_NAMES *
/* 317 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 318 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 319 */ _CFFI_OP(_CFFI_OP_FUNCTION, 928), // GENERAL_NAME *()(void)
/* 320 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 321 */ _CFFI_OP(_CFFI_OP_FUNCTION, 316), // GENERAL_NAMES *()(GENERAL_NAMES * *, unsigned char const * *, long)
/* 322 */ _CFFI_OP(_CFFI_OP_POINTER, 316), // GENERAL_NAMES * *
/* 323 */ _CFFI_OP(_CFFI_OP_NOOP, 75),
/* 324 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 325 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 326 */ _CFFI_OP(_CFFI_OP_FUNCTION, 316), // GENERAL_NAMES *()(void)
/* 327 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 328 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1084), // GENERAL_SUBTREE *()(Cryptography_STACK_OF_GENERAL_SUBTREE *, int)
/* 329 */ _CFFI_OP(_CFFI_OP_POINTER, 3038), // Cryptography_STACK_OF_GENERAL_SUBTREE *
/* 330 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 331 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 332 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1084), // GENERAL_SUBTREE *()(void)
/* 333 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 334 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1515), // HMAC_CTX *()(void)
/* 335 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 336 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2702), // ISSUING_DIST_POINT *()(void)
/* 337 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 338 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2705), // NAME_CONSTRAINTS *()(void)
/* 339 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 340 */ _CFFI_OP(_CFFI_OP_FUNCTION, 288), // NETSCAPE_SPKI *()(char const *, int)
/* 341 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 342 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 343 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 344 */ _CFFI_OP(_CFFI_OP_FUNCTION, 288), // NETSCAPE_SPKI *()(void)
/* 345 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 346 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2711), // NOTICEREF *()(void)
/* 347 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 348 */ _CFFI_OP(_CFFI_OP_FUNCTION, 387), // OCSP_BASICRESP *()(OCSP_RESPONSE *)
/* 349 */ _CFFI_OP(_CFFI_OP_POINTER, 3085), // OCSP_RESPONSE *
/* 350 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 351 */ _CFFI_OP(_CFFI_OP_FUNCTION, 387), // OCSP_BASICRESP *()(void)
/* 352 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 353 */ _CFFI_OP(_CFFI_OP_FUNCTION, 366), // OCSP_CERTID *()(EVP_MD const *, X509 const *, X509 const *)
/* 354 */ _CFFI_OP(_CFFI_OP_POINTER, 3064), // EVP_MD const *
/* 355 */ _CFFI_OP(_CFFI_OP_POINTER, 3119), // X509 const *
/* 356 */ _CFFI_OP(_CFFI_OP_NOOP, 355),
/* 357 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 358 */ _CFFI_OP(_CFFI_OP_FUNCTION, 366), // OCSP_CERTID *()(OCSP_ONEREQ *)
/* 359 */ _CFFI_OP(_CFFI_OP_POINTER, 3081), // OCSP_ONEREQ *
/* 360 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 361 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3079), // OCSP_CERTID const *()(OCSP_SINGLERESP const *)
/* 362 */ _CFFI_OP(_CFFI_OP_POINTER, 3086), // OCSP_SINGLERESP const *
/* 363 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 364 */ _CFFI_OP(_CFFI_OP_FUNCTION, 359), // OCSP_ONEREQ *()(OCSP_REQUEST *, OCSP_CERTID *)
/* 365 */ _CFFI_OP(_CFFI_OP_POINTER, 3082), // OCSP_REQUEST *
/* 366 */ _CFFI_OP(_CFFI_OP_POINTER, 3080), // OCSP_CERTID *
/* 367 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 368 */ _CFFI_OP(_CFFI_OP_FUNCTION, 359), // OCSP_ONEREQ *()(OCSP_REQUEST *, int)
/* 369 */ _CFFI_OP(_CFFI_OP_NOOP, 365),
/* 370 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 371 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 372 */ _CFFI_OP(_CFFI_OP_FUNCTION, 365), // OCSP_REQUEST *()(BIO *, OCSP_REQUEST * *)
/* 373 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 374 */ _CFFI_OP(_CFFI_OP_POINTER, 365), // OCSP_REQUEST * *
/* 375 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 376 */ _CFFI_OP(_CFFI_OP_FUNCTION, 365), // OCSP_REQUEST *()(void)
/* 377 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 378 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3083), // OCSP_RESPDATA const *()(OCSP_BASICRESP const *)
/* 379 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 380 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 381 */ _CFFI_OP(_CFFI_OP_FUNCTION, 349), // OCSP_RESPONSE *()(BIO *, OCSP_RESPONSE * *)
/* 382 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 383 */ _CFFI_OP(_CFFI_OP_POINTER, 349), // OCSP_RESPONSE * *
/* 384 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 385 */ _CFFI_OP(_CFFI_OP_FUNCTION, 349), // OCSP_RESPONSE *()(int, OCSP_BASICRESP *)
/* 386 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 387 */ _CFFI_OP(_CFFI_OP_POINTER, 3078), // OCSP_BASICRESP *
/* 388 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 389 */ _CFFI_OP(_CFFI_OP_FUNCTION, 594), // OCSP_SINGLERESP *()(OCSP_BASICRESP *, OCSP_CERTID *, int, int, ASN1_OCTET_STRING *, ASN1_OCTET_STRING *, ASN1_OCTET_STRING *)
/* 390 */ _CFFI_OP(_CFFI_OP_NOOP, 387),
/* 391 */ _CFFI_OP(_CFFI_OP_NOOP, 366),
/* 392 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 393 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 394 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 395 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 396 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 397 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 398 */ _CFFI_OP(_CFFI_OP_FUNCTION, 594), // OCSP_SINGLERESP *()(OCSP_BASICRESP *, int)
/* 399 */ _CFFI_OP(_CFFI_OP_NOOP, 387),
/* 400 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 401 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 402 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2726), // OTHERNAME *()(void)
/* 403 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 404 */ _CFFI_OP(_CFFI_OP_FUNCTION, 940), // PKCS12 *()(BIO *, PKCS12 * *)
/* 405 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 406 */ _CFFI_OP(_CFFI_OP_POINTER, 940), // PKCS12 * *
/* 407 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 408 */ _CFFI_OP(_CFFI_OP_FUNCTION, 940), // PKCS12 *()(char *, char *, EVP_PKEY *, X509 *, Cryptography_STACK_OF_X509 *, int, int, int, int, int)
/* 409 */ _CFFI_OP(_CFFI_OP_POINTER, 3135), // char *
/* 410 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 411 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 412 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 413 */ _CFFI_OP(_CFFI_OP_POINTER, 3043), // Cryptography_STACK_OF_X509 *
/* 414 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 415 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 416 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 417 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 418 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 419 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 420 */ _CFFI_OP(_CFFI_OP_FUNCTION, 442), // PKCS7 *()(BIO *, BIO * *)
/* 421 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 422 */ _CFFI_OP(_CFFI_OP_POINTER, 186), // BIO * *
/* 423 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 424 */ _CFFI_OP(_CFFI_OP_FUNCTION, 442), // PKCS7 *()(BIO *, PKCS7 * *)
/* 425 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 426 */ _CFFI_OP(_CFFI_OP_POINTER, 442), // PKCS7 * *
/* 427 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 428 */ _CFFI_OP(_CFFI_OP_FUNCTION, 442), // PKCS7 *()(BIO *, PKCS7 * *, int(*)(char *, int, int, void *), void *)
/* 429 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 430 */ _CFFI_OP(_CFFI_OP_NOOP, 426),
/* 431 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 432 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 433 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 434 */ _CFFI_OP(_CFFI_OP_FUNCTION, 442), // PKCS7 *()(X509 *, EVP_PKEY *, Cryptography_STACK_OF_X509 *, BIO *, int)
/* 435 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 436 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 437 */ _CFFI_OP(_CFFI_OP_NOOP, 413),
/* 438 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 439 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 440 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 441 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3098), // PKCS7_SIGNER_INFO *()(PKCS7 *, X509 *, EVP_PKEY *, EVP_MD const *, int)
/* 442 */ _CFFI_OP(_CFFI_OP_POINTER, 3089), // PKCS7 *
/* 443 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 444 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 445 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 446 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 447 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 448 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1091), // POLICYINFO *()(Cryptography_STACK_OF_POLICYINFO *, int)
/* 449 */ _CFFI_OP(_CFFI_OP_POINTER, 3039), // Cryptography_STACK_OF_POLICYINFO *
/* 450 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 451 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 452 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1091), // POLICYINFO *()(void)
/* 453 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 454 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1098), // POLICYQUALINFO *()(Cryptography_STACK_OF_POLICYQUALINFO *, int)
/* 455 */ _CFFI_OP(_CFFI_OP_POINTER, 3040), // Cryptography_STACK_OF_POLICYQUALINFO *
/* 456 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 457 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 458 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1098), // POLICYQUALINFO *()(void)
/* 459 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 460 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2741), // POLICY_CONSTRAINTS *()(void)
/* 461 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 462 */ _CFFI_OP(_CFFI_OP_FUNCTION, 476), // RSA *()(BIO *, RSA * *)
/* 463 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 464 */ _CFFI_OP(_CFFI_OP_POINTER, 476), // RSA * *
/* 465 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 466 */ _CFFI_OP(_CFFI_OP_FUNCTION, 476), // RSA *()(BIO *, RSA * *, int(*)(char *, int, int, void *), void *)
/* 467 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 468 */ _CFFI_OP(_CFFI_OP_NOOP, 464),
/* 469 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 470 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 471 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 472 */ _CFFI_OP(_CFFI_OP_FUNCTION, 476), // RSA *()(EVP_PKEY *)
/* 473 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 474 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 475 */ _CFFI_OP(_CFFI_OP_FUNCTION, 476), // RSA *()(RSA *)
/* 476 */ _CFFI_OP(_CFFI_OP_POINTER, 3107), // RSA *
/* 477 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 478 */ _CFFI_OP(_CFFI_OP_FUNCTION, 476), // RSA *()(void)
/* 479 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 480 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1102), // SCT *()(Cryptography_STACK_OF_SCT const *, int)
/* 481 */ _CFFI_OP(_CFFI_OP_POINTER, 3041), // Cryptography_STACK_OF_SCT const *
/* 482 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 483 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 484 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1102), // SCT *()(void)
/* 485 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 486 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3109), // SRTP_PROTECTION_PROFILE *()(SSL *)
/* 487 */ _CFFI_OP(_CFFI_OP_POINTER, 3111), // SSL *
/* 488 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 489 */ _CFFI_OP(_CFFI_OP_FUNCTION, 487), // SSL *()(SSL_CTX *)
/* 490 */ _CFFI_OP(_CFFI_OP_POINTER, 3113), // SSL_CTX *
/* 491 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 492 */ _CFFI_OP(_CFFI_OP_FUNCTION, 733), // SSL_CIPHER const *()(Cryptography_STACK_OF_SSL_CIPHER *, int)
/* 493 */ _CFFI_OP(_CFFI_OP_POINTER, 3042), // Cryptography_STACK_OF_SSL_CIPHER *
/* 494 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 495 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 496 */ _CFFI_OP(_CFFI_OP_FUNCTION, 733), // SSL_CIPHER const *()(SSL const *)
/* 497 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 498 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 499 */ _CFFI_OP(_CFFI_OP_FUNCTION, 490), // SSL_CTX *()(SSL *, SSL_CTX *)
/* 500 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 501 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 502 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 503 */ _CFFI_OP(_CFFI_OP_FUNCTION, 490), // SSL_CTX *()(SSL_METHOD *)
/* 504 */ _CFFI_OP(_CFFI_OP_POINTER, 3115), // SSL_METHOD *
/* 505 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 506 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3114), // SSL_METHOD const *()(void)
/* 507 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 508 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1666), // SSL_SESSION *()(SSL *)
/* 509 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 510 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 511 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1666), // SSL_SESSION *()(SSL const *)
/* 512 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 513 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 514 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2873), // USERNOTICE *()(void)
/* 515 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 516 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(BIO *, X509 * *)
/* 517 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 518 */ _CFFI_OP(_CFFI_OP_POINTER, 28), // X509 * *
/* 519 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 520 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(BIO *, X509 * *, int(*)(char *, int, int, void *), void *)
/* 521 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 522 */ _CFFI_OP(_CFFI_OP_NOOP, 518),
/* 523 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 524 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 525 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 526 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(Cryptography_STACK_OF_X509 *, int)
/* 527 */ _CFFI_OP(_CFFI_OP_NOOP, 413),
/* 528 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 529 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 530 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(SSL const *)
/* 531 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 532 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 533 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(X509 *)
/* 534 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 535 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 536 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(X509_OBJECT *)
/* 537 */ _CFFI_OP(_CFFI_OP_POINTER, 3129), // X509_OBJECT *
/* 538 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 539 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(X509_STORE_CTX *)
/* 540 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 541 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 542 */ _CFFI_OP(_CFFI_OP_FUNCTION, 28), // X509 *()(void)
/* 543 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 544 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3121), // X509_ALGOR const *()(OCSP_BASICRESP const *)
/* 545 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 546 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 547 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3121), // X509_ALGOR const *()(X509 const *)
/* 548 */ _CFFI_OP(_CFFI_OP_NOOP, 355),
/* 549 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 550 */ _CFFI_OP(_CFFI_OP_FUNCTION, 79), // X509_ATTRIBUTE *()(X509_REQ const *, int)
/* 551 */ _CFFI_OP(_CFFI_OP_POINTER, 3130), // X509_REQ const *
/* 552 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 553 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 554 */ _CFFI_OP(_CFFI_OP_FUNCTION, 57), // X509_CRL *()(BIO *, X509_CRL * *)
/* 555 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 556 */ _CFFI_OP(_CFFI_OP_POINTER, 57), // X509_CRL * *
/* 557 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 558 */ _CFFI_OP(_CFFI_OP_FUNCTION, 57), // X509_CRL *()(BIO *, X509_CRL * *, int(*)(char *, int, int, void *), void *)
/* 559 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 560 */ _CFFI_OP(_CFFI_OP_NOOP, 556),
/* 561 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 562 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 563 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 564 */ _CFFI_OP(_CFFI_OP_FUNCTION, 57), // X509_CRL *()(Cryptography_STACK_OF_X509_CRL *, int)
/* 565 */ _CFFI_OP(_CFFI_OP_POINTER, 3044), // Cryptography_STACK_OF_X509_CRL *
/* 566 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 567 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 568 */ _CFFI_OP(_CFFI_OP_FUNCTION, 57), // X509_CRL *()(X509_CRL *)
/* 569 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 570 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 571 */ _CFFI_OP(_CFFI_OP_FUNCTION, 57), // X509_CRL *()(void)
/* 572 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 573 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(CONF *, X509V3_CTX *, char *, char *)
/* 574 */ _CFFI_OP(_CFFI_OP_POINTER, 3028), // CONF *
/* 575 */ _CFFI_OP(_CFFI_OP_POINTER, 3120), // X509V3_CTX *
/* 576 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 577 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 578 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 579 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(Cryptography_LHASH_OF_CONF_VALUE *, X509V3_CTX *, int, char *)
/* 580 */ _CFFI_OP(_CFFI_OP_POINTER, 3033), // Cryptography_LHASH_OF_CONF_VALUE *
/* 581 */ _CFFI_OP(_CFFI_OP_NOOP, 575),
/* 582 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 583 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 584 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 585 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(OCSP_BASICRESP *, int)
/* 586 */ _CFFI_OP(_CFFI_OP_NOOP, 387),
/* 587 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 588 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 589 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(OCSP_REQUEST *, int)
/* 590 */ _CFFI_OP(_CFFI_OP_NOOP, 365),
/* 591 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 592 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 593 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(OCSP_SINGLERESP *, int)
/* 594 */ _CFFI_OP(_CFFI_OP_POINTER, 3086), // OCSP_SINGLERESP *
/* 595 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 596 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 597 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(X509 *, int)
/* 598 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 599 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 600 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 601 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(X509_CRL *, int)
/* 602 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 603 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 604 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 605 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(X509_EXTENSION * *, ASN1_OBJECT *, int, ASN1_OCTET_STRING *)
/* 606 */ _CFFI_OP(_CFFI_OP_POINTER, 40), // X509_EXTENSION * *
/* 607 */ _CFFI_OP(_CFFI_OP_POINTER, 3015), // ASN1_OBJECT *
/* 608 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 609 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 610 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 611 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(X509_EXTENSION *)
/* 612 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 613 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 614 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(X509_EXTENSIONS *, int)
/* 615 */ _CFFI_OP(_CFFI_OP_POINTER, 3126), // X509_EXTENSIONS *
/* 616 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 617 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 618 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(X509_REVOKED *, int)
/* 619 */ _CFFI_OP(_CFFI_OP_POINTER, 3131), // X509_REVOKED *
/* 620 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 621 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 622 */ _CFFI_OP(_CFFI_OP_FUNCTION, 40), // X509_EXTENSION *()(int, int, void *)
/* 623 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 624 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 625 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 626 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 627 */ _CFFI_OP(_CFFI_OP_FUNCTION, 615), // X509_EXTENSIONS *()(X509_REQ *)
/* 628 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 629 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 630 */ _CFFI_OP(_CFFI_OP_FUNCTION, 615), // X509_EXTENSIONS *()(void)
/* 631 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 632 */ _CFFI_OP(_CFFI_OP_FUNCTION, 643), // X509_NAME *()(Cryptography_STACK_OF_X509_NAME *, int)
/* 633 */ _CFFI_OP(_CFFI_OP_POINTER, 3045), // Cryptography_STACK_OF_X509_NAME *
/* 634 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 635 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 636 */ _CFFI_OP(_CFFI_OP_FUNCTION, 643), // X509_NAME *()(X509 *)
/* 637 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 638 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 639 */ _CFFI_OP(_CFFI_OP_FUNCTION, 643), // X509_NAME *()(X509_CRL *)
/* 640 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 641 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 642 */ _CFFI_OP(_CFFI_OP_FUNCTION, 643), // X509_NAME *()(X509_NAME *)
/* 643 */ _CFFI_OP(_CFFI_OP_POINTER, 3127), // X509_NAME *
/* 644 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 645 */ _CFFI_OP(_CFFI_OP_FUNCTION, 643), // X509_NAME *()(X509_REQ *)
/* 646 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 647 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 648 */ _CFFI_OP(_CFFI_OP_FUNCTION, 643), // X509_NAME *()(void)
/* 649 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 650 */ _CFFI_OP(_CFFI_OP_FUNCTION, 43), // X509_NAME_ENTRY *()(Cryptography_STACK_OF_X509_NAME_ENTRY *, int)
/* 651 */ _CFFI_OP(_CFFI_OP_NOOP, 175),
/* 652 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 653 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 654 */ _CFFI_OP(_CFFI_OP_FUNCTION, 43), // X509_NAME_ENTRY *()(X509_NAME *, int)
/* 655 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 656 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 657 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 658 */ _CFFI_OP(_CFFI_OP_FUNCTION, 43), // X509_NAME_ENTRY *()(X509_NAME_ENTRY * *, ASN1_OBJECT *, int, unsigned char const *, int)
/* 659 */ _CFFI_OP(_CFFI_OP_POINTER, 43), // X509_NAME_ENTRY * *
/* 660 */ _CFFI_OP(_CFFI_OP_NOOP, 607),
/* 661 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 662 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 663 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 664 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 665 */ _CFFI_OP(_CFFI_OP_FUNCTION, 537), // X509_OBJECT *()(Cryptography_STACK_OF_X509_OBJECT *, int)
/* 666 */ _CFFI_OP(_CFFI_OP_POINTER, 3047), // Cryptography_STACK_OF_X509_OBJECT *
/* 667 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 668 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 669 */ _CFFI_OP(_CFFI_OP_FUNCTION, 294), // X509_REQ *()(BIO *, X509_REQ * *)
/* 670 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 671 */ _CFFI_OP(_CFFI_OP_POINTER, 294), // X509_REQ * *
/* 672 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 673 */ _CFFI_OP(_CFFI_OP_FUNCTION, 294), // X509_REQ *()(BIO *, X509_REQ * *, int(*)(char *, int, int, void *), void *)
/* 674 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 675 */ _CFFI_OP(_CFFI_OP_NOOP, 671),
/* 676 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 677 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 678 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 679 */ _CFFI_OP(_CFFI_OP_FUNCTION, 294), // X509_REQ *()(void)
/* 680 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 681 */ _CFFI_OP(_CFFI_OP_FUNCTION, 619), // X509_REVOKED *()(Cryptography_STACK_OF_X509_REVOKED *, int)
/* 682 */ _CFFI_OP(_CFFI_OP_POINTER, 3048), // Cryptography_STACK_OF_X509_REVOKED *
/* 683 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 684 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 685 */ _CFFI_OP(_CFFI_OP_FUNCTION, 619), // X509_REVOKED *()(X509_REVOKED *)
/* 686 */ _CFFI_OP(_CFFI_OP_NOOP, 619),
/* 687 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 688 */ _CFFI_OP(_CFFI_OP_FUNCTION, 619), // X509_REVOKED *()(void)
/* 689 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 690 */ _CFFI_OP(_CFFI_OP_FUNCTION, 180), // X509_STORE *()(SSL_CTX const *)
/* 691 */ _CFFI_OP(_CFFI_OP_POINTER, 3113), // SSL_CTX const *
/* 692 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 693 */ _CFFI_OP(_CFFI_OP_FUNCTION, 180), // X509_STORE *()(void)
/* 694 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 695 */ _CFFI_OP(_CFFI_OP_FUNCTION, 160), // X509_STORE_CTX *()(void)
/* 696 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 697 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2085), // X509_VERIFY_PARAM *()(SSL *)
/* 698 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 699 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 700 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2085), // X509_VERIFY_PARAM *()(SSL_CTX *)
/* 701 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 702 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 703 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2085), // X509_VERIFY_PARAM *()(X509_STORE *)
/* 704 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 705 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 706 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2085), // X509_VERIFY_PARAM *()(X509_STORE_CTX *)
/* 707 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 708 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 709 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2085), // X509_VERIFY_PARAM *()(void)
/* 710 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 711 */ _CFFI_OP(_CFFI_OP_FUNCTION, 409), // char *()(BIGNUM const *)
/* 712 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 713 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 714 */ _CFFI_OP(_CFFI_OP_FUNCTION, 409), // char *()(NETSCAPE_SPKI *)
/* 715 */ _CFFI_OP(_CFFI_OP_NOOP, 288),
/* 716 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 717 */ _CFFI_OP(_CFFI_OP_FUNCTION, 409), // char *()(X509_NAME *, char *, int)
/* 718 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 719 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 720 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 721 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 722 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // char const *()(ENGINE const *)
/* 723 */ _CFFI_OP(_CFFI_OP_POINTER, 3061), // ENGINE const *
/* 724 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 725 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // char const *()(SSL const *)
/* 726 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 727 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 728 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // char const *()(SSL const *, int)
/* 729 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 730 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 731 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 732 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // char const *()(SSL_CIPHER const *)
/* 733 */ _CFFI_OP(_CFFI_OP_POINTER, 3112), // SSL_CIPHER const *
/* 734 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 735 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // char const *()(int)
/* 736 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 737 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 738 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // char const *()(long)
/* 739 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 740 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 741 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // char const *()(unsigned long)
/* 742 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10), // unsigned long
/* 743 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 744 */ _CFFI_OP(_CFFI_OP_FUNCTION, 46), // char const *()(void)
/* 745 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 746 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1646), // ct_log_entry_type_t()(SCT const *)
/* 747 */ _CFFI_OP(_CFFI_OP_POINTER, 3108), // SCT const *
/* 748 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 749 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(AES_KEY *, unsigned char const *, unsigned char *, unsigned char const *, unsigned int)
/* 750 */ _CFFI_OP(_CFFI_OP_POINTER, 3007), // AES_KEY *
/* 751 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 752 */ _CFFI_OP(_CFFI_OP_POINTER, 3140), // unsigned char *
/* 753 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 754 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8), // unsigned int
/* 755 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 756 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_ENUMERATED *, long)
/* 757 */ _CFFI_OP(_CFFI_OP_POINTER, 3008), // ASN1_ENUMERATED *
/* 758 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 759 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 760 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_INTEGER *, long)
/* 761 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 762 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 763 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 764 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OBJECT const *)
/* 765 */ _CFFI_OP(_CFFI_OP_POINTER, 3015), // ASN1_OBJECT const *
/* 766 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 767 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING * *, ASN1_OBJECT * *, ASN1_OCTET_STRING * *, ASN1_INTEGER * *, OCSP_CERTID *)
/* 768 */ _CFFI_OP(_CFFI_OP_POINTER, 13), // ASN1_OCTET_STRING * *
/* 769 */ _CFFI_OP(_CFFI_OP_POINTER, 607), // ASN1_OBJECT * *
/* 770 */ _CFFI_OP(_CFFI_OP_NOOP, 768),
/* 771 */ _CFFI_OP(_CFFI_OP_POINTER, 21), // ASN1_INTEGER * *
/* 772 */ _CFFI_OP(_CFFI_OP_NOOP, 366),
/* 773 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 774 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *)
/* 775 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 776 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 777 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *, char const *)
/* 778 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 779 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 780 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 781 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *, int)
/* 782 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 783 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 784 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 785 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *, int, int)
/* 786 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 787 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 788 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 789 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 790 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *, unsigned char const *, int)
/* 791 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 792 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 793 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 794 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 795 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_OCTET_STRING *, void const *, int)
/* 796 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 797 */ _CFFI_OP(_CFFI_OP_NOOP, 126),
/* 798 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 799 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 800 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ASN1_TYPE *, unsigned char * *)
/* 801 */ _CFFI_OP(_CFFI_OP_POINTER, 3018), // ASN1_TYPE *
/* 802 */ _CFFI_OP(_CFFI_OP_POINTER, 752), // unsigned char * *
/* 803 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 804 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM * *, char const *)
/* 805 */ _CFFI_OP(_CFFI_OP_POINTER, 20), // BIGNUM * *
/* 806 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 807 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 808 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BIGNUM const *)
/* 809 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 810 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 811 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 812 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *)
/* 813 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 814 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 815 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 816 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 817 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 818 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 819 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 820 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 821 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 822 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 823 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 824 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *, BIGNUM const *, BN_CTX *, BN_MONT_CTX *)
/* 825 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 826 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 827 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 828 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 829 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 830 */ _CFFI_OP(_CFFI_OP_POINTER, 3026), // BN_MONT_CTX *
/* 831 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 832 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 833 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 834 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 835 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 836 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 837 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 838 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, BN_ULONG)
/* 839 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 840 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, _cffi_prim_int(sizeof(BN_ULONG), (
           ((BN_ULONG)-1) | 0 /* check that BN_ULONG is an integer type */
         ) <= 0)), // BN_ULONG
/* 841 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 842 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM *, int, int, BIGNUM const *, BIGNUM const *, BN_GENCB *)
/* 843 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 844 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 845 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 846 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 847 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 848 */ _CFFI_OP(_CFFI_OP_POINTER, 3025), // BN_GENCB *
/* 849 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 850 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM const *)
/* 851 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 852 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 853 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM const *, BIGNUM const *)
/* 854 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 855 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 856 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 857 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM const *, int, BN_CTX *, BN_GENCB *)
/* 858 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 859 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 860 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 861 */ _CFFI_OP(_CFFI_OP_NOOP, 848),
/* 862 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 863 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIGNUM const *, unsigned char *)
/* 864 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 865 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 866 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 867 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *)
/* 868 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 869 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 870 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, ASN1_INTEGER *)
/* 871 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 872 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 873 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 874 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, DH *)
/* 875 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 876 */ _CFFI_OP(_CFFI_OP_NOOP, 196),
/* 877 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 878 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, DSA *)
/* 879 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 880 */ _CFFI_OP(_CFFI_OP_NOOP, 212),
/* 881 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 882 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, DSA *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 883 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 884 */ _CFFI_OP(_CFFI_OP_NOOP, 212),
/* 885 */ _CFFI_OP(_CFFI_OP_POINTER, 3062), // EVP_CIPHER const *
/* 886 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 887 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 888 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 889 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 890 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 891 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EC_KEY *)
/* 892 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 893 */ _CFFI_OP(_CFFI_OP_POINTER, 3055), // EC_KEY *
/* 894 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 895 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EC_KEY *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 896 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 897 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 898 */ _CFFI_OP(_CFFI_OP_NOOP, 885),
/* 899 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 900 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 901 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 902 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 903 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 904 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EVP_PKEY *)
/* 905 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 906 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 907 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 908 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EVP_PKEY *, EVP_CIPHER const *, char *, int, int(*)(char *, int, int, void *), void *)
/* 909 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 910 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 911 */ _CFFI_OP(_CFFI_OP_NOOP, 885),
/* 912 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 913 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 914 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 915 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 916 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 917 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, EVP_PKEY *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 918 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 919 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 920 */ _CFFI_OP(_CFFI_OP_NOOP, 885),
/* 921 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 922 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 923 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 924 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 925 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 926 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, GENERAL_NAME *)
/* 927 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 928 */ _CFFI_OP(_CFFI_OP_POINTER, 3069), // GENERAL_NAME *
/* 929 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 930 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, OCSP_REQUEST *)
/* 931 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 932 */ _CFFI_OP(_CFFI_OP_NOOP, 365),
/* 933 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 934 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, OCSP_RESPONSE *)
/* 935 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 936 */ _CFFI_OP(_CFFI_OP_NOOP, 349),
/* 937 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 938 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, PKCS12 *)
/* 939 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 940 */ _CFFI_OP(_CFFI_OP_POINTER, 3088), // PKCS12 *
/* 941 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 942 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, PKCS7 *)
/* 943 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 944 */ _CFFI_OP(_CFFI_OP_NOOP, 442),
/* 945 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 946 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, PKCS7 *, BIO *, int)
/* 947 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 948 */ _CFFI_OP(_CFFI_OP_NOOP, 442),
/* 949 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 950 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 951 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 952 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, RSA *)
/* 953 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 954 */ _CFFI_OP(_CFFI_OP_NOOP, 476),
/* 955 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 956 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, RSA *, EVP_CIPHER const *, unsigned char *, int, int(*)(char *, int, int, void *), void *)
/* 957 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 958 */ _CFFI_OP(_CFFI_OP_NOOP, 476),
/* 959 */ _CFFI_OP(_CFFI_OP_NOOP, 885),
/* 960 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 961 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 962 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 963 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 964 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 965 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, RSA const *)
/* 966 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 967 */ _CFFI_OP(_CFFI_OP_POINTER, 3107), // RSA const *
/* 968 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 969 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, RSA const *, int)
/* 970 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 971 */ _CFFI_OP(_CFFI_OP_NOOP, 967),
/* 972 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 973 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 974 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509 *)
/* 975 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 976 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 977 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 978 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509 *, unsigned long, unsigned long)
/* 979 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 980 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 981 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 982 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 983 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 984 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509_CRL *)
/* 985 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 986 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 987 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 988 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509_EXTENSION *, unsigned long, int)
/* 989 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 990 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 991 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 992 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 993 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 994 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509_NAME *, int, unsigned long)
/* 995 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 996 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 997 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 998 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 999 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1000 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509_REQ *)
/* 1001 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1002 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 1003 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1004 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, X509_REQ *, unsigned long, unsigned long)
/* 1005 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1006 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 1007 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1008 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1009 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1010 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, char *, int)
/* 1011 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1012 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 1013 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1014 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1015 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, void *, int)
/* 1016 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1017 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1018 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1019 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1020 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BIO *, void const *, int)
/* 1021 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1022 */ _CFFI_OP(_CFFI_OP_NOOP, 126),
/* 1023 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1024 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1025 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(BN_MONT_CTX *, BIGNUM const *, BN_CTX *)
/* 1026 */ _CFFI_OP(_CFFI_OP_NOOP, 830),
/* 1027 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 1028 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1029 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1030 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(CMAC_CTX *, CMAC_CTX const *)
/* 1031 */ _CFFI_OP(_CFFI_OP_POINTER, 3027), // CMAC_CTX *
/* 1032 */ _CFFI_OP(_CFFI_OP_POINTER, 3027), // CMAC_CTX const *
/* 1033 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1034 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(CMAC_CTX *, unsigned char *, size_t *)
/* 1035 */ _CFFI_OP(_CFFI_OP_NOOP, 1031),
/* 1036 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1037 */ _CFFI_OP(_CFFI_OP_POINTER, 300), // size_t *
/* 1038 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1039 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(CMAC_CTX *, void const *, size_t)
/* 1040 */ _CFFI_OP(_CFFI_OP_NOOP, 1031),
/* 1041 */ _CFFI_OP(_CFFI_OP_NOOP, 126),
/* 1042 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1043 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1044 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(CMAC_CTX *, void const *, size_t, EVP_CIPHER const *, ENGINE *)
/* 1045 */ _CFFI_OP(_CFFI_OP_NOOP, 1031),
/* 1046 */ _CFFI_OP(_CFFI_OP_NOOP, 126),
/* 1047 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1048 */ _CFFI_OP(_CFFI_OP_NOOP, 885),
/* 1049 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 1050 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1051 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ACCESS_DESCRIPTION *)
/* 1052 */ _CFFI_OP(_CFFI_OP_NOOP, 1),
/* 1053 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1054 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ACCESS_DESCRIPTION *, ACCESS_DESCRIPTION *)
/* 1055 */ _CFFI_OP(_CFFI_OP_NOOP, 1),
/* 1056 */ _CFFI_OP(_CFFI_OP_POINTER, 3006), // ACCESS_DESCRIPTION *
/* 1057 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1058 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ASN1_INTEGER *)
/* 1059 */ _CFFI_OP(_CFFI_OP_NOOP, 24),
/* 1060 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1061 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ASN1_INTEGER *, ASN1_INTEGER *)
/* 1062 */ _CFFI_OP(_CFFI_OP_NOOP, 24),
/* 1063 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 1064 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1065 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ASN1_OBJECT *)
/* 1066 */ _CFFI_OP(_CFFI_OP_NOOP, 36),
/* 1067 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1068 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_ASN1_OBJECT *, ASN1_OBJECT *)
/* 1069 */ _CFFI_OP(_CFFI_OP_NOOP, 36),
/* 1070 */ _CFFI_OP(_CFFI_OP_NOOP, 607),
/* 1071 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1072 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_DIST_POINT *)
/* 1073 */ _CFFI_OP(_CFFI_OP_NOOP, 204),
/* 1074 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1075 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_DIST_POINT *, DIST_POINT *)
/* 1076 */ _CFFI_OP(_CFFI_OP_NOOP, 204),
/* 1077 */ _CFFI_OP(_CFFI_OP_POINTER, 3050), // DIST_POINT *
/* 1078 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1079 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_GENERAL_SUBTREE *)
/* 1080 */ _CFFI_OP(_CFFI_OP_NOOP, 329),
/* 1081 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1082 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_GENERAL_SUBTREE *, GENERAL_SUBTREE *)
/* 1083 */ _CFFI_OP(_CFFI_OP_NOOP, 329),
/* 1084 */ _CFFI_OP(_CFFI_OP_POINTER, 3071), // GENERAL_SUBTREE *
/* 1085 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1086 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_POLICYINFO *)
/* 1087 */ _CFFI_OP(_CFFI_OP_NOOP, 449),
/* 1088 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1089 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_POLICYINFO *, POLICYINFO *)
/* 1090 */ _CFFI_OP(_CFFI_OP_NOOP, 449),
/* 1091 */ _CFFI_OP(_CFFI_OP_POINTER, 3103), // POLICYINFO *
/* 1092 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1093 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_POLICYQUALINFO *)
/* 1094 */ _CFFI_OP(_CFFI_OP_NOOP, 455),
/* 1095 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1096 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_POLICYQUALINFO *, POLICYQUALINFO *)
/* 1097 */ _CFFI_OP(_CFFI_OP_NOOP, 455),
/* 1098 */ _CFFI_OP(_CFFI_OP_POINTER, 3104), // POLICYQUALINFO *
/* 1099 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1100 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_SCT *, SCT *)
/* 1101 */ _CFFI_OP(_CFFI_OP_POINTER, 3041), // Cryptography_STACK_OF_SCT *
/* 1102 */ _CFFI_OP(_CFFI_OP_POINTER, 3108), // SCT *
/* 1103 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1104 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_SCT const *)
/* 1105 */ _CFFI_OP(_CFFI_OP_NOOP, 481),
/* 1106 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1107 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_SSL_CIPHER *)
/* 1108 */ _CFFI_OP(_CFFI_OP_NOOP, 493),
/* 1109 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1110 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509 *)
/* 1111 */ _CFFI_OP(_CFFI_OP_NOOP, 413),
/* 1112 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1113 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509 *, X509 *)
/* 1114 */ _CFFI_OP(_CFFI_OP_NOOP, 413),
/* 1115 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1116 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1117 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_CRL *)
/* 1118 */ _CFFI_OP(_CFFI_OP_NOOP, 565),
/* 1119 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1120 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_CRL *, X509_CRL *)
/* 1121 */ _CFFI_OP(_CFFI_OP_NOOP, 565),
/* 1122 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1123 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1124 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_NAME *)
/* 1125 */ _CFFI_OP(_CFFI_OP_NOOP, 633),
/* 1126 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1127 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_NAME *, X509_NAME *)
/* 1128 */ _CFFI_OP(_CFFI_OP_NOOP, 633),
/* 1129 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 1130 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1131 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_NAME_ENTRY *)
/* 1132 */ _CFFI_OP(_CFFI_OP_NOOP, 175),
/* 1133 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1134 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_NAME_ENTRY *, X509_NAME_ENTRY *)
/* 1135 */ _CFFI_OP(_CFFI_OP_NOOP, 175),
/* 1136 */ _CFFI_OP(_CFFI_OP_NOOP, 43),
/* 1137 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1138 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_OBJECT *)
/* 1139 */ _CFFI_OP(_CFFI_OP_NOOP, 666),
/* 1140 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1141 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(Cryptography_STACK_OF_X509_REVOKED *)
/* 1142 */ _CFFI_OP(_CFFI_OP_NOOP, 682),
/* 1143 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1144 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH *)
/* 1145 */ _CFFI_OP(_CFFI_OP_NOOP, 196),
/* 1146 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1147 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH *, BIGNUM *, BIGNUM *)
/* 1148 */ _CFFI_OP(_CFFI_OP_NOOP, 196),
/* 1149 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1150 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1151 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1152 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH *, BIGNUM *, BIGNUM *, BIGNUM *)
/* 1153 */ _CFFI_OP(_CFFI_OP_NOOP, 196),
/* 1154 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1155 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1156 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1157 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1158 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH *, int, int, BN_GENCB *)
/* 1159 */ _CFFI_OP(_CFFI_OP_NOOP, 196),
/* 1160 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1161 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1162 */ _CFFI_OP(_CFFI_OP_NOOP, 848),
/* 1163 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1164 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH const *)
/* 1165 */ _CFFI_OP(_CFFI_OP_POINTER, 3049), // DH const *
/* 1166 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1167 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DH const *, int *)
/* 1168 */ _CFFI_OP(_CFFI_OP_NOOP, 1165),
/* 1169 */ _CFFI_OP(_CFFI_OP_POINTER, 2), // int *
/* 1170 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1171 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DSA *)
/* 1172 */ _CFFI_OP(_CFFI_OP_NOOP, 212),
/* 1173 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1174 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DSA *, BIGNUM *, BIGNUM *)
/* 1175 */ _CFFI_OP(_CFFI_OP_NOOP, 212),
/* 1176 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1177 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1178 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1179 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DSA *, BIGNUM *, BIGNUM *, BIGNUM *)
/* 1180 */ _CFFI_OP(_CFFI_OP_NOOP, 212),
/* 1181 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1182 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1183 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1184 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1185 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DSA *, int, unsigned char *, int, int *, unsigned long *, BN_GENCB *)
/* 1186 */ _CFFI_OP(_CFFI_OP_NOOP, 212),
/* 1187 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1188 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1189 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1190 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1191 */ _CFFI_OP(_CFFI_OP_POINTER, 742), // unsigned long *
/* 1192 */ _CFFI_OP(_CFFI_OP_NOOP, 848),
/* 1193 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1194 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(DSA const *)
/* 1195 */ _CFFI_OP(_CFFI_OP_POINTER, 3052), // DSA const *
/* 1196 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1197 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *)
/* 1198 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1199 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1200 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, BIGNUM *, BN_CTX *)
/* 1201 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1202 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1203 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1204 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1205 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, BIGNUM const *, BIGNUM const *, BN_CTX *)
/* 1206 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1207 */ _CFFI_OP(_CFFI_OP_POINTER, 3057), // EC_POINT *
/* 1208 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 1209 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 1210 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1211 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1212 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, BIGNUM const *, EC_POINT const *, BIGNUM const *, BN_CTX *)
/* 1213 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1214 */ _CFFI_OP(_CFFI_OP_NOOP, 1207),
/* 1215 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 1216 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 1217 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 1218 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1219 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1220 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, BIGNUM const *, int, BN_CTX *)
/* 1221 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1222 */ _CFFI_OP(_CFFI_OP_NOOP, 1207),
/* 1223 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 1224 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1225 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1226 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1227 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, BN_CTX *)
/* 1228 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1229 */ _CFFI_OP(_CFFI_OP_NOOP, 1207),
/* 1230 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1231 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1232 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, EC_POINT const *, BN_CTX *)
/* 1233 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1234 */ _CFFI_OP(_CFFI_OP_NOOP, 1207),
/* 1235 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 1236 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1237 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1238 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, EC_POINT const *, EC_POINT const *, BN_CTX *)
/* 1239 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1240 */ _CFFI_OP(_CFFI_OP_NOOP, 1207),
/* 1241 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 1242 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 1243 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1244 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1245 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT *, unsigned char const *, size_t, BN_CTX *)
/* 1246 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1247 */ _CFFI_OP(_CFFI_OP_NOOP, 1207),
/* 1248 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1249 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1250 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1251 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1252 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT const *)
/* 1253 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1254 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 1255 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1256 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT const *, BIGNUM *, BIGNUM *, BN_CTX *)
/* 1257 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1258 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 1259 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1260 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1261 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1262 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1263 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT const *, BN_CTX *)
/* 1264 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1265 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 1266 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1267 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1268 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_GROUP const *, EC_POINT const *, EC_POINT const *, BN_CTX *)
/* 1269 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1270 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 1271 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 1272 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1273 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1274 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY *)
/* 1275 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 1276 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1277 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY *, BIGNUM *, BIGNUM *)
/* 1278 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 1279 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1280 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1281 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1282 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY *, BIGNUM const *)
/* 1283 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 1284 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 1285 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1286 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY *, EC_GROUP const *)
/* 1287 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 1288 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 1289 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1290 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY *, EC_POINT const *)
/* 1291 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 1292 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 1293 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1294 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_KEY const *)
/* 1295 */ _CFFI_OP(_CFFI_OP_NOOP, 110),
/* 1296 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1297 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EC_METHOD const *)
/* 1298 */ _CFFI_OP(_CFFI_OP_POINTER, 3056), // EC_METHOD const *
/* 1299 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1300 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ENGINE *)
/* 1301 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 1302 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1303 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ENGINE *, char const *, char const *, int)
/* 1304 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 1305 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1306 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1307 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1308 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1309 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(ENGINE *, char const *, long, void *, void(*)(void), int)
/* 1310 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 1311 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1312 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 1313 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1314 */ _CFFI_OP(_CFFI_OP_POINTER, 2998), // void(*)(void)
/* 1315 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1316 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1317 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *)
/* 1318 */ _CFFI_OP(_CFFI_OP_POINTER, 3063), // EVP_CIPHER_CTX *
/* 1319 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1320 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *, EVP_CIPHER const *, ENGINE *, unsigned char const *, unsigned char const *, int)
/* 1321 */ _CFFI_OP(_CFFI_OP_NOOP, 1318),
/* 1322 */ _CFFI_OP(_CFFI_OP_NOOP, 885),
/* 1323 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 1324 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1325 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1326 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1327 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1328 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *, int)
/* 1329 */ _CFFI_OP(_CFFI_OP_NOOP, 1318),
/* 1330 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1331 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1332 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *, int, int, void *)
/* 1333 */ _CFFI_OP(_CFFI_OP_NOOP, 1318),
/* 1334 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1335 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1336 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1337 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1338 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *, unsigned char *, int *)
/* 1339 */ _CFFI_OP(_CFFI_OP_NOOP, 1318),
/* 1340 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1341 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1342 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1343 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_CIPHER_CTX *, unsigned char *, int *, unsigned char const *, int)
/* 1344 */ _CFFI_OP(_CFFI_OP_NOOP, 1318),
/* 1345 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1346 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1347 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1348 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1349 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1350 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, EVP_MD const *)
/* 1351 */ _CFFI_OP(_CFFI_OP_POINTER, 3065), // EVP_MD_CTX *
/* 1352 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 1353 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1354 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, EVP_MD const *, ENGINE *)
/* 1355 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1356 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 1357 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 1358 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1359 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, EVP_MD_CTX const *)
/* 1360 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1361 */ _CFFI_OP(_CFFI_OP_POINTER, 3065), // EVP_MD_CTX const *
/* 1362 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1363 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, EVP_PKEY_CTX * *, EVP_MD const *, ENGINE *, EVP_PKEY *)
/* 1364 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1365 */ _CFFI_OP(_CFFI_OP_POINTER, 309), // EVP_PKEY_CTX * *
/* 1366 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 1367 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 1368 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1369 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1370 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char *, size_t *)
/* 1371 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1372 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1373 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1374 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1375 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char *, size_t *, unsigned char const *, size_t)
/* 1376 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1377 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1378 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1379 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1380 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1381 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1382 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char *, size_t)
/* 1383 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1384 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1385 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1386 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1387 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char *, unsigned int *)
/* 1388 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1389 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1390 */ _CFFI_OP(_CFFI_OP_POINTER, 754), // unsigned int *
/* 1391 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1392 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char *, unsigned int *, EVP_PKEY *)
/* 1393 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1394 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1395 */ _CFFI_OP(_CFFI_OP_NOOP, 1390),
/* 1396 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1397 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1398 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char const *, size_t, unsigned char const *, size_t)
/* 1399 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1400 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1401 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1402 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1403 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1404 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1405 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, unsigned char const *, unsigned int, EVP_PKEY *)
/* 1406 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1407 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1408 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1409 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1410 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1411 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_MD_CTX *, void const *, size_t)
/* 1412 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 1413 */ _CFFI_OP(_CFFI_OP_NOOP, 126),
/* 1414 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1415 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1416 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *)
/* 1417 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1418 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1419 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, DH *)
/* 1420 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1421 */ _CFFI_OP(_CFFI_OP_NOOP, 196),
/* 1422 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1423 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, DSA *)
/* 1424 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1425 */ _CFFI_OP(_CFFI_OP_NOOP, 212),
/* 1426 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1427 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, EC_KEY *)
/* 1428 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1429 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 1430 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1431 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, RSA *)
/* 1432 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1433 */ _CFFI_OP(_CFFI_OP_NOOP, 476),
/* 1434 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1435 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, int)
/* 1436 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1437 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1438 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1439 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY *, unsigned char const *, size_t)
/* 1440 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1441 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1442 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1443 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1444 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY const *)
/* 1445 */ _CFFI_OP(_CFFI_OP_POINTER, 3066), // EVP_PKEY const *
/* 1446 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1447 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY const *, EVP_PKEY const *)
/* 1448 */ _CFFI_OP(_CFFI_OP_NOOP, 1445),
/* 1449 */ _CFFI_OP(_CFFI_OP_NOOP, 1445),
/* 1450 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1451 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY const *, unsigned char *, size_t *)
/* 1452 */ _CFFI_OP(_CFFI_OP_NOOP, 1445),
/* 1453 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1454 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1455 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1456 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *)
/* 1457 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 1458 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1459 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, EVP_MD *)
/* 1460 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 1461 */ _CFFI_OP(_CFFI_OP_POINTER, 3064), // EVP_MD *
/* 1462 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1463 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, EVP_MD const *)
/* 1464 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 1465 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 1466 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1467 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, EVP_PKEY * *)
/* 1468 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 1469 */ _CFFI_OP(_CFFI_OP_NOOP, 273),
/* 1470 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1471 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, EVP_PKEY *)
/* 1472 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 1473 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1474 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1475 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, int)
/* 1476 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 1477 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1478 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1479 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, unsigned char *, int)
/* 1480 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 1481 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1482 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1483 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1484 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, unsigned char *, size_t *)
/* 1485 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 1486 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1487 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1488 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1489 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, unsigned char *, size_t *, unsigned char const *, size_t)
/* 1490 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 1491 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1492 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1493 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1494 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1495 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1496 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(EVP_PKEY_CTX *, unsigned char const *, size_t, unsigned char const *, size_t)
/* 1497 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 1498 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1499 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1500 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1501 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1502 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1503 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(GENERAL_NAMES *)
/* 1504 */ _CFFI_OP(_CFFI_OP_NOOP, 316),
/* 1505 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1506 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(GENERAL_NAMES *, GENERAL_NAME *)
/* 1507 */ _CFFI_OP(_CFFI_OP_NOOP, 316),
/* 1508 */ _CFFI_OP(_CFFI_OP_NOOP, 928),
/* 1509 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1510 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(GENERAL_NAMES *, unsigned char * *)
/* 1511 */ _CFFI_OP(_CFFI_OP_NOOP, 316),
/* 1512 */ _CFFI_OP(_CFFI_OP_NOOP, 802),
/* 1513 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1514 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(HMAC_CTX *, HMAC_CTX *)
/* 1515 */ _CFFI_OP(_CFFI_OP_POINTER, 3072), // HMAC_CTX *
/* 1516 */ _CFFI_OP(_CFFI_OP_NOOP, 1515),
/* 1517 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1518 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(HMAC_CTX *, unsigned char *, unsigned int *)
/* 1519 */ _CFFI_OP(_CFFI_OP_NOOP, 1515),
/* 1520 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1521 */ _CFFI_OP(_CFFI_OP_NOOP, 1390),
/* 1522 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1523 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(HMAC_CTX *, unsigned char const *, size_t)
/* 1524 */ _CFFI_OP(_CFFI_OP_NOOP, 1515),
/* 1525 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1526 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1527 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1528 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(HMAC_CTX *, void const *, int, EVP_MD const *, ENGINE *)
/* 1529 */ _CFFI_OP(_CFFI_OP_NOOP, 1515),
/* 1530 */ _CFFI_OP(_CFFI_OP_NOOP, 126),
/* 1531 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1532 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 1533 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 1534 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1535 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(NETSCAPE_SPKI *, EVP_PKEY *)
/* 1536 */ _CFFI_OP(_CFFI_OP_NOOP, 288),
/* 1537 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1538 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1539 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(NETSCAPE_SPKI *, EVP_PKEY *, EVP_MD const *)
/* 1540 */ _CFFI_OP(_CFFI_OP_NOOP, 288),
/* 1541 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1542 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 1543 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1544 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP *)
/* 1545 */ _CFFI_OP(_CFFI_OP_NOOP, 387),
/* 1546 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1547 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP *, X509 *)
/* 1548 */ _CFFI_OP(_CFFI_OP_NOOP, 387),
/* 1549 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1550 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1551 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP *, X509 *, EVP_PKEY *, EVP_MD const *, Cryptography_STACK_OF_X509 *, unsigned long)
/* 1552 */ _CFFI_OP(_CFFI_OP_NOOP, 387),
/* 1553 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1554 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1555 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 1556 */ _CFFI_OP(_CFFI_OP_NOOP, 413),
/* 1557 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 1558 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1559 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP *, X509_EXTENSION *, int)
/* 1560 */ _CFFI_OP(_CFFI_OP_NOOP, 387),
/* 1561 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 1562 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1563 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1564 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_BASICRESP const *, ASN1_OCTET_STRING const * *, X509_NAME const * *)
/* 1565 */ _CFFI_OP(_CFFI_OP_NOOP, 17),
/* 1566 */ _CFFI_OP(_CFFI_OP_POINTER, 3016), // ASN1_OCTET_STRING const * *
/* 1567 */ _CFFI_OP(_CFFI_OP_POINTER, 2003), // X509_NAME const * *
/* 1568 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1569 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_REQUEST *)
/* 1570 */ _CFFI_OP(_CFFI_OP_NOOP, 365),
/* 1571 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1572 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_REQUEST *, X509_EXTENSION *, int)
/* 1573 */ _CFFI_OP(_CFFI_OP_NOOP, 365),
/* 1574 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 1575 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1576 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1577 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_RESPDATA *, unsigned char * *)
/* 1578 */ _CFFI_OP(_CFFI_OP_POINTER, 3084), // OCSP_RESPDATA *
/* 1579 */ _CFFI_OP(_CFFI_OP_NOOP, 802),
/* 1580 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1581 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_RESPONSE *)
/* 1582 */ _CFFI_OP(_CFFI_OP_NOOP, 349),
/* 1583 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1584 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_SINGLERESP *)
/* 1585 */ _CFFI_OP(_CFFI_OP_NOOP, 594),
/* 1586 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1587 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(OCSP_SINGLERESP *, int *, ASN1_GENERALIZEDTIME * *, ASN1_GENERALIZEDTIME * *, ASN1_GENERALIZEDTIME * *)
/* 1588 */ _CFFI_OP(_CFFI_OP_NOOP, 594),
/* 1589 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1590 */ _CFFI_OP(_CFFI_OP_NOOP, 14),
/* 1591 */ _CFFI_OP(_CFFI_OP_NOOP, 14),
/* 1592 */ _CFFI_OP(_CFFI_OP_NOOP, 14),
/* 1593 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1594 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(PKCS12 *, char const *, EVP_PKEY * *, X509 * *, Cryptography_STACK_OF_X509 * *)
/* 1595 */ _CFFI_OP(_CFFI_OP_NOOP, 940),
/* 1596 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1597 */ _CFFI_OP(_CFFI_OP_NOOP, 273),
/* 1598 */ _CFFI_OP(_CFFI_OP_NOOP, 518),
/* 1599 */ _CFFI_OP(_CFFI_OP_POINTER, 413), // Cryptography_STACK_OF_X509 * *
/* 1600 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1601 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(PKCS7 *)
/* 1602 */ _CFFI_OP(_CFFI_OP_NOOP, 442),
/* 1603 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1604 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(PKCS7 *, BIO *, int)
/* 1605 */ _CFFI_OP(_CFFI_OP_NOOP, 442),
/* 1606 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1607 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1608 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1609 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(PKCS7 *, Cryptography_STACK_OF_X509 *, X509_STORE *, BIO *, BIO *, int)
/* 1610 */ _CFFI_OP(_CFFI_OP_NOOP, 442),
/* 1611 */ _CFFI_OP(_CFFI_OP_NOOP, 413),
/* 1612 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 1613 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1614 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 1615 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1616 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1617 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RAND_METHOD const *)
/* 1618 */ _CFFI_OP(_CFFI_OP_POINTER, 3106), // RAND_METHOD const *
/* 1619 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1620 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RSA *, BIGNUM *, BIGNUM *)
/* 1621 */ _CFFI_OP(_CFFI_OP_NOOP, 476),
/* 1622 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1623 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1624 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1625 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RSA *, BIGNUM *, BIGNUM *, BIGNUM *)
/* 1626 */ _CFFI_OP(_CFFI_OP_NOOP, 476),
/* 1627 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1628 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1629 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1630 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1631 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RSA *, BN_CTX *)
/* 1632 */ _CFFI_OP(_CFFI_OP_NOOP, 476),
/* 1633 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 1634 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1635 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RSA *, int, BIGNUM *, BN_GENCB *)
/* 1636 */ _CFFI_OP(_CFFI_OP_NOOP, 476),
/* 1637 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1638 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 1639 */ _CFFI_OP(_CFFI_OP_NOOP, 848),
/* 1640 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1641 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(RSA const *)
/* 1642 */ _CFFI_OP(_CFFI_OP_NOOP, 967),
/* 1643 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1644 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SCT *, ct_log_entry_type_t)
/* 1645 */ _CFFI_OP(_CFFI_OP_NOOP, 1102),
/* 1646 */ _CFFI_OP(_CFFI_OP_ENUM, 0), // ct_log_entry_type_t
/* 1647 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1648 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SCT *, sct_source_t)
/* 1649 */ _CFFI_OP(_CFFI_OP_NOOP, 1102),
/* 1650 */ _CFFI_OP(_CFFI_OP_ENUM, 2), // sct_source_t
/* 1651 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1652 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SCT *, sct_version_t)
/* 1653 */ _CFFI_OP(_CFFI_OP_NOOP, 1102),
/* 1654 */ _CFFI_OP(_CFFI_OP_ENUM, 3), // sct_version_t
/* 1655 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1656 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SCT *, unsigned char *, size_t)
/* 1657 */ _CFFI_OP(_CFFI_OP_NOOP, 1102),
/* 1658 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1659 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1660 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1661 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *)
/* 1662 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1663 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1664 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, SSL_SESSION *)
/* 1665 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1666 */ _CFFI_OP(_CFFI_OP_POINTER, 3116), // SSL_SESSION *
/* 1667 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1668 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, char const *)
/* 1669 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1670 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1671 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1672 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, int *, void *)
/* 1673 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1674 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1675 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1676 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1677 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, int)
/* 1678 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1679 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1680 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1681 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, int, int *, int *, int *, unsigned char *, unsigned char *)
/* 1682 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1683 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1684 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1685 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1686 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1687 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1688 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1689 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1690 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, int, void *)
/* 1691 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1692 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1693 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1694 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1695 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char *, size_t, char const *, size_t, unsigned char const *, size_t, int)
/* 1696 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1697 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1698 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1699 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1700 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1701 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1702 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1703 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1704 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1705 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char *, unsigned int *)
/* 1706 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1707 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1708 */ _CFFI_OP(_CFFI_OP_NOOP, 1390),
/* 1709 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1710 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *)
/* 1711 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1712 */ _CFFI_OP(_CFFI_OP_NOOP, 75),
/* 1713 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1714 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1715 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1716 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1717 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1718 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned char const *, unsigned int)
/* 1719 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1720 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1721 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1722 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1723 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *)
/* 1724 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1725 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1726 */ _CFFI_OP(_CFFI_OP_NOOP, 75),
/* 1727 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1728 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1729 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1730 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1731 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, unsigned int, unsigned char const *, size_t, int *, void *)
/* 1732 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1733 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1734 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1735 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1736 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1737 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1738 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1739 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, void *)
/* 1740 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1741 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1742 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1743 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, void *, int)
/* 1744 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1745 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1746 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1747 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1748 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, void *, size_t, size_t *)
/* 1749 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1750 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1751 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1752 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1753 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1754 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, void const *, int)
/* 1755 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1756 */ _CFFI_OP(_CFFI_OP_NOOP, 126),
/* 1757 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1758 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1759 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL *, void const *, size_t, size_t *)
/* 1760 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 1761 */ _CFFI_OP(_CFFI_OP_NOOP, 126),
/* 1762 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 1763 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 1764 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1765 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL const *)
/* 1766 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 1767 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1768 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL const *, int)
/* 1769 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 1770 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1771 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1772 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CIPHER const *)
/* 1773 */ _CFFI_OP(_CFFI_OP_NOOP, 733),
/* 1774 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1775 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CIPHER const *, int *)
/* 1776 */ _CFFI_OP(_CFFI_OP_NOOP, 733),
/* 1777 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 1778 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1779 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *)
/* 1780 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1781 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1782 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, EVP_PKEY *)
/* 1783 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1784 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1785 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1786 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, X509 *)
/* 1787 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1788 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1789 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1790 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, char const *)
/* 1791 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1792 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1793 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1794 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, char const *, char const *)
/* 1795 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1796 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1797 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1798 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1799 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, char const *, int)
/* 1800 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1801 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1802 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1803 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1804 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, int, void *)
/* 1805 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1806 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1807 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1808 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1809 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, uint32_t)
/* 1810 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1811 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 22), // uint32_t
/* 1812 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1813 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, unsigned char const *, unsigned int)
/* 1814 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1815 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1816 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1817 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1818 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX *, unsigned int, int(*)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *), void(*)(SSL *, unsigned int, unsigned char const *, void *), void *, int(*)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *), void *)
/* 1819 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 1820 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1821 */ _CFFI_OP(_CFFI_OP_POINTER, 1723), // int(*)(SSL *, unsigned int, unsigned char const * *, size_t *, int *, void *)
/* 1822 */ _CFFI_OP(_CFFI_OP_POINTER, 2782), // void(*)(SSL *, unsigned int, unsigned char const *, void *)
/* 1823 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1824 */ _CFFI_OP(_CFFI_OP_POINTER, 1731), // int(*)(SSL *, unsigned int, unsigned char const *, size_t, int *, void *)
/* 1825 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 1826 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1827 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_CTX const *)
/* 1828 */ _CFFI_OP(_CFFI_OP_NOOP, 691),
/* 1829 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1830 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_SESSION *, unsigned char const *, unsigned int)
/* 1831 */ _CFFI_OP(_CFFI_OP_NOOP, 1666),
/* 1832 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1833 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 1834 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1835 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(SSL_SESSION const *)
/* 1836 */ _CFFI_OP(_CFFI_OP_POINTER, 3116), // SSL_SESSION const *
/* 1837 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1838 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 * *, X509_STORE_CTX *, X509 *)
/* 1839 */ _CFFI_OP(_CFFI_OP_NOOP, 518),
/* 1840 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 1841 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1842 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1843 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *)
/* 1844 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1845 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1846 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, ASN1_INTEGER *)
/* 1847 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1848 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 1849 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1850 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, ASN1_OCTET_STRING *)
/* 1851 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1852 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 1853 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1854 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, EVP_PKEY *)
/* 1855 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1856 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1857 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1858 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, EVP_PKEY *, EVP_MD const *)
/* 1859 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1860 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1861 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 1862 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1863 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, X509_EXTENSION *, int)
/* 1864 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1865 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 1866 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1867 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1868 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, X509_NAME *)
/* 1869 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1870 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 1871 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1872 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, long)
/* 1873 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1874 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 1875 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1876 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 *, unsigned char * *)
/* 1877 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 1878 */ _CFFI_OP(_CFFI_OP_NOOP, 802),
/* 1879 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1880 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 const *)
/* 1881 */ _CFFI_OP(_CFFI_OP_NOOP, 355),
/* 1882 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1883 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 const *, EVP_MD const *, unsigned char *, unsigned int *)
/* 1884 */ _CFFI_OP(_CFFI_OP_NOOP, 355),
/* 1885 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 1886 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1887 */ _CFFI_OP(_CFFI_OP_NOOP, 1390),
/* 1888 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1889 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509 const *, X509 const *)
/* 1890 */ _CFFI_OP(_CFFI_OP_NOOP, 355),
/* 1891 */ _CFFI_OP(_CFFI_OP_NOOP, 355),
/* 1892 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1893 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_ATTRIBUTE const *)
/* 1894 */ _CFFI_OP(_CFFI_OP_POINTER, 3123), // X509_ATTRIBUTE const *
/* 1895 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1896 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *)
/* 1897 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1898 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1899 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, ASN1_OCTET_STRING *)
/* 1900 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1901 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 1902 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1903 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, EVP_PKEY *)
/* 1904 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1905 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1906 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1907 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, EVP_PKEY *, EVP_MD const *)
/* 1908 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1909 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 1910 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 1911 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1912 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, X509_EXTENSION *, int)
/* 1913 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1914 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 1915 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1916 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1917 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, X509_NAME *)
/* 1918 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1919 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 1920 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1921 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, X509_REVOKED * *, ASN1_INTEGER *)
/* 1922 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1923 */ _CFFI_OP(_CFFI_OP_POINTER, 619), // X509_REVOKED * *
/* 1924 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 1925 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1926 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, X509_REVOKED *)
/* 1927 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1928 */ _CFFI_OP(_CFFI_OP_NOOP, 619),
/* 1929 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1930 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, long)
/* 1931 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1932 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 1933 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1934 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL *, unsigned char * *)
/* 1935 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 1936 */ _CFFI_OP(_CFFI_OP_NOOP, 802),
/* 1937 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1938 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_CRL const *, X509_CRL const *)
/* 1939 */ _CFFI_OP(_CFFI_OP_POINTER, 3124), // X509_CRL const *
/* 1940 */ _CFFI_OP(_CFFI_OP_NOOP, 1939),
/* 1941 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1942 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_EXTENSION *)
/* 1943 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 1944 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1945 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_EXTENSIONS *)
/* 1946 */ _CFFI_OP(_CFFI_OP_NOOP, 615),
/* 1947 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1948 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_EXTENSIONS *, X509_EXTENSION *)
/* 1949 */ _CFFI_OP(_CFFI_OP_NOOP, 615),
/* 1950 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 1951 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1952 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_EXTENSIONS *, X509_EXTENSION *, int)
/* 1953 */ _CFFI_OP(_CFFI_OP_NOOP, 615),
/* 1954 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 1955 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1956 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1957 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *)
/* 1958 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 1959 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1960 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, ASN1_OBJECT *, int, unsigned char *, int, int, int)
/* 1961 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 1962 */ _CFFI_OP(_CFFI_OP_NOOP, 607),
/* 1963 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1964 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1965 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1966 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1967 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1968 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1969 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, X509_NAME_ENTRY *, int, int)
/* 1970 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 1971 */ _CFFI_OP(_CFFI_OP_NOOP, 43),
/* 1972 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1973 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1974 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1975 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, char const *, int, unsigned char const *, int, int, int)
/* 1976 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 1977 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 1978 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1979 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 1980 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1981 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1982 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1983 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1984 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, int, int)
/* 1985 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 1986 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1987 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1988 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1989 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, int, int, unsigned char *, int, int, int)
/* 1990 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 1991 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1992 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1993 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 1994 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1995 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1996 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 1997 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 1998 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME *, unsigned char * *)
/* 1999 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 2000 */ _CFFI_OP(_CFFI_OP_NOOP, 802),
/* 2001 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2002 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME const *, X509_NAME const *)
/* 2003 */ _CFFI_OP(_CFFI_OP_POINTER, 3127), // X509_NAME const *
/* 2004 */ _CFFI_OP(_CFFI_OP_NOOP, 2003),
/* 2005 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2006 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_NAME_ENTRY *)
/* 2007 */ _CFFI_OP(_CFFI_OP_NOOP, 43),
/* 2008 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2009 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_OBJECT const *)
/* 2010 */ _CFFI_OP(_CFFI_OP_POINTER, 3129), // X509_OBJECT const *
/* 2011 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2012 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, ASN1_OBJECT const *, int, unsigned char const *, int)
/* 2013 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 2014 */ _CFFI_OP(_CFFI_OP_NOOP, 765),
/* 2015 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2016 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2017 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2018 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2019 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, EVP_PKEY *)
/* 2020 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 2021 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 2022 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2023 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, EVP_PKEY *, EVP_MD const *)
/* 2024 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 2025 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 2026 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 2027 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2028 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, X509_EXTENSIONS *)
/* 2029 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 2030 */ _CFFI_OP(_CFFI_OP_NOOP, 615),
/* 2031 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2032 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, X509_NAME *)
/* 2033 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 2034 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 2035 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2036 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, long)
/* 2037 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 2038 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2039 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2040 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ *, unsigned char * *)
/* 2041 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 2042 */ _CFFI_OP(_CFFI_OP_NOOP, 802),
/* 2043 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2044 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REQ const *, ASN1_OBJECT const *, int)
/* 2045 */ _CFFI_OP(_CFFI_OP_NOOP, 551),
/* 2046 */ _CFFI_OP(_CFFI_OP_NOOP, 765),
/* 2047 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2048 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2049 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REVOKED *)
/* 2050 */ _CFFI_OP(_CFFI_OP_NOOP, 619),
/* 2051 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2052 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REVOKED *, ASN1_INTEGER *)
/* 2053 */ _CFFI_OP(_CFFI_OP_NOOP, 619),
/* 2054 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 2055 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2056 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REVOKED *, ASN1_OCTET_STRING *)
/* 2057 */ _CFFI_OP(_CFFI_OP_NOOP, 619),
/* 2058 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 2059 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2060 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REVOKED *, X509_EXTENSION *, int)
/* 2061 */ _CFFI_OP(_CFFI_OP_NOOP, 619),
/* 2062 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 2063 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2064 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2065 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_REVOKED *, int, void *, int, unsigned long)
/* 2066 */ _CFFI_OP(_CFFI_OP_NOOP, 619),
/* 2067 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2068 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2069 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2070 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2071 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2072 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *)
/* 2073 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2074 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2075 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *, X509 *)
/* 2076 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2077 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2078 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2079 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *, X509_CRL *)
/* 2080 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2081 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 2082 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2083 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *, X509_VERIFY_PARAM *)
/* 2084 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2085 */ _CFFI_OP(_CFFI_OP_POINTER, 3134), // X509_VERIFY_PARAM *
/* 2086 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2087 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *, char const *, char const *)
/* 2088 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2089 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2090 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2091 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2092 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE *, unsigned long)
/* 2093 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2094 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2095 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2096 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE_CTX *)
/* 2097 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2098 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2099 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE_CTX *, X509_STORE *, X509 *, Cryptography_STACK_OF_X509 *)
/* 2100 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2101 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2102 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2103 */ _CFFI_OP(_CFFI_OP_NOOP, 413),
/* 2104 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2105 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE_CTX *, char const *)
/* 2106 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2107 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2108 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2109 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE_CTX *, int, void *)
/* 2110 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2111 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2112 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2113 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2114 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_STORE_CTX *, void *)
/* 2115 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2116 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2117 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2118 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, ASN1_OBJECT *)
/* 2119 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2120 */ _CFFI_OP(_CFFI_OP_NOOP, 607),
/* 2121 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2122 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, Cryptography_STACK_OF_ASN1_OBJECT *)
/* 2123 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2124 */ _CFFI_OP(_CFFI_OP_NOOP, 36),
/* 2125 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2126 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, char const *)
/* 2127 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2128 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2129 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2130 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, char const *, size_t)
/* 2131 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2132 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2133 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2134 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2135 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, int)
/* 2136 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2137 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2138 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2139 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, unsigned char const *, size_t)
/* 2140 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2141 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2142 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2143 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2144 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM *, unsigned long)
/* 2145 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2146 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2147 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2148 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(X509_VERIFY_PARAM const *)
/* 2149 */ _CFFI_OP(_CFFI_OP_POINTER, 3134), // X509_VERIFY_PARAM const *
/* 2150 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2151 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char *)
/* 2152 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 2153 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2154 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char *, int, ASN1_OBJECT const *, int)
/* 2155 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 2156 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2157 */ _CFFI_OP(_CFFI_OP_NOOP, 765),
/* 2158 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2159 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2160 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char *, int, int, void *)
/* 2161 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 2162 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2163 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2164 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2165 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2166 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char const *)
/* 2167 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2168 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2169 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char const *, int, unsigned char const *, int, int, EVP_MD const *, int, unsigned char *)
/* 2170 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2171 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2172 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2173 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2174 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2175 */ _CFFI_OP(_CFFI_OP_NOOP, 354),
/* 2176 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2177 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2178 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2179 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(char const *, size_t, unsigned char const *, size_t, uint64_t, uint64_t, uint64_t, uint64_t, unsigned char *, size_t)
/* 2180 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2181 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2182 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2183 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2184 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24), // uint64_t
/* 2185 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 2186 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 2187 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 2188 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2189 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2190 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2191 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int)
/* 2192 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2193 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2194 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, X509_STORE_CTX *)
/* 2195 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2196 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2197 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2198 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, int)
/* 2199 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2200 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2201 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2202 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, unsigned char const *, int, unsigned char *, unsigned int *, DSA *)
/* 2203 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2204 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2205 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2206 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2207 */ _CFFI_OP(_CFFI_OP_NOOP, 1390),
/* 2208 */ _CFFI_OP(_CFFI_OP_NOOP, 212),
/* 2209 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2210 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, unsigned char const *, int, unsigned char *, unsigned int *, EC_KEY *)
/* 2211 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2212 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2213 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2214 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2215 */ _CFFI_OP(_CFFI_OP_NOOP, 1390),
/* 2216 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 2217 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2218 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, unsigned char const *, int, unsigned char const *, int, DSA *)
/* 2219 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2220 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2221 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2222 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2223 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2224 */ _CFFI_OP(_CFFI_OP_NOOP, 212),
/* 2225 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2226 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(int, unsigned char const *, int, unsigned char const *, int, EC_KEY *)
/* 2227 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2228 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2229 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2230 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2231 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2232 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 2233 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2234 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(long, void *, CRYPTO_EX_new *, CRYPTO_EX_dup *, CRYPTO_EX_free *)
/* 2235 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2236 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2237 */ _CFFI_OP(_CFFI_OP_POINTER, 3032), // CRYPTO_EX_new *
/* 2238 */ _CFFI_OP(_CFFI_OP_POINTER, 3030), // CRYPTO_EX_dup *
/* 2239 */ _CFFI_OP(_CFFI_OP_POINTER, 3031), // CRYPTO_EX_free *
/* 2240 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2241 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned char * *, ASN1_OCTET_STRING *)
/* 2242 */ _CFFI_OP(_CFFI_OP_NOOP, 802),
/* 2243 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 2244 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2245 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned char * *, unsigned char *, unsigned char const *, unsigned int, unsigned char const *, unsigned int)
/* 2246 */ _CFFI_OP(_CFFI_OP_NOOP, 802),
/* 2247 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2248 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2249 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2250 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2251 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2252 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2253 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned char *, BIGNUM const *, DH *)
/* 2254 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2255 */ _CFFI_OP(_CFFI_OP_NOOP, 92),
/* 2256 */ _CFFI_OP(_CFFI_OP_NOOP, 196),
/* 2257 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2258 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned char *, int)
/* 2259 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2260 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2261 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2262 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned int)
/* 2263 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2264 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2265 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(unsigned long)
/* 2266 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2267 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2268 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(void *(*)(size_t, char const *, int), void *(*)(void *, size_t, char const *, int), void(*)(void *, char const *, int))
/* 2269 */ _CFFI_OP(_CFFI_OP_POINTER, 2511), // void *(*)(size_t, char const *, int)
/* 2270 */ _CFFI_OP(_CFFI_OP_POINTER, 2516), // void *(*)(void *, size_t, char const *, int)
/* 2271 */ _CFFI_OP(_CFFI_OP_POINTER, 2988), // void(*)(void *, char const *, int)
/* 2272 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2273 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(void *, size_t, EC_POINT const *, EC_KEY *, void *(*)(void const *, size_t, void *, size_t *))
/* 2274 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2275 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2276 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 2277 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 2278 */ _CFFI_OP(_CFFI_OP_POINTER, 2527), // void *(*)(void const *, size_t, void *, size_t *)
/* 2279 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2280 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2), // int()(void)
/* 2281 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2282 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2926), // int(*()(X509_STORE *))(X509 * *, X509_STORE_CTX *, X509 *)
/* 2283 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2284 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2285 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(ASN1_ENUMERATED *)
/* 2286 */ _CFFI_OP(_CFFI_OP_NOOP, 757),
/* 2287 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2288 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(BIO *, char * *)
/* 2289 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 2290 */ _CFFI_OP(_CFFI_OP_POINTER, 409), // char * *
/* 2291 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2292 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(BIO *, int)
/* 2293 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 2294 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2295 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2296 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL *)
/* 2297 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2298 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2299 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL *, EVP_PKEY * *)
/* 2300 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2301 */ _CFFI_OP(_CFFI_OP_NOOP, 273),
/* 2302 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2303 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL *, int)
/* 2304 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2305 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2306 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2307 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL *, long)
/* 2308 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2309 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2310 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2311 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL *, time_t *, long *)
/* 2312 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2313 */ _CFFI_OP(_CFFI_OP_POINTER, 10), // time_t *
/* 2314 */ _CFFI_OP(_CFFI_OP_POINTER, 51), // long *
/* 2315 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2316 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL *, unsigned char *, int)
/* 2317 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2318 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2319 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2320 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2321 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL *, unsigned char const * *)
/* 2322 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2323 */ _CFFI_OP(_CFFI_OP_NOOP, 75),
/* 2324 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2325 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL const *)
/* 2326 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 2327 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2328 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL_CTX *)
/* 2329 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2330 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2331 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL_CTX *, char const *)
/* 2332 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2333 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2334 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2335 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL_CTX *, int(*)(SSL *, void *))
/* 2336 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2337 */ _CFFI_OP(_CFFI_OP_POINTER, 1739), // int(*)(SSL *, void *)
/* 2338 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2339 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL_CTX *, int)
/* 2340 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2341 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2342 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2343 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL_CTX *, long)
/* 2344 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2345 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2346 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2347 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL_CTX *, void *)
/* 2348 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2349 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2350 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2351 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL_CTX const *)
/* 2352 */ _CFFI_OP(_CFFI_OP_NOOP, 691),
/* 2353 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2354 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(SSL_SESSION const *)
/* 2355 */ _CFFI_OP(_CFFI_OP_NOOP, 1836),
/* 2356 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2357 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(X509 *)
/* 2358 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2359 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2360 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(X509_CRL *)
/* 2361 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 2362 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2363 */ _CFFI_OP(_CFFI_OP_FUNCTION, 51), // long()(X509_REQ *)
/* 2364 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 2365 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2366 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1654), // sct_version_t()(SCT const *)
/* 2367 */ _CFFI_OP(_CFFI_OP_NOOP, 747),
/* 2368 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2369 */ _CFFI_OP(_CFFI_OP_FUNCTION, 300), // size_t()(BIO *)
/* 2370 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 2371 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2372 */ _CFFI_OP(_CFFI_OP_FUNCTION, 300), // size_t()(EC_GROUP const *, EC_POINT const *, point_conversion_form_t, unsigned char *, size_t, BN_CTX *)
/* 2373 */ _CFFI_OP(_CFFI_OP_NOOP, 238),
/* 2374 */ _CFFI_OP(_CFFI_OP_NOOP, 244),
/* 2375 */ _CFFI_OP(_CFFI_OP_ENUM, 1), // point_conversion_form_t
/* 2376 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2377 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2378 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 2379 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2380 */ _CFFI_OP(_CFFI_OP_FUNCTION, 300), // size_t()(EC_builtin_curve *, size_t)
/* 2381 */ _CFFI_OP(_CFFI_OP_POINTER, 3058), // EC_builtin_curve *
/* 2382 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2383 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2384 */ _CFFI_OP(_CFFI_OP_FUNCTION, 300), // size_t()(EVP_PKEY *, unsigned char * *)
/* 2385 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 2386 */ _CFFI_OP(_CFFI_OP_NOOP, 802),
/* 2387 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2388 */ _CFFI_OP(_CFFI_OP_FUNCTION, 300), // size_t()(SCT const *, unsigned char * *)
/* 2389 */ _CFFI_OP(_CFFI_OP_NOOP, 747),
/* 2390 */ _CFFI_OP(_CFFI_OP_NOOP, 802),
/* 2391 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2392 */ _CFFI_OP(_CFFI_OP_FUNCTION, 300), // size_t()(SSL const *, unsigned char *, size_t)
/* 2393 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 2394 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2395 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2396 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2397 */ _CFFI_OP(_CFFI_OP_FUNCTION, 300), // size_t()(SSL const *, void *, size_t)
/* 2398 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 2399 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2400 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2401 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2402 */ _CFFI_OP(_CFFI_OP_FUNCTION, 300), // size_t()(SSL_SESSION const *, unsigned char *, size_t)
/* 2403 */ _CFFI_OP(_CFFI_OP_NOOP, 1836),
/* 2404 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2405 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2406 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2407 */ _CFFI_OP(_CFFI_OP_FUNCTION, 1811), // uint32_t()(SSL_SESSION const *)
/* 2408 */ _CFFI_OP(_CFFI_OP_NOOP, 1836),
/* 2409 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2410 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2184), // uint64_t()(SCT const *)
/* 2411 */ _CFFI_OP(_CFFI_OP_NOOP, 747),
/* 2412 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2413 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2184), // uint64_t()(SSL_CIPHER const *)
/* 2414 */ _CFFI_OP(_CFFI_OP_NOOP, 733),
/* 2415 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2416 */ _CFFI_OP(_CFFI_OP_FUNCTION, 752), // unsigned char *()(ASN1_OCTET_STRING *)
/* 2417 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 2418 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2419 */ _CFFI_OP(_CFFI_OP_FUNCTION, 752), // unsigned char *()(X509 *, int *)
/* 2420 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2421 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 2422 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2423 */ _CFFI_OP(_CFFI_OP_FUNCTION, 103), // unsigned char const *()(SSL_SESSION const *, unsigned int *)
/* 2424 */ _CFFI_OP(_CFFI_OP_NOOP, 1836),
/* 2425 */ _CFFI_OP(_CFFI_OP_NOOP, 1390),
/* 2426 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2427 */ _CFFI_OP(_CFFI_OP_FUNCTION, 754), // unsigned int()(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int)
/* 2428 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2429 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2430 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 2431 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2432 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2433 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2434 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2435 */ _CFFI_OP(_CFFI_OP_FUNCTION, 754), // unsigned int()(SSL *, char const *, unsigned char *, unsigned int)
/* 2436 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2437 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2438 */ _CFFI_OP(_CFFI_OP_NOOP, 752),
/* 2439 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2440 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2441 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(SSL *)
/* 2442 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2443 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2444 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(SSL *, unsigned long)
/* 2445 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2446 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2447 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2448 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(SSL_CTX *)
/* 2449 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2450 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2451 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(SSL_CTX *, DH *)
/* 2452 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2453 */ _CFFI_OP(_CFFI_OP_NOOP, 196),
/* 2454 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2455 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(SSL_CTX *, EC_KEY *)
/* 2456 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2457 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 2458 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2459 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(SSL_CTX *, X509 *)
/* 2460 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2461 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2462 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2463 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(SSL_CTX *, unsigned long)
/* 2464 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2465 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2466 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2467 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(X509 *)
/* 2468 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2469 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2470 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(X509_NAME *)
/* 2471 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 2472 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2473 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(X509_VERIFY_PARAM *)
/* 2474 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2475 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2476 */ _CFFI_OP(_CFFI_OP_FUNCTION, 742), // unsigned long()(void)
/* 2477 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2478 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(SSL const *, int)
/* 2479 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 2480 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2481 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2482 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(SSL_CTX const *, int)
/* 2483 */ _CFFI_OP(_CFFI_OP_NOOP, 691),
/* 2484 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2485 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2486 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(X509 *, int, int *, int *)
/* 2487 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2488 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2489 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 2490 */ _CFFI_OP(_CFFI_OP_NOOP, 1169),
/* 2491 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2492 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(X509V3_CTX *)
/* 2493 */ _CFFI_OP(_CFFI_OP_NOOP, 575),
/* 2494 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2495 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(X509_ATTRIBUTE *, int, int, void *)
/* 2496 */ _CFFI_OP(_CFFI_OP_NOOP, 79),
/* 2497 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2498 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2499 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2500 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2501 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(X509_EXTENSION *)
/* 2502 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 2503 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2504 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(X509_STORE_CTX *, int)
/* 2505 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2506 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2507 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2508 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(size_t)
/* 2509 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2510 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2511 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(size_t, char const *, int)
/* 2512 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2513 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2514 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2515 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2516 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(void *, size_t, char const *, int)
/* 2517 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2518 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2519 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2520 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2521 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2522 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(void *, unsigned char const * *, long)
/* 2523 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2524 */ _CFFI_OP(_CFFI_OP_NOOP, 75),
/* 2525 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 9),
/* 2526 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2527 */ _CFFI_OP(_CFFI_OP_FUNCTION, 193), // void *()(void const *, size_t, void *, size_t *)
/* 2528 */ _CFFI_OP(_CFFI_OP_NOOP, 126),
/* 2529 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2530 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2531 */ _CFFI_OP(_CFFI_OP_NOOP, 1037),
/* 2532 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2533 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(ACCESS_DESCRIPTION *)
/* 2534 */ _CFFI_OP(_CFFI_OP_NOOP, 1056),
/* 2535 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2536 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(ASN1_ENUMERATED *)
/* 2537 */ _CFFI_OP(_CFFI_OP_NOOP, 757),
/* 2538 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2539 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(ASN1_GENERALIZEDTIME *)
/* 2540 */ _CFFI_OP(_CFFI_OP_NOOP, 9),
/* 2541 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2542 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(ASN1_INTEGER *)
/* 2543 */ _CFFI_OP(_CFFI_OP_NOOP, 21),
/* 2544 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2545 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(ASN1_OBJECT *)
/* 2546 */ _CFFI_OP(_CFFI_OP_NOOP, 607),
/* 2547 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2548 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(ASN1_OCTET_STRING *)
/* 2549 */ _CFFI_OP(_CFFI_OP_NOOP, 13),
/* 2550 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2551 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(ASN1_OCTET_STRING const * *, X509_ALGOR const * *, X509 const *)
/* 2552 */ _CFFI_OP(_CFFI_OP_NOOP, 1566),
/* 2553 */ _CFFI_OP(_CFFI_OP_POINTER, 3121), // X509_ALGOR const * *
/* 2554 */ _CFFI_OP(_CFFI_OP_NOOP, 355),
/* 2555 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2556 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(AUTHORITY_KEYID *)
/* 2557 */ _CFFI_OP(_CFFI_OP_POINTER, 3019), // AUTHORITY_KEYID *
/* 2558 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2559 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(BASIC_CONSTRAINTS *)
/* 2560 */ _CFFI_OP(_CFFI_OP_POINTER, 3020), // BASIC_CONSTRAINTS *
/* 2561 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2562 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(BIGNUM *)
/* 2563 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 2564 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2565 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(BIGNUM *, int)
/* 2566 */ _CFFI_OP(_CFFI_OP_NOOP, 20),
/* 2567 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2568 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2569 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(BIO *)
/* 2570 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 2571 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2572 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(BN_CTX *)
/* 2573 */ _CFFI_OP(_CFFI_OP_NOOP, 94),
/* 2574 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2575 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(BN_MONT_CTX *)
/* 2576 */ _CFFI_OP(_CFFI_OP_NOOP, 830),
/* 2577 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2578 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(CMAC_CTX *)
/* 2579 */ _CFFI_OP(_CFFI_OP_NOOP, 1031),
/* 2580 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2581 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_ACCESS_DESCRIPTION *)
/* 2582 */ _CFFI_OP(_CFFI_OP_NOOP, 1),
/* 2583 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2584 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_ACCESS_DESCRIPTION *, void(*)(ACCESS_DESCRIPTION *))
/* 2585 */ _CFFI_OP(_CFFI_OP_NOOP, 1),
/* 2586 */ _CFFI_OP(_CFFI_OP_POINTER, 2533), // void(*)(ACCESS_DESCRIPTION *)
/* 2587 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2588 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_ASN1_INTEGER *)
/* 2589 */ _CFFI_OP(_CFFI_OP_NOOP, 24),
/* 2590 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2591 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_ASN1_OBJECT *)
/* 2592 */ _CFFI_OP(_CFFI_OP_NOOP, 36),
/* 2593 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2594 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_DIST_POINT *)
/* 2595 */ _CFFI_OP(_CFFI_OP_NOOP, 204),
/* 2596 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2597 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_DIST_POINT *, void(*)(DIST_POINT *))
/* 2598 */ _CFFI_OP(_CFFI_OP_NOOP, 204),
/* 2599 */ _CFFI_OP(_CFFI_OP_POINTER, 2640), // void(*)(DIST_POINT *)
/* 2600 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2601 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_GENERAL_SUBTREE *)
/* 2602 */ _CFFI_OP(_CFFI_OP_NOOP, 329),
/* 2603 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2604 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_POLICYINFO *)
/* 2605 */ _CFFI_OP(_CFFI_OP_NOOP, 449),
/* 2606 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2607 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_POLICYINFO *, void(*)(POLICYINFO *))
/* 2608 */ _CFFI_OP(_CFFI_OP_NOOP, 449),
/* 2609 */ _CFFI_OP(_CFFI_OP_POINTER, 2734), // void(*)(POLICYINFO *)
/* 2610 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2611 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_POLICYQUALINFO *)
/* 2612 */ _CFFI_OP(_CFFI_OP_NOOP, 455),
/* 2613 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2614 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_SCT *)
/* 2615 */ _CFFI_OP(_CFFI_OP_NOOP, 1101),
/* 2616 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2617 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_X509 *)
/* 2618 */ _CFFI_OP(_CFFI_OP_NOOP, 413),
/* 2619 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2620 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_X509_CRL *)
/* 2621 */ _CFFI_OP(_CFFI_OP_NOOP, 565),
/* 2622 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2623 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(Cryptography_STACK_OF_X509_NAME *)
/* 2624 */ _CFFI_OP(_CFFI_OP_NOOP, 633),
/* 2625 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2626 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(DH *)
/* 2627 */ _CFFI_OP(_CFFI_OP_NOOP, 196),
/* 2628 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2629 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(DH const *, BIGNUM const * *, BIGNUM const * *)
/* 2630 */ _CFFI_OP(_CFFI_OP_NOOP, 1165),
/* 2631 */ _CFFI_OP(_CFFI_OP_POINTER, 92), // BIGNUM const * *
/* 2632 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2633 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2634 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(DH const *, BIGNUM const * *, BIGNUM const * *, BIGNUM const * *)
/* 2635 */ _CFFI_OP(_CFFI_OP_NOOP, 1165),
/* 2636 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2637 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2638 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2639 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2640 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(DIST_POINT *)
/* 2641 */ _CFFI_OP(_CFFI_OP_NOOP, 1077),
/* 2642 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2643 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(DIST_POINT_NAME *)
/* 2644 */ _CFFI_OP(_CFFI_OP_POINTER, 3051), // DIST_POINT_NAME *
/* 2645 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2646 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(DSA *)
/* 2647 */ _CFFI_OP(_CFFI_OP_NOOP, 212),
/* 2648 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2649 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(DSA const *, BIGNUM const * *, BIGNUM const * *)
/* 2650 */ _CFFI_OP(_CFFI_OP_NOOP, 1195),
/* 2651 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2652 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2653 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2654 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(DSA const *, BIGNUM const * *, BIGNUM const * *, BIGNUM const * *)
/* 2655 */ _CFFI_OP(_CFFI_OP_NOOP, 1195),
/* 2656 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2657 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2658 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2659 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2660 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(EC_GROUP *)
/* 2661 */ _CFFI_OP(_CFFI_OP_POINTER, 3054), // EC_GROUP *
/* 2662 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2663 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(EC_KEY *)
/* 2664 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 2665 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2666 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(EC_KEY *, int)
/* 2667 */ _CFFI_OP(_CFFI_OP_NOOP, 893),
/* 2668 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2669 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2670 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(EC_POINT *)
/* 2671 */ _CFFI_OP(_CFFI_OP_NOOP, 1207),
/* 2672 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2673 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(ENGINE *)
/* 2674 */ _CFFI_OP(_CFFI_OP_NOOP, 282),
/* 2675 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2676 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(EVP_CIPHER_CTX *)
/* 2677 */ _CFFI_OP(_CFFI_OP_NOOP, 1318),
/* 2678 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2679 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(EVP_MD_CTX *)
/* 2680 */ _CFFI_OP(_CFFI_OP_NOOP, 1351),
/* 2681 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2682 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(EVP_PKEY *)
/* 2683 */ _CFFI_OP(_CFFI_OP_NOOP, 199),
/* 2684 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2685 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(EVP_PKEY_CTX *)
/* 2686 */ _CFFI_OP(_CFFI_OP_NOOP, 309),
/* 2687 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2688 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(GENERAL_NAME *)
/* 2689 */ _CFFI_OP(_CFFI_OP_NOOP, 928),
/* 2690 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2691 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(GENERAL_NAMES *)
/* 2692 */ _CFFI_OP(_CFFI_OP_NOOP, 316),
/* 2693 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2694 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(GENERAL_NAMES *, void(*)(GENERAL_NAME *))
/* 2695 */ _CFFI_OP(_CFFI_OP_NOOP, 316),
/* 2696 */ _CFFI_OP(_CFFI_OP_POINTER, 2688), // void(*)(GENERAL_NAME *)
/* 2697 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2698 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(HMAC_CTX *)
/* 2699 */ _CFFI_OP(_CFFI_OP_NOOP, 1515),
/* 2700 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2701 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(ISSUING_DIST_POINT *)
/* 2702 */ _CFFI_OP(_CFFI_OP_POINTER, 3073), // ISSUING_DIST_POINT *
/* 2703 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2704 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(NAME_CONSTRAINTS *)
/* 2705 */ _CFFI_OP(_CFFI_OP_POINTER, 3074), // NAME_CONSTRAINTS *
/* 2706 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2707 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(NETSCAPE_SPKI *)
/* 2708 */ _CFFI_OP(_CFFI_OP_NOOP, 288),
/* 2709 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2710 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(NOTICEREF *)
/* 2711 */ _CFFI_OP(_CFFI_OP_POINTER, 3076), // NOTICEREF *
/* 2712 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2713 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(OCSP_BASICRESP *)
/* 2714 */ _CFFI_OP(_CFFI_OP_NOOP, 387),
/* 2715 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2716 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(OCSP_CERTID *)
/* 2717 */ _CFFI_OP(_CFFI_OP_NOOP, 366),
/* 2718 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2719 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(OCSP_REQUEST *)
/* 2720 */ _CFFI_OP(_CFFI_OP_NOOP, 365),
/* 2721 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2722 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(OCSP_RESPONSE *)
/* 2723 */ _CFFI_OP(_CFFI_OP_NOOP, 349),
/* 2724 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2725 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(OTHERNAME *)
/* 2726 */ _CFFI_OP(_CFFI_OP_POINTER, 3087), // OTHERNAME *
/* 2727 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2728 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(PKCS12 *)
/* 2729 */ _CFFI_OP(_CFFI_OP_NOOP, 940),
/* 2730 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2731 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(PKCS7 *)
/* 2732 */ _CFFI_OP(_CFFI_OP_NOOP, 442),
/* 2733 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2734 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(POLICYINFO *)
/* 2735 */ _CFFI_OP(_CFFI_OP_NOOP, 1091),
/* 2736 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2737 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(POLICYQUALINFO *)
/* 2738 */ _CFFI_OP(_CFFI_OP_NOOP, 1098),
/* 2739 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2740 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(POLICY_CONSTRAINTS *)
/* 2741 */ _CFFI_OP(_CFFI_OP_POINTER, 3105), // POLICY_CONSTRAINTS *
/* 2742 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2743 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(RSA *)
/* 2744 */ _CFFI_OP(_CFFI_OP_NOOP, 476),
/* 2745 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2746 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(RSA const *, BIGNUM const * *, BIGNUM const * *)
/* 2747 */ _CFFI_OP(_CFFI_OP_NOOP, 967),
/* 2748 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2749 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2750 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2751 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(RSA const *, BIGNUM const * *, BIGNUM const * *, BIGNUM const * *)
/* 2752 */ _CFFI_OP(_CFFI_OP_NOOP, 967),
/* 2753 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2754 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2755 */ _CFFI_OP(_CFFI_OP_NOOP, 2631),
/* 2756 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2757 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SCT *, uint64_t)
/* 2758 */ _CFFI_OP(_CFFI_OP_NOOP, 1102),
/* 2759 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 24),
/* 2760 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2761 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL *)
/* 2762 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2763 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2764 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL *, BIO *, BIO *)
/* 2765 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2766 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 2767 */ _CFFI_OP(_CFFI_OP_NOOP, 186),
/* 2768 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2769 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL *, char *)
/* 2770 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2771 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 2772 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2773 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL *, int(*)(SSL *, void *), void *)
/* 2774 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2775 */ _CFFI_OP(_CFFI_OP_NOOP, 2337),
/* 2776 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2777 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2778 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL *, int)
/* 2779 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2780 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2781 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2782 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL *, unsigned int, unsigned char const *, void *)
/* 2783 */ _CFFI_OP(_CFFI_OP_NOOP, 487),
/* 2784 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2785 */ _CFFI_OP(_CFFI_OP_NOOP, 103),
/* 2786 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2787 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2788 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL const *, char const *)
/* 2789 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 2790 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2791 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2792 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL const *, int, int)
/* 2793 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 2794 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2795 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2796 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2797 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL const *, unsigned char const * *, unsigned int *)
/* 2798 */ _CFFI_OP(_CFFI_OP_NOOP, 157),
/* 2799 */ _CFFI_OP(_CFFI_OP_NOOP, 75),
/* 2800 */ _CFFI_OP(_CFFI_OP_NOOP, 1390),
/* 2801 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2802 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *)
/* 2803 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2804 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2805 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, Cryptography_STACK_OF_X509_NAME *)
/* 2806 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2807 */ _CFFI_OP(_CFFI_OP_NOOP, 633),
/* 2808 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2809 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, X509_STORE *)
/* 2810 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2811 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2812 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2813 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, int(*)(SSL *, int *, void *))
/* 2814 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2815 */ _CFFI_OP(_CFFI_OP_POINTER, 1672), // int(*)(SSL *, int *, void *)
/* 2816 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2817 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, int(*)(SSL *, unsigned char *, unsigned int *))
/* 2818 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2819 */ _CFFI_OP(_CFFI_OP_POINTER, 1705), // int(*)(SSL *, unsigned char *, unsigned int *)
/* 2820 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2821 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, int(*)(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *), void *)
/* 2822 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2823 */ _CFFI_OP(_CFFI_OP_POINTER, 1710), // int(*)(SSL *, unsigned char const * *, unsigned char *, unsigned char const *, unsigned int, void *)
/* 2824 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2825 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2826 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, int(*)(SSL *, void *), void *)
/* 2827 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2828 */ _CFFI_OP(_CFFI_OP_NOOP, 2337),
/* 2829 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2830 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2831 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, int(*)(X509_STORE_CTX *, void *), void *)
/* 2832 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2833 */ _CFFI_OP(_CFFI_OP_POINTER, 2114), // int(*)(X509_STORE_CTX *, void *)
/* 2834 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2835 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2836 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, int(*)(char *, int, int, void *))
/* 2837 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2838 */ _CFFI_OP(_CFFI_OP_NOOP, 192),
/* 2839 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2840 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, int)
/* 2841 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2842 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2843 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2844 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, int, int(*)(int, X509_STORE_CTX *))
/* 2845 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2846 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2847 */ _CFFI_OP(_CFFI_OP_POINTER, 2194), // int(*)(int, X509_STORE_CTX *)
/* 2848 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2849 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, unsigned int(*)(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int))
/* 2850 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2851 */ _CFFI_OP(_CFFI_OP_POINTER, 2427), // unsigned int(*)(SSL *, char const *, char *, unsigned int, unsigned char *, unsigned int)
/* 2852 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2853 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, unsigned int(*)(SSL *, char const *, unsigned char *, unsigned int))
/* 2854 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2855 */ _CFFI_OP(_CFFI_OP_POINTER, 2435), // unsigned int(*)(SSL *, char const *, unsigned char *, unsigned int)
/* 2856 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2857 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, void *)
/* 2858 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2859 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2860 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2861 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, void(*)(SSL const *, char const *))
/* 2862 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2863 */ _CFFI_OP(_CFFI_OP_POINTER, 2788), // void(*)(SSL const *, char const *)
/* 2864 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2865 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_CTX *, void(*)(SSL const *, int, int))
/* 2866 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 2867 */ _CFFI_OP(_CFFI_OP_POINTER, 2792), // void(*)(SSL const *, int, int)
/* 2868 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2869 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(SSL_SESSION *)
/* 2870 */ _CFFI_OP(_CFFI_OP_NOOP, 1666),
/* 2871 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2872 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(USERNOTICE *)
/* 2873 */ _CFFI_OP(_CFFI_OP_POINTER, 3118), // USERNOTICE *
/* 2874 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2875 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509 *)
/* 2876 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2877 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2878 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509V3_CTX *, X509 *, X509 *, X509_REQ *, X509_CRL *, int)
/* 2879 */ _CFFI_OP(_CFFI_OP_NOOP, 575),
/* 2880 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2881 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2882 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 2883 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 2884 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2885 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2886 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_CRL *)
/* 2887 */ _CFFI_OP(_CFFI_OP_NOOP, 57),
/* 2888 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2889 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_CRL const *, ASN1_OCTET_STRING const * *, X509_ALGOR const * *)
/* 2890 */ _CFFI_OP(_CFFI_OP_NOOP, 1939),
/* 2891 */ _CFFI_OP(_CFFI_OP_NOOP, 1566),
/* 2892 */ _CFFI_OP(_CFFI_OP_NOOP, 2553),
/* 2893 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2894 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_EXTENSION *)
/* 2895 */ _CFFI_OP(_CFFI_OP_NOOP, 40),
/* 2896 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2897 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_EXTENSIONS *)
/* 2898 */ _CFFI_OP(_CFFI_OP_NOOP, 615),
/* 2899 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2900 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_EXTENSIONS *, void(*)(X509_EXTENSION *))
/* 2901 */ _CFFI_OP(_CFFI_OP_NOOP, 615),
/* 2902 */ _CFFI_OP(_CFFI_OP_POINTER, 2894), // void(*)(X509_EXTENSION *)
/* 2903 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2904 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_NAME *)
/* 2905 */ _CFFI_OP(_CFFI_OP_NOOP, 643),
/* 2906 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2907 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_NAME_ENTRY *)
/* 2908 */ _CFFI_OP(_CFFI_OP_NOOP, 43),
/* 2909 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2910 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_REQ *)
/* 2911 */ _CFFI_OP(_CFFI_OP_NOOP, 294),
/* 2912 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2913 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_REQ const *, ASN1_OCTET_STRING const * *, X509_ALGOR const * *)
/* 2914 */ _CFFI_OP(_CFFI_OP_NOOP, 551),
/* 2915 */ _CFFI_OP(_CFFI_OP_NOOP, 1566),
/* 2916 */ _CFFI_OP(_CFFI_OP_NOOP, 2553),
/* 2917 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2918 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_REVOKED *)
/* 2919 */ _CFFI_OP(_CFFI_OP_NOOP, 619),
/* 2920 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2921 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_STORE *)
/* 2922 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2923 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2924 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_STORE *, int(*)(X509 * *, X509_STORE_CTX *, X509 *))
/* 2925 */ _CFFI_OP(_CFFI_OP_NOOP, 180),
/* 2926 */ _CFFI_OP(_CFFI_OP_POINTER, 1838), // int(*)(X509 * *, X509_STORE_CTX *, X509 *)
/* 2927 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2928 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_STORE_CTX *)
/* 2929 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2930 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2931 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_STORE_CTX *, Cryptography_STACK_OF_X509 *)
/* 2932 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2933 */ _CFFI_OP(_CFFI_OP_NOOP, 413),
/* 2934 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2935 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_STORE_CTX *, Cryptography_STACK_OF_X509_CRL *)
/* 2936 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2937 */ _CFFI_OP(_CFFI_OP_NOOP, 565),
/* 2938 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2939 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_STORE_CTX *, X509 *)
/* 2940 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2941 */ _CFFI_OP(_CFFI_OP_NOOP, 28),
/* 2942 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2943 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_STORE_CTX *, X509_VERIFY_PARAM *)
/* 2944 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2945 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2946 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2947 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_STORE_CTX *, int(*)(int, X509_STORE_CTX *))
/* 2948 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2949 */ _CFFI_OP(_CFFI_OP_NOOP, 2847),
/* 2950 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2951 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_STORE_CTX *, int)
/* 2952 */ _CFFI_OP(_CFFI_OP_NOOP, 160),
/* 2953 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2954 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2955 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_VERIFY_PARAM *)
/* 2956 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2957 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2958 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_VERIFY_PARAM *, int)
/* 2959 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2960 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2961 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2962 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_VERIFY_PARAM *, time_t)
/* 2963 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2964 */ _CFFI_OP(_CFFI_OP_NOOP, 10),
/* 2965 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2966 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(X509_VERIFY_PARAM *, unsigned int)
/* 2967 */ _CFFI_OP(_CFFI_OP_NOOP, 2085),
/* 2968 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 8),
/* 2969 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2970 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(char const *)
/* 2971 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2972 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2973 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(int, int, int, char const *, int)
/* 2974 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2975 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2976 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2977 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2978 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2979 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2980 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(unsigned long, char *, size_t)
/* 2981 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 10),
/* 2982 */ _CFFI_OP(_CFFI_OP_NOOP, 409),
/* 2983 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 28),
/* 2984 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2985 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(void *)
/* 2986 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2987 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2988 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(void *, char const *, int)
/* 2989 */ _CFFI_OP(_CFFI_OP_NOOP, 193),
/* 2990 */ _CFFI_OP(_CFFI_OP_NOOP, 46),
/* 2991 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2992 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2993 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(void const *, int, double)
/* 2994 */ _CFFI_OP(_CFFI_OP_NOOP, 126),
/* 2995 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 7),
/* 2996 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 14), // double
/* 2997 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 2998 */ _CFFI_OP(_CFFI_OP_FUNCTION, 3142), // void()(void)
/* 2999 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3000 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2863), // void(*()(SSL_CTX *))(SSL const *, char const *)
/* 3001 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 3002 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3003 */ _CFFI_OP(_CFFI_OP_FUNCTION, 2867), // void(*()(SSL_CTX *))(SSL const *, int, int)
/* 3004 */ _CFFI_OP(_CFFI_OP_NOOP, 490),
/* 3005 */ _CFFI_OP(_CFFI_OP_FUNCTION_END, 0),
/* 3006 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 4), // ACCESS_DESCRIPTION
/* 3007 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 5), // AES_KEY
/* 3008 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 6), // ASN1_ENUMERATED
/* 3009 */ _CFFI_OP(_CFFI_OP_POINTER, 3010), // ASN1_GENERALIZEDTIME const *
/* 3010 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 7), // ASN1_GENERALIZEDTIME
/* 3011 */ _CFFI_OP(_CFFI_OP_POINTER, 3012), // ASN1_INTEGER const *
/* 3012 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 8), // ASN1_INTEGER
/* 3013 */ _CFFI_OP(_CFFI_OP_POINTER, 3014), // ASN1_NULL *
/* 3014 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 9), // ASN1_NULL
/* 3015 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 10), // ASN1_OBJECT
/* 3016 */ _CFFI_OP(_CFFI_OP_POINTER, 3017), // ASN1_OCTET_STRING const *
/* 3017 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 115), // ASN1_OCTET_STRING
/* 3018 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 11), // ASN1_TYPE
/* 3019 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 12), // AUTHORITY_KEYID
/* 3020 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 13), // BASIC_CONSTRAINTS
/* 3021 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 14), // BIGNUM
/* 3022 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 15), // BIO
/* 3023 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 16), // BIO_METHOD
/* 3024 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 17), // BN_CTX
/* 3025 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 18), // BN_GENCB
/* 3026 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 19), // BN_MONT_CTX
/* 3027 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 20), // CMAC_CTX
/* 3028 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 21), // CONF
/* 3029 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 22), // CRYPTOGRAPHY_PASSWORD_DATA
/* 3030 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 23), // CRYPTO_EX_dup
/* 3031 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 24), // CRYPTO_EX_free
/* 3032 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 25), // CRYPTO_EX_new
/* 3033 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 26), // Cryptography_LHASH_OF_CONF_VALUE
/* 3034 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 27), // Cryptography_STACK_OF_ACCESS_DESCRIPTION
/* 3035 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 28), // Cryptography_STACK_OF_ASN1_INTEGER
/* 3036 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 29), // Cryptography_STACK_OF_ASN1_OBJECT
/* 3037 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 30), // Cryptography_STACK_OF_DIST_POINT
/* 3038 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 31), // Cryptography_STACK_OF_GENERAL_SUBTREE
/* 3039 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 32), // Cryptography_STACK_OF_POLICYINFO
/* 3040 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 33), // Cryptography_STACK_OF_POLICYQUALINFO
/* 3041 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 34), // Cryptography_STACK_OF_SCT
/* 3042 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 35), // Cryptography_STACK_OF_SSL_CIPHER
/* 3043 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 36), // Cryptography_STACK_OF_X509
/* 3044 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 37), // Cryptography_STACK_OF_X509_CRL
/* 3045 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 38), // Cryptography_STACK_OF_X509_NAME
/* 3046 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 39), // Cryptography_STACK_OF_X509_NAME_ENTRY
/* 3047 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 40), // Cryptography_STACK_OF_X509_OBJECT
/* 3048 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 41), // Cryptography_STACK_OF_X509_REVOKED
/* 3049 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 42), // DH
/* 3050 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 43), // DIST_POINT
/* 3051 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 44), // DIST_POINT_NAME
/* 3052 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 45), // DSA
/* 3053 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 46), // ECDSA_SIG
/* 3054 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 47), // EC_GROUP
/* 3055 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 48), // EC_KEY
/* 3056 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 49), // EC_METHOD
/* 3057 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 50), // EC_POINT
/* 3058 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 51), // EC_builtin_curve
/* 3059 */ _CFFI_OP(_CFFI_OP_POINTER, 3060), // EDIPARTYNAME *
/* 3060 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 52), // EDIPARTYNAME
/* 3061 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 53), // ENGINE
/* 3062 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 54), // EVP_CIPHER
/* 3063 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 55), // EVP_CIPHER_CTX
/* 3064 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 56), // EVP_MD
/* 3065 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 57), // EVP_MD_CTX
/* 3066 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 58), // EVP_PKEY
/* 3067 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 59), // EVP_PKEY_CTX
/* 3068 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 60), // EXTENDED_KEY_USAGE
/* 3069 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 61), // GENERAL_NAME
/* 3070 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 116), // GENERAL_NAMES
/* 3071 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 62), // GENERAL_SUBTREE
/* 3072 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 63), // HMAC_CTX
/* 3073 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 64), // ISSUING_DIST_POINT
/* 3074 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 65), // NAME_CONSTRAINTS
/* 3075 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 66), // NETSCAPE_SPKI
/* 3076 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 67), // NOTICEREF
/* 3077 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 68), // OBJ_NAME
/* 3078 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 69), // OCSP_BASICRESP
/* 3079 */ _CFFI_OP(_CFFI_OP_POINTER, 3080), // OCSP_CERTID const *
/* 3080 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 70), // OCSP_CERTID
/* 3081 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 71), // OCSP_ONEREQ
/* 3082 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 72), // OCSP_REQUEST
/* 3083 */ _CFFI_OP(_CFFI_OP_POINTER, 3084), // OCSP_RESPDATA const *
/* 3084 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 73), // OCSP_RESPDATA
/* 3085 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 74), // OCSP_RESPONSE
/* 3086 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 75), // OCSP_SINGLERESP
/* 3087 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 76), // OTHERNAME
/* 3088 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 77), // PKCS12
/* 3089 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 78), // PKCS7
/* 3090 */ _CFFI_OP(_CFFI_OP_POINTER, 3091), // PKCS7_DIGEST *
/* 3091 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 79), // PKCS7_DIGEST
/* 3092 */ _CFFI_OP(_CFFI_OP_POINTER, 3093), // PKCS7_ENCRYPT *
/* 3093 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 80), // PKCS7_ENCRYPT
/* 3094 */ _CFFI_OP(_CFFI_OP_POINTER, 3095), // PKCS7_ENVELOPE *
/* 3095 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 81), // PKCS7_ENVELOPE
/* 3096 */ _CFFI_OP(_CFFI_OP_POINTER, 3097), // PKCS7_SIGNED *
/* 3097 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 82), // PKCS7_SIGNED
/* 3098 */ _CFFI_OP(_CFFI_OP_POINTER, 3099), // PKCS7_SIGNER_INFO *
/* 3099 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 83), // PKCS7_SIGNER_INFO
/* 3100 */ _CFFI_OP(_CFFI_OP_POINTER, 3101), // PKCS7_SIGN_ENVELOPE *
/* 3101 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 84), // PKCS7_SIGN_ENVELOPE
/* 3102 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 85), // PKCS8_PRIV_KEY_INFO
/* 3103 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 86), // POLICYINFO
/* 3104 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 87), // POLICYQUALINFO
/* 3105 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 88), // POLICY_CONSTRAINTS
/* 3106 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 89), // RAND_METHOD
/* 3107 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 90), // RSA
/* 3108 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 91), // SCT
/* 3109 */ _CFFI_OP(_CFFI_OP_POINTER, 3110), // SRTP_PROTECTION_PROFILE *
/* 3110 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 92), // SRTP_PROTECTION_PROFILE
/* 3111 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 93), // SSL
/* 3112 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 94), // SSL_CIPHER
/* 3113 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 95), // SSL_CTX
/* 3114 */ _CFFI_OP(_CFFI_OP_POINTER, 3115), // SSL_METHOD const *
/* 3115 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 96), // SSL_METHOD
/* 3116 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 97), // SSL_SESSION
/* 3117 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 98), // UI_METHOD
/* 3118 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 99), // USERNOTICE
/* 3119 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 100), // X509
/* 3120 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 101), // X509V3_CTX
/* 3121 */ _CFFI_OP(_CFFI_OP_POINTER, 3122), // X509_ALGOR const *
/* 3122 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 102), // X509_ALGOR
/* 3123 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 103), // X509_ATTRIBUTE
/* 3124 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 104), // X509_CRL
/* 3125 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 105), // X509_EXTENSION
/* 3126 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 106), // X509_EXTENSIONS
/* 3127 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 107), // X509_NAME
/* 3128 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 108), // X509_NAME_ENTRY
/* 3129 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 109), // X509_OBJECT
/* 3130 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 110), // X509_REQ
/* 3131 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 111), // X509_REVOKED
/* 3132 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 112), // X509_STORE
/* 3133 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 113), // X509_STORE_CTX
/* 3134 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 114), // X509_VERIFY_PARAM
/* 3135 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 2), // char
/* 3136 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 0), // union $1
/* 3137 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 1), // union $2
/* 3138 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 2), // union $3
/* 3139 */ _CFFI_OP(_CFFI_OP_STRUCT_UNION, 3), // union $4
/* 3140 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 4), // unsigned char
/* 3141 */ _CFFI_OP(_CFFI_OP_POINTER, 2522), // void *(*)(void *, unsigned char const * *, long)
/* 3142 */ _CFFI_OP(_CFFI_OP_PRIMITIVE, 0), // void
};

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_ACCESS_DESCRIPTION(ACCESS_DESCRIPTION *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->method; (void)tmp; }
  { GENERAL_NAME * *tmp = &p->location; (void)tmp; }
}
struct _cffi_align_typedef_ACCESS_DESCRIPTION { char x; ACCESS_DESCRIPTION y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_ASN1_TYPE(ASN1_TYPE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->type) | 0);  /* check that 'ASN1_TYPE.type' is an integer */
}
struct _cffi_align_typedef_ASN1_TYPE { char x; ASN1_TYPE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_AUTHORITY_KEYID(AUTHORITY_KEYID *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OCTET_STRING * *tmp = &p->keyid; (void)tmp; }
  { GENERAL_NAMES * *tmp = &p->issuer; (void)tmp; }
  { ASN1_INTEGER * *tmp = &p->serial; (void)tmp; }
}
struct _cffi_align_typedef_AUTHORITY_KEYID { char x; AUTHORITY_KEYID y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_BASIC_CONSTRAINTS(BASIC_CONSTRAINTS *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->ca) | 0);  /* check that 'BASIC_CONSTRAINTS.ca' is an integer */
  { ASN1_INTEGER * *tmp = &p->pathlen; (void)tmp; }
}
struct _cffi_align_typedef_BASIC_CONSTRAINTS { char x; BASIC_CONSTRAINTS y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_CRYPTOGRAPHY_PASSWORD_DATA(CRYPTOGRAPHY_PASSWORD_DATA *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { char * *tmp = &p->password; (void)tmp; }
  (void)((p->length) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.length' is an integer */
  (void)((p->called) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.called' is an integer */
  (void)((p->error) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.error' is an integer */
  (void)((p->maxsize) | 0);  /* check that 'CRYPTOGRAPHY_PASSWORD_DATA.maxsize' is an integer */
}
struct _cffi_align_typedef_CRYPTOGRAPHY_PASSWORD_DATA { char x; CRYPTOGRAPHY_PASSWORD_DATA y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_DIST_POINT(DIST_POINT *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { DIST_POINT_NAME * *tmp = &p->distpoint; (void)tmp; }
  { ASN1_OCTET_STRING * *tmp = &p->reasons; (void)tmp; }
  { GENERAL_NAMES * *tmp = &p->CRLissuer; (void)tmp; }
}
struct _cffi_align_typedef_DIST_POINT { char x; DIST_POINT y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_DIST_POINT_NAME(DIST_POINT_NAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->type) | 0);  /* check that 'DIST_POINT_NAME.type' is an integer */
  /* cannot generate 'union $2' in field 'name': unknown type name */
}
struct _cffi_align_typedef_DIST_POINT_NAME { char x; DIST_POINT_NAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_EC_builtin_curve(EC_builtin_curve *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->nid) | 0);  /* check that 'EC_builtin_curve.nid' is an integer */
  { char const * *tmp = &p->comment; (void)tmp; }
}
struct _cffi_align_typedef_EC_builtin_curve { char x; EC_builtin_curve y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_EDIPARTYNAME(EDIPARTYNAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
}
struct _cffi_align_typedef_EDIPARTYNAME { char x; EDIPARTYNAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_GENERAL_NAME(GENERAL_NAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->type) | 0);  /* check that 'GENERAL_NAME.type' is an integer */
  /* cannot generate 'union $1' in field 'd': unknown type name */
}
struct _cffi_align_typedef_GENERAL_NAME { char x; GENERAL_NAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_GENERAL_SUBTREE(GENERAL_SUBTREE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { GENERAL_NAME * *tmp = &p->base; (void)tmp; }
  { ASN1_INTEGER * *tmp = &p->minimum; (void)tmp; }
  { ASN1_INTEGER * *tmp = &p->maximum; (void)tmp; }
}
struct _cffi_align_typedef_GENERAL_SUBTREE { char x; GENERAL_SUBTREE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_ISSUING_DIST_POINT(ISSUING_DIST_POINT *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { DIST_POINT_NAME * *tmp = &p->distpoint; (void)tmp; }
  (void)((p->onlyuser) | 0);  /* check that 'ISSUING_DIST_POINT.onlyuser' is an integer */
  (void)((p->onlyCA) | 0);  /* check that 'ISSUING_DIST_POINT.onlyCA' is an integer */
  { ASN1_OCTET_STRING * *tmp = &p->onlysomereasons; (void)tmp; }
  (void)((p->indirectCRL) | 0);  /* check that 'ISSUING_DIST_POINT.indirectCRL' is an integer */
  (void)((p->onlyattr) | 0);  /* check that 'ISSUING_DIST_POINT.onlyattr' is an integer */
}
struct _cffi_align_typedef_ISSUING_DIST_POINT { char x; ISSUING_DIST_POINT y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_NAME_CONSTRAINTS(NAME_CONSTRAINTS *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { Cryptography_STACK_OF_GENERAL_SUBTREE * *tmp = &p->permittedSubtrees; (void)tmp; }
  { Cryptography_STACK_OF_GENERAL_SUBTREE * *tmp = &p->excludedSubtrees; (void)tmp; }
}
struct _cffi_align_typedef_NAME_CONSTRAINTS { char x; NAME_CONSTRAINTS y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_NOTICEREF(NOTICEREF *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OCTET_STRING * *tmp = &p->organization; (void)tmp; }
  { Cryptography_STACK_OF_ASN1_INTEGER * *tmp = &p->noticenos; (void)tmp; }
}
struct _cffi_align_typedef_NOTICEREF { char x; NOTICEREF y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_OBJ_NAME(OBJ_NAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  (void)((p->type) | 0);  /* check that 'OBJ_NAME.type' is an integer */
  (void)((p->alias) | 0);  /* check that 'OBJ_NAME.alias' is an integer */
  { char const * *tmp = &p->name; (void)tmp; }
  { char const * *tmp = &p->data; (void)tmp; }
}
struct _cffi_align_typedef_OBJ_NAME { char x; OBJ_NAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_OTHERNAME(OTHERNAME *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->type_id; (void)tmp; }
  { ASN1_TYPE * *tmp = &p->value; (void)tmp; }
}
struct _cffi_align_typedef_OTHERNAME { char x; OTHERNAME y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_PKCS7(PKCS7 *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->type; (void)tmp; }
  /* cannot generate 'union $4' in field 'd': unknown type name */
}
struct _cffi_align_typedef_PKCS7 { char x; PKCS7 y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_PKCS7_SIGNED(PKCS7_SIGNED *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { Cryptography_STACK_OF_X509 * *tmp = &p->cert; (void)tmp; }
  { Cryptography_STACK_OF_X509_CRL * *tmp = &p->crl; (void)tmp; }
}
struct _cffi_align_typedef_PKCS7_SIGNED { char x; PKCS7_SIGNED y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_PKCS7_SIGN_ENVELOPE(PKCS7_SIGN_ENVELOPE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { Cryptography_STACK_OF_X509 * *tmp = &p->cert; (void)tmp; }
  { Cryptography_STACK_OF_X509_CRL * *tmp = &p->crl; (void)tmp; }
}
struct _cffi_align_typedef_PKCS7_SIGN_ENVELOPE { char x; PKCS7_SIGN_ENVELOPE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_POLICYINFO(POLICYINFO *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->policyid; (void)tmp; }
  { Cryptography_STACK_OF_POLICYQUALINFO * *tmp = &p->qualifiers; (void)tmp; }
}
struct _cffi_align_typedef_POLICYINFO { char x; POLICYINFO y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_POLICYQUALINFO(POLICYQUALINFO *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->pqualid; (void)tmp; }
  /* cannot generate 'union $3' in field 'd': unknown type name */
}
struct _cffi_align_typedef_POLICYQUALINFO { char x; POLICYQUALINFO y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_POLICY_CONSTRAINTS(POLICY_CONSTRAINTS *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_INTEGER * *tmp = &p->requireExplicitPolicy; (void)tmp; }
  { ASN1_INTEGER * *tmp = &p->inhibitPolicyMapping; (void)tmp; }
}
struct _cffi_align_typedef_POLICY_CONSTRAINTS { char x; POLICY_CONSTRAINTS y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_SRTP_PROTECTION_PROFILE(SRTP_PROTECTION_PROFILE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { char const * *tmp = &p->name; (void)tmp; }
  (void)((p->id) | 0);  /* check that 'SRTP_PROTECTION_PROFILE.id' is an integer */
}
struct _cffi_align_typedef_SRTP_PROTECTION_PROFILE { char x; SRTP_PROTECTION_PROFILE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_USERNOTICE(USERNOTICE *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { NOTICEREF * *tmp = &p->noticeref; (void)tmp; }
  { ASN1_OCTET_STRING * *tmp = &p->exptext; (void)tmp; }
}
struct _cffi_align_typedef_USERNOTICE { char x; USERNOTICE y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_X509V3_CTX(X509V3_CTX *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { X509 * *tmp = &p->issuer_cert; (void)tmp; }
  { X509 * *tmp = &p->subject_cert; (void)tmp; }
}
struct _cffi_align_typedef_X509V3_CTX { char x; X509V3_CTX y; };

_CFFI_UNUSED_FN
static void _cffi_checkfld_typedef_X509_ALGOR(X509_ALGOR *p)
{
  /* only to generate compile-time warnings or errors */
  (void)p;
  { ASN1_OBJECT * *tmp = &p->algorithm; (void)tmp; }
}
struct _cffi_align_typedef_X509_ALGOR { char x; X509_ALGOR y; };

static int _cffi_const_CT_LOG_ENTRY_TYPE_NOT_SET(unsigned long long *o)
{
  int n = (CT_LOG_ENTRY_TYPE_NOT_SET) <= 0;
  *o = (unsigned long long)((CT_LOG_ENTRY_TYPE_NOT_SET) | 0);  /* check that CT_LOG_ENTRY_TYPE_NOT_SET is an integer */
  return n;
}

static int _cffi_const_CT_LOG_ENTRY_TYPE_X509(unsigned long long *o)
{
  int n = (CT_LOG_ENTRY_TYPE_X509) <= 0;
  *o = (unsigned long long)((CT_LOG_ENTRY_TYPE_X509) | 0);  /* check that CT_LOG_ENTRY_TYPE_X509 is an integer */
  return n;
}

static int _cffi_const_CT_LOG_ENTRY_TYPE_PRECERT(unsigned long long *o)
{
  int n = (CT_LOG_ENTRY_TYPE_PRECERT) <= 0;
  *o = (unsigned long long)((CT_LOG_ENTRY_TYPE_PRECERT) | 0);  /* check that CT_LOG_ENTRY_TYPE_PRECERT is an integer */
  return n;
}

static int _cffi_const_POINT_CONVERSION_COMPRESSED(unsigned long long *o)
{
  int n = (POINT_CONVERSION_COMPRESSED) <= 0;
  *o = (unsigned long long)((POINT_CONVERSION_COMPRESSED) | 0);  /* check that POINT_CONVERSION_COMPRESSED is an integer */
  return n;
}

static int _cffi_const_POINT_CONVERSION_UNCOMPRESSED(unsigned long long *o)
{
  int n = (POINT_CONVERSION_UNCOMPRESSED) <= 0;
  *o = (unsigned long long)((POINT_CONVERSION_UNCOMPRESSED) | 0);  /* check that POINT_CONVERSION_UNCOMPRESSED is an integer */
  return n;
}

static int _cffi_const_SCT_SOURCE_UNKNOWN(unsigned long long *o)
{
  int n = (SCT_SOURCE_UNKNOWN) <= 0;
  *o = (unsigned long long)((SCT_SOURCE_UNKNOWN) | 0);  /* check that SCT_SOURCE_UNKNOWN is an integer */
  return n;
}

static int _cffi_const_SCT_SOURCE_TLS_EXTENSION(unsigned long long *o)
{
  int n = (SCT_SOURCE_TLS_EXTENSION) <= 0;
  *o = (unsigned long long)((SCT_SOURCE_TLS_EXTENSION) | 0);  /* check that SCT_SOURCE_TLS_EXTENSION is an integer */
  return n;
}

static int _cffi_const_SCT_SOURCE_X509V3_EXTENSION(unsigned long long *o)
{
  int n = (SCT_SOURCE_X509V3_EXTENSION) <= 0;
  *o = (unsigned long long)((SCT_SOURCE_X509V3_EXTENSION) | 0);  /* check that SCT_SOURCE_X509V3_EXTENSION is an integer */
  return n;
}

static int _cffi_const_SCT_SOURCE_OCSP_STAPLED_RESPONSE(unsigned long long *o)
{
  int n = (SCT_SOURCE_OCSP_STAPLED_RESPONSE) <= 0;
  *o = (unsigned long long)((SCT_SOURCE_OCSP_STAPLED_RESPONSE) | 0);  /* check that SCT_SOURCE_OCSP_STAPLED_RESPONSE is an integer */
  return n;
}

static int _cffi_const_SCT_VERSION_NOT_SET(unsigned long long *o)
{
  int n = (SCT_VERSION_NOT_SET) <= 0;
  *o = (unsigned long long)((SCT_VERSION_NOT_SET) | 0);  /* check that SCT_VERSION_NOT_SET is an integer */
  return n;
}

static int _cffi_const_SCT_VERSION_V1(unsigned long long *o)
{
  int n = (SCT_VERSION_V1) <= 0;
  *o = (unsigned long long)((SCT_VERSION_V1) | 0);  /* check that SCT_VERSION_V1 is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_IS_LIBRESSL(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_IS_LIBRESSL) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_IS_LIBRESSL) | 0);  /* check that CRYPTOGRAPHY_IS_LIBRESSL is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE) | 0);  /* check that CRYPTOGRAPHY_NEEDS_OSRANDOM_ENGINE is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_110F_OR_GREATER is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_LESS_THAN_111(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_LESS_THAN_111) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_LESS_THAN_111 is an integer */
  return n;
}

static int _cffi_const_CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B(unsigned long long *o)
{
  int n = (CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B) <= 0;
  *o = (unsigned long long)((CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B) | 0);  /* check that CRYPTOGRAPHY_OPENSSL_LESS_THAN_111B is an integer */
  return n;
}

static void _cffi_const_CRYPTOGRAPHY_PACKAGE_VERSION(char *o)
{
  *(char const * *)o = CRYPTOGRAPHY_PACKAGE_VERSION;
}

static int _cffi_const_Cryptography_HAS_102_VERIFICATION(unsigned long long *o)
{
  int n = (Cryptography_HAS_102_VERIFICATION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_102_VERIFICATION) | 0);  /* check that Cryptography_HAS_102_VERIFICATION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_110_VERIFICATION_PARAMS(unsigned long long *o)
{
  int n = (Cryptography_HAS_110_VERIFICATION_PARAMS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_110_VERIFICATION_PARAMS) | 0);  /* check that Cryptography_HAS_110_VERIFICATION_PARAMS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ALPN(unsigned long long *o)
{
  int n = (Cryptography_HAS_ALPN) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ALPN) | 0);  /* check that Cryptography_HAS_ALPN is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_CUSTOM_EXT(unsigned long long *o)
{
  int n = (Cryptography_HAS_CUSTOM_EXT) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_CUSTOM_EXT) | 0);  /* check that Cryptography_HAS_CUSTOM_EXT is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_DTLS(unsigned long long *o)
{
  int n = (Cryptography_HAS_DTLS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_DTLS) | 0);  /* check that Cryptography_HAS_DTLS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EC2M(unsigned long long *o)
{
  int n = (Cryptography_HAS_EC2M) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EC2M) | 0);  /* check that Cryptography_HAS_EC2M is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ED25519(unsigned long long *o)
{
  int n = (Cryptography_HAS_ED25519) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ED25519) | 0);  /* check that Cryptography_HAS_ED25519 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ED448(unsigned long long *o)
{
  int n = (Cryptography_HAS_ED448) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ED448) | 0);  /* check that Cryptography_HAS_ED448 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ENGINE(unsigned long long *o)
{
  int n = (Cryptography_HAS_ENGINE) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ENGINE) | 0);  /* check that Cryptography_HAS_ENGINE is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_DIGESTFINAL_XOF(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_DIGESTFINAL_XOF) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_DIGESTFINAL_XOF) | 0);  /* check that Cryptography_HAS_EVP_DIGESTFINAL_XOF is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_PKEY_DHX(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_PKEY_DHX) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_PKEY_DHX) | 0);  /* check that Cryptography_HAS_EVP_PKEY_DHX is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint(unsigned long long *o)
{
  int n = (Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint) | 0);  /* check that Cryptography_HAS_EVP_PKEY_get_set_tls_encodedpoint is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_FIPS(unsigned long long *o)
{
  int n = (Cryptography_HAS_FIPS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_FIPS) | 0);  /* check that Cryptography_HAS_FIPS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_GET_PROTO_VERSION(unsigned long long *o)
{
  int n = (Cryptography_HAS_GET_PROTO_VERSION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_GET_PROTO_VERSION) | 0);  /* check that Cryptography_HAS_GET_PROTO_VERSION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_KEYLOG(unsigned long long *o)
{
  int n = (Cryptography_HAS_KEYLOG) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_KEYLOG) | 0);  /* check that Cryptography_HAS_KEYLOG is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_MEM_FUNCTIONS(unsigned long long *o)
{
  int n = (Cryptography_HAS_MEM_FUNCTIONS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_MEM_FUNCTIONS) | 0);  /* check that Cryptography_HAS_MEM_FUNCTIONS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_NEXTPROTONEG(unsigned long long *o)
{
  int n = (Cryptography_HAS_NEXTPROTONEG) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_NEXTPROTONEG) | 0);  /* check that Cryptography_HAS_NEXTPROTONEG is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY(unsigned long long *o)
{
  int n = (Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY) | 0);  /* check that Cryptography_HAS_ONESHOT_EVP_DIGEST_SIGN_VERIFY is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_OPENSSL_CLEANUP(unsigned long long *o)
{
  int n = (Cryptography_HAS_OPENSSL_CLEANUP) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_OPENSSL_CLEANUP) | 0);  /* check that Cryptography_HAS_OPENSSL_CLEANUP is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_OP_NO_COMPRESSION(unsigned long long *o)
{
  int n = (Cryptography_HAS_OP_NO_COMPRESSION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_OP_NO_COMPRESSION) | 0);  /* check that Cryptography_HAS_OP_NO_COMPRESSION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_POLY1305(unsigned long long *o)
{
  int n = (Cryptography_HAS_POLY1305) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_POLY1305) | 0);  /* check that Cryptography_HAS_POLY1305 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_PSK(unsigned long long *o)
{
  int n = (Cryptography_HAS_PSK) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_PSK) | 0);  /* check that Cryptography_HAS_PSK is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RAW_KEY(unsigned long long *o)
{
  int n = (Cryptography_HAS_RAW_KEY) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RAW_KEY) | 0);  /* check that Cryptography_HAS_RAW_KEY is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RELEASE_BUFFERS(unsigned long long *o)
{
  int n = (Cryptography_HAS_RELEASE_BUFFERS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RELEASE_BUFFERS) | 0);  /* check that Cryptography_HAS_RELEASE_BUFFERS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RSA_OAEP_LABEL(unsigned long long *o)
{
  int n = (Cryptography_HAS_RSA_OAEP_LABEL) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RSA_OAEP_LABEL) | 0);  /* check that Cryptography_HAS_RSA_OAEP_LABEL is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_RSA_OAEP_MD(unsigned long long *o)
{
  int n = (Cryptography_HAS_RSA_OAEP_MD) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_RSA_OAEP_MD) | 0);  /* check that Cryptography_HAS_RSA_OAEP_MD is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SCRYPT(unsigned long long *o)
{
  int n = (Cryptography_HAS_SCRYPT) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SCRYPT) | 0);  /* check that Cryptography_HAS_SCRYPT is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SCT(unsigned long long *o)
{
  int n = (Cryptography_HAS_SCT) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SCT) | 0);  /* check that Cryptography_HAS_SCT is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SECURE_RENEGOTIATION(unsigned long long *o)
{
  int n = (Cryptography_HAS_SECURE_RENEGOTIATION) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SECURE_RENEGOTIATION) | 0);  /* check that Cryptography_HAS_SECURE_RENEGOTIATION is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SET_CERT_CB(unsigned long long *o)
{
  int n = (Cryptography_HAS_SET_CERT_CB) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SET_CERT_CB) | 0);  /* check that Cryptography_HAS_SET_CERT_CB is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SIGALGS(unsigned long long *o)
{
  int n = (Cryptography_HAS_SIGALGS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SIGALGS) | 0);  /* check that Cryptography_HAS_SIGALGS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SRTP(unsigned long long *o)
{
  int n = (Cryptography_HAS_SRTP) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SRTP) | 0);  /* check that Cryptography_HAS_SRTP is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL3_METHOD(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL3_METHOD) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL3_METHOD) | 0);  /* check that Cryptography_HAS_SSL3_METHOD is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS) | 0);  /* check that Cryptography_HAS_SSL_CTX_CLEAR_OPTIONS is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING) | 0);  /* check that Cryptography_HAS_SSL_OP_MSIE_SSLV2_RSA_PADDING is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_OP_NO_TICKET(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_OP_NO_TICKET) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_OP_NO_TICKET) | 0);  /* check that Cryptography_HAS_SSL_OP_NO_TICKET is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_SET_SSL_CTX(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_SET_SSL_CTX) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_SET_SSL_CTX) | 0);  /* check that Cryptography_HAS_SSL_SET_SSL_CTX is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_SSL_ST(unsigned long long *o)
{
  int n = (Cryptography_HAS_SSL_ST) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_SSL_ST) | 0);  /* check that Cryptography_HAS_SSL_ST is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSEXT_HOSTNAME(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSEXT_HOSTNAME) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSEXT_HOSTNAME) | 0);  /* check that Cryptography_HAS_TLSEXT_HOSTNAME is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLS_ST(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLS_ST) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLS_ST) | 0);  /* check that Cryptography_HAS_TLS_ST is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSv1_1(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSv1_1) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSv1_1) | 0);  /* check that Cryptography_HAS_TLSv1_1 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSv1_2(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSv1_2) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSv1_2) | 0);  /* check that Cryptography_HAS_TLSv1_2 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_TLSv1_3(unsigned long long *o)
{
  int n = (Cryptography_HAS_TLSv1_3) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_TLSv1_3) | 0);  /* check that Cryptography_HAS_TLSv1_3 is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_VERIFIED_CHAIN(unsigned long long *o)
{
  int n = (Cryptography_HAS_VERIFIED_CHAIN) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_VERIFIED_CHAIN) | 0);  /* check that Cryptography_HAS_VERIFIED_CHAIN is an integer */
  return n;
}

static int _cffi_const_Cryptography_HAS_X509_STORE_CTX_GET_ISSUER(unsigned long long *o)
{
  int n = (Cryptography_HAS_X509_STORE_CTX_GET_ISSUER) <= 0;
  *o = (unsigned long long)((Cryptography_HAS_X509_STORE_CTX_GET_ISSUER) | 0);  /* check that Cryptography_HAS_X509_STORE_CTX_GET_ISSUER is an integer */
  return n;
}

static void _cffi_const_Cryptography_osrandom_engine_id(char *o)
{
  *(char const * *)o = Cryptography_osrandom_engine_id;
}

static void _cffi_const_Cryptography_osrandom_engine_name(char *o)
{
  *(char const * *)o = Cryptography_osrandom_engine_name;
}

static int _cffi_const_DH_NOT_SUITABLE_GENERATOR(unsigned long long *o)
{
  int n = (DH_NOT_SUITABLE_GENERATOR) <= 0;
  *o = (unsigned long long)((DH_NOT_SUITABLE_GENERATOR) | 0);  /* check that DH_NOT_SUITABLE_GENERATOR is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_ASN1(unsigned long long *o)
{
  int n = (ERR_LIB_ASN1) <= 0;
  *o = (unsigned long long)((ERR_LIB_ASN1) | 0);  /* check that ERR_LIB_ASN1 is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_EVP(unsigned long long *o)
{
  int n = (ERR_LIB_EVP) <= 0;
  *o = (unsigned long long)((ERR_LIB_EVP) | 0);  /* check that ERR_LIB_EVP is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_PEM(unsigned long long *o)
{
  int n = (ERR_LIB_PEM) <= 0;
  *o = (unsigned long long)((ERR_LIB_PEM) | 0);  /* check that ERR_LIB_PEM is an integer */
  return n;
}

static int _cffi_const_ERR_LIB_PKCS12(unsigned long long *o)
{
  int n = (ERR_LIB_PKCS12) <= 0;
  *o = (unsigned long long)((ERR_LIB_PKCS12) | 0);  /* check that ERR_LIB_PKCS12 is an integer */
  return n;
}

static int _cffi_const_EVP_CTRL_AEAD_GET_TAG(unsigned long long *o)
{
  int n = (EVP_CTRL_AEAD_GET_TAG) <= 0;
  *o = (unsigned long long)((EVP_CTRL_AEAD_GET_TAG) | 0);  /* check that EVP_CTRL_AEAD_GET_TAG is an integer */
  return n;
}

static int _cffi_const_EVP_CTRL_AEAD_SET_IVLEN(unsigned long long *o)
{
  int n = (EVP_CTRL_AEAD_SET_IVLEN) <= 0;
  *o = (unsigned long long)((EVP_CTRL_AEAD_SET_IVLEN) | 0);  /* check that EVP_CTRL_AEAD_SET_IVLEN is an integer */
  return n;
}

static int _cffi_const_EVP_CTRL_AEAD_SET_TAG(unsigned long long *o)
{
  int n = (EVP_CTRL_AEAD_SET_TAG) <= 0;
  *o = (unsigned long long)((EVP_CTRL_AEAD_SET_TAG) | 0);  /* check that EVP_CTRL_AEAD_SET_TAG is an integer */
  return n;
}

static int _cffi_const_EVP_F_EVP_ENCRYPTFINAL_EX(unsigned long long *o)
{
  int n = (EVP_F_EVP_ENCRYPTFINAL_EX) <= 0;
  *o = (unsigned long long)((EVP_F_EVP_ENCRYPTFINAL_EX) | 0);  /* check that EVP_F_EVP_ENCRYPTFINAL_EX is an integer */
  return n;
}

static int _cffi_const_EVP_MAX_MD_SIZE(unsigned long long *o)
{
  int n = (EVP_MAX_MD_SIZE) <= 0;
  *o = (unsigned long long)((EVP_MAX_MD_SIZE) | 0);  /* check that EVP_MAX_MD_SIZE is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_DH(unsigned long long *o)
{
  int n = (EVP_PKEY_DH) <= 0;
  *o = (unsigned long long)((EVP_PKEY_DH) | 0);  /* check that EVP_PKEY_DH is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_DHX(unsigned long long *o)
{
  int n = (EVP_PKEY_DHX) <= 0;
  *o = (unsigned long long)((EVP_PKEY_DHX) | 0);  /* check that EVP_PKEY_DHX is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_DSA(unsigned long long *o)
{
  int n = (EVP_PKEY_DSA) <= 0;
  *o = (unsigned long long)((EVP_PKEY_DSA) | 0);  /* check that EVP_PKEY_DSA is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_EC(unsigned long long *o)
{
  int n = (EVP_PKEY_EC) <= 0;
  *o = (unsigned long long)((EVP_PKEY_EC) | 0);  /* check that EVP_PKEY_EC is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_ED25519(unsigned long long *o)
{
  int n = (EVP_PKEY_ED25519) <= 0;
  *o = (unsigned long long)((EVP_PKEY_ED25519) | 0);  /* check that EVP_PKEY_ED25519 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_ED448(unsigned long long *o)
{
  int n = (EVP_PKEY_ED448) <= 0;
  *o = (unsigned long long)((EVP_PKEY_ED448) | 0);  /* check that EVP_PKEY_ED448 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_POLY1305(unsigned long long *o)
{
  int n = (EVP_PKEY_POLY1305) <= 0;
  *o = (unsigned long long)((EVP_PKEY_POLY1305) | 0);  /* check that EVP_PKEY_POLY1305 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_RSA(unsigned long long *o)
{
  int n = (EVP_PKEY_RSA) <= 0;
  *o = (unsigned long long)((EVP_PKEY_RSA) | 0);  /* check that EVP_PKEY_RSA is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_X25519(unsigned long long *o)
{
  int n = (EVP_PKEY_X25519) <= 0;
  *o = (unsigned long long)((EVP_PKEY_X25519) | 0);  /* check that EVP_PKEY_X25519 is an integer */
  return n;
}

static int _cffi_const_EVP_PKEY_X448(unsigned long long *o)
{
  int n = (EVP_PKEY_X448) <= 0;
  *o = (unsigned long long)((EVP_PKEY_X448) | 0);  /* check that EVP_PKEY_X448 is an integer */
  return n;
}

static int _cffi_const_EVP_R_BAD_DECRYPT(unsigned long long *o)
{
  int n = (EVP_R_BAD_DECRYPT) <= 0;
  *o = (unsigned long long)((EVP_R_BAD_DECRYPT) | 0);  /* check that EVP_R_BAD_DECRYPT is an integer */
  return n;
}

static int _cffi_const_EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH(unsigned long long *o)
{
  int n = (EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH) <= 0;
  *o = (unsigned long long)((EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH) | 0);  /* check that EVP_R_DATA_NOT_MULTIPLE_OF_BLOCK_LENGTH is an integer */
  return n;
}

static int _cffi_const_EVP_R_UNKNOWN_PBE_ALGORITHM(unsigned long long *o)
{
  int n = (EVP_R_UNKNOWN_PBE_ALGORITHM) <= 0;
  *o = (unsigned long long)((EVP_R_UNKNOWN_PBE_ALGORITHM) | 0);  /* check that EVP_R_UNKNOWN_PBE_ALGORITHM is an integer */
  return n;
}

static int _cffi_const_EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM(unsigned long long *o)
{
  int n = (EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM) <= 0;
  *o = (unsigned long long)((EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM) | 0);  /* check that EVP_R_UNSUPPORTED_PRIVATE_KEY_ALGORITHM is an integer */
  return n;
}

static int _cffi_const_GEN_DIRNAME(unsigned long long *o)
{
  int n = (GEN_DIRNAME) <= 0;
  *o = (unsigned long long)((GEN_DIRNAME) | 0);  /* check that GEN_DIRNAME is an integer */
  return n;
}

static int _cffi_const_GEN_DNS(unsigned long long *o)
{
  int n = (GEN_DNS) <= 0;
  *o = (unsigned long long)((GEN_DNS) | 0);  /* check that GEN_DNS is an integer */
  return n;
}

static int _cffi_const_GEN_EDIPARTY(unsigned long long *o)
{
  int n = (GEN_EDIPARTY) <= 0;
  *o = (unsigned long long)((GEN_EDIPARTY) | 0);  /* check that GEN_EDIPARTY is an integer */
  return n;
}

static int _cffi_const_GEN_EMAIL(unsigned long long *o)
{
  int n = (GEN_EMAIL) <= 0;
  *o = (unsigned long long)((GEN_EMAIL) | 0);  /* check that GEN_EMAIL is an integer */
  return n;
}

static int _cffi_const_GEN_IPADD(unsigned long long *o)
{
  int n = (GEN_IPADD) <= 0;
  *o = (unsigned long long)((GEN_IPADD) | 0);  /* check that GEN_IPADD is an integer */
  return n;
}

static int _cffi_const_GEN_OTHERNAME(unsigned long long *o)
{
  int n = (GEN_OTHERNAME) <= 0;
  *o = (unsigned long long)((GEN_OTHERNAME) | 0);  /* check that GEN_OTHERNAME is an integer */
  return n;
}

static int _cffi_const_GEN_RID(unsigned long long *o)
{
  int n = (GEN_RID) <= 0;
  *o = (unsigned long long)((GEN_RID) | 0);  /* check that GEN_RID is an integer */
  return n;
}

static int _cffi_const_GEN_URI(unsigned long long *o)
{
  int n = (GEN_URI) <= 0;
  *o = (unsigned long long)((GEN_URI) | 0);  /* check that GEN_URI is an integer */
  return n;
}

static int _cffi_const_GEN_X400(unsigned long long *o)
{
  int n = (GEN_X400) <= 0;
  *o = (unsigned long long)((GEN_X400) | 0);  /* check that GEN_X400 is an integer */
  return n;
}

static int _cffi_const_MBSTRING_UTF8(unsigned long long *o)
{
  int n = (MBSTRING_UTF8) <= 0;
  *o = (unsigned long long)((MBSTRING_UTF8) | 0);  /* check that MBSTRING_UTF8 is an integer */
  return n;
}

static int _cffi_const_NID_ED25519(unsigned long long *o)
{
  int n = (NID_ED25519) <= 0;
  *o = (unsigned long long)((NID_ED25519) | 0);  /* check that NID_ED25519 is an integer */
  return n;
}

static int _cffi_const_NID_ED448(unsigned long long *o)
{
  int n = (NID_ED448) <= 0;
  *o = (unsigned long long)((NID_ED448) | 0);  /* check that NID_ED448 is an integer */
  return n;
}

static int _cffi_const_NID_X25519(unsigned long long *o)
{
  int n = (NID_X25519) <= 0;
  *o = (unsigned long long)((NID_X25519) | 0);  /* check that NID_X25519 is an integer */
  return n;
}

static int _cffi_const_NID_X448(unsigned long long *o)
{
  int n = (NID_X448) <= 0;
  *o = (unsigned long long)((NID_X448) | 0);  /* check that NID_X448 is an integer */
  return n;
}

static int _cffi_const_NID_crl_reason(unsigned long long *o)
{
  int n = (NID_crl_reason) <= 0;
  *o = (unsigned long long)((NID_crl_reason) | 0);  /* check that NID_crl_reason is an integer */
  return n;
}

static int _cffi_const_NID_pbe_WithSHA1And3_Key_TripleDES_CBC(unsigned long long *o)
{
  int n = (NID_pbe_WithSHA1And3_Key_TripleDES_CBC) <= 0;
  *o = (unsigned long long)((NID_pbe_WithSHA1And3_Key_TripleDES_CBC) | 0);  /* check that NID_pbe_WithSHA1And3_Key_TripleDES_CBC is an integer */
  return n;
}

static int _cffi_const_NID_pkcs7_signed(unsigned long long *o)
{
  int n = (NID_pkcs7_signed) <= 0;
  *o = (unsigned long long)((NID_pkcs7_signed) | 0);  /* check that NID_pkcs7_signed is an integer */
  return n;
}

static int _cffi_const_NID_poly1305(unsigned long long *o)
{
  int n = (NID_poly1305) <= 0;
  *o = (unsigned long long)((NID_poly1305) | 0);  /* check that NID_poly1305 is an integer */
  return n;
}

static int _cffi_const_NID_subject_alt_name(unsigned long long *o)
{
  int n = (NID_subject_alt_name) <= 0;
  *o = (unsigned long long)((NID_subject_alt_name) | 0);  /* check that NID_subject_alt_name is an integer */
  return n;
}

static int _cffi_const_NID_undef(unsigned long long *o)
{
  int n = (NID_undef) <= 0;
  *o = (unsigned long long)((NID_undef) | 0);  /* check that NID_undef is an integer */
  return n;
}

static int _cffi_const_OBJ_NAME_TYPE_MD_METH(unsigned long long *o)
{
  int n = (OBJ_NAME_TYPE_MD_METH) <= 0;
  *o = (unsigned long long)((OBJ_NAME_TYPE_MD_METH) | 0);  /* check that OBJ_NAME_TYPE_MD_METH is an integer */
  return n;
}

static int _cffi_const_OCSP_NOCERTS(unsigned long long *o)
{
  int n = (OCSP_NOCERTS) <= 0;
  *o = (unsigned long long)((OCSP_NOCERTS) | 0);  /* check that OCSP_NOCERTS is an integer */
  return n;
}

static int _cffi_const_OCSP_RESPID_KEY(unsigned long long *o)
{
  int n = (OCSP_RESPID_KEY) <= 0;
  *o = (unsigned long long)((OCSP_RESPID_KEY) | 0);  /* check that OCSP_RESPID_KEY is an integer */
  return n;
}

static int _cffi_const_OPENSSL_BUILT_ON(unsigned long long *o)
{
  int n = (OPENSSL_BUILT_ON) <= 0;
  *o = (unsigned long long)((OPENSSL_BUILT_ON) | 0);  /* check that OPENSSL_BUILT_ON is an integer */
  return n;
}

static int _cffi_const_OPENSSL_CFLAGS(unsigned long long *o)
{
  int n = (OPENSSL_CFLAGS) <= 0;
  *o = (unsigned long long)((OPENSSL_CFLAGS) | 0);  /* check that OPENSSL_CFLAGS is an integer */
  return n;
}

static int _cffi_const_OPENSSL_DIR(unsigned long long *o)
{
  int n = (OPENSSL_DIR) <= 0;
  *o = (unsigned long long)((OPENSSL_DIR) | 0);  /* check that OPENSSL_DIR is an integer */
  return n;
}

static int _cffi_const_OPENSSL_EC_NAMED_CURVE(unsigned long long *o)
{
  int n = (OPENSSL_EC_NAMED_CURVE) <= 0;
  *o = (unsigned long long)((OPENSSL_EC_NAMED_CURVE) | 0);  /* check that OPENSSL_EC_NAMED_CURVE is an integer */
  return n;
}

static int _cffi_const_OPENSSL_PLATFORM(unsigned long long *o)
{
  int n = (OPENSSL_PLATFORM) <= 0;
  *o = (unsigned long long)((OPENSSL_PLATFORM) | 0);  /* check that OPENSSL_PLATFORM is an integer */
  return n;
}

static int _cffi_const_OPENSSL_VERSION(unsigned long long *o)
{
  int n = (OPENSSL_VERSION) <= 0;
  *o = (unsigned long long)((OPENSSL_VERSION) | 0);  /* check that OPENSSL_VERSION is an integer */
  return n;
}

static int _cffi_const_OPENSSL_VERSION_NUMBER(unsigned long long *o)
{
  int n = (OPENSSL_VERSION_NUMBER) <= 0;
  *o = (unsigned long long)((OPENSSL_VERSION_NUMBER) | 0);  /* check that OPENSSL_VERSION_NUMBER is an integer */
  return n;
}

static void _cffi_const_OPENSSL_VERSION_TEXT(char *o)
{
  *(char const * *)o = OPENSSL_VERSION_TEXT;
}

static int _cffi_const_PEM_R_UNSUPPORTED_ENCRYPTION(unsigned long long *o)
{
  int n = (PEM_R_UNSUPPORTED_ENCRYPTION) <= 0;
  *o = (unsigned long long)((PEM_R_UNSUPPORTED_ENCRYPTION) | 0);  /* check that PEM_R_UNSUPPORTED_ENCRYPTION is an integer */
  return n;
}

static int _cffi_const_PKCS12_R_PKCS12_CIPHERFINAL_ERROR(unsigned long long *o)
{
  int n = (PKCS12_R_PKCS12_CIPHERFINAL_ERROR) <= 0;
  *o = (unsigned long long)((PKCS12_R_PKCS12_CIPHERFINAL_ERROR) | 0);  /* check that PKCS12_R_PKCS12_CIPHERFINAL_ERROR is an integer */
  return n;
}

static int _cffi_const_PKCS7_BINARY(unsigned long long *o)
{
  int n = (PKCS7_BINARY) <= 0;
  *o = (unsigned long long)((PKCS7_BINARY) | 0);  /* check that PKCS7_BINARY is an integer */
  return n;
}

static int _cffi_const_PKCS7_DETACHED(unsigned long long *o)
{
  int n = (PKCS7_DETACHED) <= 0;
  *o = (unsigned long long)((PKCS7_DETACHED) | 0);  /* check that PKCS7_DETACHED is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOATTR(unsigned long long *o)
{
  int n = (PKCS7_NOATTR) <= 0;
  *o = (unsigned long long)((PKCS7_NOATTR) | 0);  /* check that PKCS7_NOATTR is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOCERTS(unsigned long long *o)
{
  int n = (PKCS7_NOCERTS) <= 0;
  *o = (unsigned long long)((PKCS7_NOCERTS) | 0);  /* check that PKCS7_NOCERTS is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOCHAIN(unsigned long long *o)
{
  int n = (PKCS7_NOCHAIN) <= 0;
  *o = (unsigned long long)((PKCS7_NOCHAIN) | 0);  /* check that PKCS7_NOCHAIN is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOINTERN(unsigned long long *o)
{
  int n = (PKCS7_NOINTERN) <= 0;
  *o = (unsigned long long)((PKCS7_NOINTERN) | 0);  /* check that PKCS7_NOINTERN is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOSIGS(unsigned long long *o)
{
  int n = (PKCS7_NOSIGS) <= 0;
  *o = (unsigned long long)((PKCS7_NOSIGS) | 0);  /* check that PKCS7_NOSIGS is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOSMIMECAP(unsigned long long *o)
{
  int n = (PKCS7_NOSMIMECAP) <= 0;
  *o = (unsigned long long)((PKCS7_NOSMIMECAP) | 0);  /* check that PKCS7_NOSMIMECAP is an integer */
  return n;
}

static int _cffi_const_PKCS7_NOVERIFY(unsigned long long *o)
{
  int n = (PKCS7_NOVERIFY) <= 0;
  *o = (unsigned long long)((PKCS7_NOVERIFY) | 0);  /* check that PKCS7_NOVERIFY is an integer */
  return n;
}

static int _cffi_const_PKCS7_PARTIAL(unsigned long long *o)
{
  int n = (PKCS7_PARTIAL) <= 0;
  *o = (unsigned long long)((PKCS7_PARTIAL) | 0);  /* check that PKCS7_PARTIAL is an integer */
  return n;
}

static int _cffi_const_PKCS7_STREAM(unsigned long long *o)
{
  int n = (PKCS7_STREAM) <= 0;
  *o = (unsigned long long)((PKCS7_STREAM) | 0);  /* check that PKCS7_STREAM is an integer */
  return n;
}

static int _cffi_const_PKCS7_TEXT(unsigned long long *o)
{
  int n = (PKCS7_TEXT) <= 0;
  *o = (unsigned long long)((PKCS7_TEXT) | 0);  /* check that PKCS7_TEXT is an integer */
  return n;
}

static int _cffi_const_RSA_F4(unsigned long long *o)
{
  int n = (RSA_F4) <= 0;
  *o = (unsigned long long)((RSA_F4) | 0);  /* check that RSA_F4 is an integer */
  return n;
}

static int _cffi_const_RSA_NO_PADDING(unsigned long long *o)
{
  int n = (RSA_NO_PADDING) <= 0;
  *o = (unsigned long long)((RSA_NO_PADDING) | 0);  /* check that RSA_NO_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_PKCS1_OAEP_PADDING(unsigned long long *o)
{
  int n = (RSA_PKCS1_OAEP_PADDING) <= 0;
  *o = (unsigned long long)((RSA_PKCS1_OAEP_PADDING) | 0);  /* check that RSA_PKCS1_OAEP_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_PKCS1_PADDING(unsigned long long *o)
{
  int n = (RSA_PKCS1_PADDING) <= 0;
  *o = (unsigned long long)((RSA_PKCS1_PADDING) | 0);  /* check that RSA_PKCS1_PADDING is an integer */
  return n;
}

static int _cffi_const_RSA_PKCS1_PSS_PADDING(unsigned long long *o)
{
  int n = (RSA_PKCS1_PSS_PADDING) <= 0;
  *o = (unsigned long long)((RSA_PKCS1_PSS_PADDING) | 0);  /* check that RSA_PKCS1_PSS_PADDING is an integer */
  return n;
}

static int _cffi_const_SSL3_RANDOM_SIZE(unsigned long long *o)
{
  int n = (SSL3_RANDOM_SIZE) <= 0;
  *o = (unsigned long long)((SSL3_RANDOM_SIZE) | 0);  /* check that SSL3_RANDOM_SIZE is an integer */
  return n;
}

static int _cffi_const_SSLEAY_BUILT_ON(unsigned long long *o)
{
  int n = (SSLEAY_BUILT_ON) <= 0;
  *o = (unsigned long long)((SSLEAY_BUILT_ON) | 0);  /* check that SSLEAY_BUILT_ON is an integer */
  return n;
}

static int _cffi_const_SSLEAY_CFLAGS(unsigned long long *o)
{
  int n = (SSLEAY_CFLAGS) <= 0;
  *o = (unsigned long long)((SSLEAY_CFLAGS) | 0);  /* check that SSLEAY_CFLAGS is an integer */
  return n;
}

static int _cffi_const_SSLEAY_DIR(unsigned long long *o)
{
  int n = (SSLEAY_DIR) <= 0;
  *o = (unsigned long long)((SSLEAY_DIR) | 0);  /* check that SSLEAY_DIR is an integer */
  return n;
}

static int _cffi_const_SSLEAY_PLATFORM(unsigned long long *o)
{
  int n = (SSLEAY_PLATFORM) <= 0;
  *o = (unsigned long long)((SSLEAY_PLATFORM) | 0);  /* check that SSLEAY_PLATFORM is an integer */
  return n;
}

static int _cffi_const_SSLEAY_VERSION(unsigned long long *o)
{
  int n = (SSLEAY_VERSION) <= 0;
  *o = (unsigned long long)((SSLEAY_VERSION) | 0);  /* check that SSLEAY_VERSION is an integer */
  return n;
}

static int _cffi_const_SSL_CB_ACCEPT_EXIT(unsigned long long *o)
{
  int n = (SSL_CB_ACCEPT_EXIT) <= 0;
  *o = (unsigned long long)((SSL_CB_ACCEPT_EXIT) | 0);  /* check that SSL_CB_ACCEPT_EXIT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_ACCEPT_LOOP(unsigned long long *o)
{
  int n = (SSL_CB_ACCEPT_LOOP) <= 0;
  *o = (unsigned long long)((SSL_CB_ACCEPT_LOOP) | 0);  /* check that SSL_CB_ACCEPT_LOOP is an integer */
  return n;
}

static int _cffi_const_SSL_CB_ALERT(unsigned long long *o)
{
  int n = (SSL_CB_ALERT) <= 0;
  *o = (unsigned long long)((SSL_CB_ALERT) | 0);  /* check that SSL_CB_ALERT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_CONNECT_EXIT(unsigned long long *o)
{
  int n = (SSL_CB_CONNECT_EXIT) <= 0;
  *o = (unsigned long long)((SSL_CB_CONNECT_EXIT) | 0);  /* check that SSL_CB_CONNECT_EXIT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_CONNECT_LOOP(unsigned long long *o)
{
  int n = (SSL_CB_CONNECT_LOOP) <= 0;
  *o = (unsigned long long)((SSL_CB_CONNECT_LOOP) | 0);  /* check that SSL_CB_CONNECT_LOOP is an integer */
  return n;
}

static int _cffi_const_SSL_CB_EXIT(unsigned long long *o)
{
  int n = (SSL_CB_EXIT) <= 0;
  *o = (unsigned long long)((SSL_CB_EXIT) | 0);  /* check that SSL_CB_EXIT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_HANDSHAKE_DONE(unsigned long long *o)
{
  int n = (SSL_CB_HANDSHAKE_DONE) <= 0;
  *o = (unsigned long long)((SSL_CB_HANDSHAKE_DONE) | 0);  /* check that SSL_CB_HANDSHAKE_DONE is an integer */
  return n;
}

static int _cffi_const_SSL_CB_HANDSHAKE_START(unsigned long long *o)
{
  int n = (SSL_CB_HANDSHAKE_START) <= 0;
  *o = (unsigned long long)((SSL_CB_HANDSHAKE_START) | 0);  /* check that SSL_CB_HANDSHAKE_START is an integer */
  return n;
}

static int _cffi_const_SSL_CB_LOOP(unsigned long long *o)
{
  int n = (SSL_CB_LOOP) <= 0;
  *o = (unsigned long long)((SSL_CB_LOOP) | 0);  /* check that SSL_CB_LOOP is an integer */
  return n;
}

static int _cffi_const_SSL_CB_READ(unsigned long long *o)
{
  int n = (SSL_CB_READ) <= 0;
  *o = (unsigned long long)((SSL_CB_READ) | 0);  /* check that SSL_CB_READ is an integer */
  return n;
}

static int _cffi_const_SSL_CB_READ_ALERT(unsigned long long *o)
{
  int n = (SSL_CB_READ_ALERT) <= 0;
  *o = (unsigned long long)((SSL_CB_READ_ALERT) | 0);  /* check that SSL_CB_READ_ALERT is an integer */
  return n;
}

static int _cffi_const_SSL_CB_WRITE(unsigned long long *o)
{
  int n = (SSL_CB_WRITE) <= 0;
  *o = (unsigned long long)((SSL_CB_WRITE) | 0);  /* check that SSL_CB_WRITE is an integer */
  return n;
}

static int _cffi_const_SSL_CB_WRITE_ALERT(unsigned long long *o)
{
  int n = (SSL_CB_WRITE_ALERT) <= 0;
  *o = (unsigned long long)((SSL_CB_WRITE_ALERT) | 0);  /* check that SSL_CB_WRITE_ALERT is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_NONE(unsigned long long *o)
{
  int n = (SSL_ERROR_NONE) <= 0;
  *o = (unsigned long long)((SSL_ERROR_NONE) | 0);  /* check that SSL_ERROR_NONE is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_SSL(unsigned long long *o)
{
  int n = (SSL_ERROR_SSL) <= 0;
  *o = (unsigned long long)((SSL_ERROR_SSL) | 0);  /* check that SSL_ERROR_SSL is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_SYSCALL(unsigned long long *o)
{
  int n = (SSL_ERROR_SYSCALL) <= 0;
  *o = (unsigned long long)((SSL_ERROR_SYSCALL) | 0);  /* check that SSL_ERROR_SYSCALL is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_CONNECT(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_CONNECT) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_CONNECT) | 0);  /* check that SSL_ERROR_WANT_CONNECT is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_READ(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_READ) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_READ) | 0);  /* check that SSL_ERROR_WANT_READ is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_WRITE(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_WRITE) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_WRITE) | 0);  /* check that SSL_ERROR_WANT_WRITE is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_WANT_X509_LOOKUP(unsigned long long *o)
{
  int n = (SSL_ERROR_WANT_X509_LOOKUP) <= 0;
  *o = (unsigned long long)((SSL_ERROR_WANT_X509_LOOKUP) | 0);  /* check that SSL_ERROR_WANT_X509_LOOKUP is an integer */
  return n;
}

static int _cffi_const_SSL_ERROR_ZERO_RETURN(unsigned long long *o)
{
  int n = (SSL_ERROR_ZERO_RETURN) <= 0;
  *o = (unsigned long long)((SSL_ERROR_ZERO_RETURN) | 0);  /* check that SSL_ERROR_ZERO_RETURN is an integer */
  return n;
}

static int _cffi_const_SSL_FILETYPE_ASN1(unsigned long long *o)
{
  int n = (SSL_FILETYPE_ASN1) <= 0;
  *o = (unsigned long long)((SSL_FILETYPE_ASN1) | 0);  /* check that SSL_FILETYPE_ASN1 is an integer */
  return n;
}

static int _cffi_const_SSL_FILETYPE_PEM(unsigned long long *o)
{
  int n = (SSL_FILETYPE_PEM) <= 0;
  *o = (unsigned long long)((SSL_FILETYPE_PEM) | 0);  /* check that SSL_FILETYPE_PEM is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER(unsigned long long *o)
{
  int n = (SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER) <= 0;
  *o = (unsigned long long)((SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER) | 0);  /* check that SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_AUTO_RETRY(unsigned long long *o)
{
  int n = (SSL_MODE_AUTO_RETRY) <= 0;
  *o = (unsigned long long)((SSL_MODE_AUTO_RETRY) | 0);  /* check that SSL_MODE_AUTO_RETRY is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_ENABLE_PARTIAL_WRITE(unsigned long long *o)
{
  int n = (SSL_MODE_ENABLE_PARTIAL_WRITE) <= 0;
  *o = (unsigned long long)((SSL_MODE_ENABLE_PARTIAL_WRITE) | 0);  /* check that SSL_MODE_ENABLE_PARTIAL_WRITE is an integer */
  return n;
}

static int _cffi_const_SSL_MODE_RELEASE_BUFFERS(unsigned long long *o)
{
  int n = (SSL_MODE_RELEASE_BUFFERS) <= 0;
  *o = (unsigned long long)((SSL_MODE_RELEASE_BUFFERS) | 0);  /* check that SSL_MODE_RELEASE_BUFFERS is an integer */
  return n;
}

static int _cffi_const_SSL_OP_ALL(unsigned long long *o)
{
  int n = (SSL_OP_ALL) <= 0;
  *o = (unsigned long long)((SSL_OP_ALL) | 0);  /* check that SSL_OP_ALL is an integer */
  return n;
}

static int _cffi_const_SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION(unsigned long long *o)
{
  int n = (SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION) <= 0;
  *o = (unsigned long long)((SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION) | 0);  /* check that SSL_OP_ALLOW_UNSAFE_LEGACY_RENEGOTIATION is an integer */
  return n;
}

static int _cffi_const_SSL_OP_CIPHER_SERVER_PREFERENCE(unsigned long long *o)
{
  int n = (SSL_OP_CIPHER_SERVER_PREFERENCE) <= 0;
  *o = (unsigned long long)((SSL_OP_CIPHER_SERVER_PREFERENCE) | 0);  /* check that SSL_OP_CIPHER_SERVER_PREFERENCE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_COOKIE_EXCHANGE(unsigned long long *o)
{
  int n = (SSL_OP_COOKIE_EXCHANGE) <= 0;
  *o = (unsigned long long)((SSL_OP_COOKIE_EXCHANGE) | 0);  /* check that SSL_OP_COOKIE_EXCHANGE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS(unsigned long long *o)
{
  int n = (SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS) <= 0;
  *o = (unsigned long long)((SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS) | 0);  /* check that SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS is an integer */
  return n;
}

static int _cffi_const_SSL_OP_EPHEMERAL_RSA(unsigned long long *o)
{
  int n = (SSL_OP_EPHEMERAL_RSA) <= 0;
  *o = (unsigned long long)((SSL_OP_EPHEMERAL_RSA) | 0);  /* check that SSL_OP_EPHEMERAL_RSA is an integer */
  return n;
}

static int _cffi_const_SSL_OP_LEGACY_SERVER_CONNECT(unsigned long long *o)
{
  int n = (SSL_OP_LEGACY_SERVER_CONNECT) <= 0;
  *o = (unsigned long long)((SSL_OP_LEGACY_SERVER_CONNECT) | 0);  /* check that SSL_OP_LEGACY_SERVER_CONNECT is an integer */
  return n;
}

static int _cffi_const_SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER(unsigned long long *o)
{
  int n = (SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER) <= 0;
  *o = (unsigned long long)((SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER) | 0);  /* check that SSL_OP_MICROSOFT_BIG_SSLV3_BUFFER is an integer */
  return n;
}

static int _cffi_const_SSL_OP_MICROSOFT_SESS_ID_BUG(unsigned long long *o)
{
  int n = (SSL_OP_MICROSOFT_SESS_ID_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_MICROSOFT_SESS_ID_BUG) | 0);  /* check that SSL_OP_MICROSOFT_SESS_ID_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_MSIE_SSLV2_RSA_PADDING(unsigned long long *o)
{
  int n = (SSL_OP_MSIE_SSLV2_RSA_PADDING) <= 0;
  *o = (unsigned long long)((SSL_OP_MSIE_SSLV2_RSA_PADDING) | 0);  /* check that SSL_OP_MSIE_SSLV2_RSA_PADDING is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_CA_DN_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_CA_DN_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_CA_DN_BUG) | 0);  /* check that SSL_OP_NETSCAPE_CA_DN_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_CHALLENGE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_CHALLENGE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_CHALLENGE_BUG) | 0);  /* check that SSL_OP_NETSCAPE_CHALLENGE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG) | 0);  /* check that SSL_OP_NETSCAPE_DEMO_CIPHER_CHANGE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG) | 0);  /* check that SSL_OP_NETSCAPE_REUSE_CIPHER_CHANGE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_COMPRESSION(unsigned long long *o)
{
  int n = (SSL_OP_NO_COMPRESSION) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_COMPRESSION) | 0);  /* check that SSL_OP_NO_COMPRESSION is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_DTLSv1(unsigned long long *o)
{
  int n = (SSL_OP_NO_DTLSv1) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_DTLSv1) | 0);  /* check that SSL_OP_NO_DTLSv1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_DTLSv1_2(unsigned long long *o)
{
  int n = (SSL_OP_NO_DTLSv1_2) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_DTLSv1_2) | 0);  /* check that SSL_OP_NO_DTLSv1_2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_QUERY_MTU(unsigned long long *o)
{
  int n = (SSL_OP_NO_QUERY_MTU) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_QUERY_MTU) | 0);  /* check that SSL_OP_NO_QUERY_MTU is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_SSLv2(unsigned long long *o)
{
  int n = (SSL_OP_NO_SSLv2) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_SSLv2) | 0);  /* check that SSL_OP_NO_SSLv2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_SSLv3(unsigned long long *o)
{
  int n = (SSL_OP_NO_SSLv3) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_SSLv3) | 0);  /* check that SSL_OP_NO_SSLv3 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TICKET(unsigned long long *o)
{
  int n = (SSL_OP_NO_TICKET) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TICKET) | 0);  /* check that SSL_OP_NO_TICKET is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1) | 0);  /* check that SSL_OP_NO_TLSv1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1_1(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1_1) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1_1) | 0);  /* check that SSL_OP_NO_TLSv1_1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1_2(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1_2) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1_2) | 0);  /* check that SSL_OP_NO_TLSv1_2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_NO_TLSv1_3(unsigned long long *o)
{
  int n = (SSL_OP_NO_TLSv1_3) <= 0;
  *o = (unsigned long long)((SSL_OP_NO_TLSv1_3) | 0);  /* check that SSL_OP_NO_TLSv1_3 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_PKCS1_CHECK_1(unsigned long long *o)
{
  int n = (SSL_OP_PKCS1_CHECK_1) <= 0;
  *o = (unsigned long long)((SSL_OP_PKCS1_CHECK_1) | 0);  /* check that SSL_OP_PKCS1_CHECK_1 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_PKCS1_CHECK_2(unsigned long long *o)
{
  int n = (SSL_OP_PKCS1_CHECK_2) <= 0;
  *o = (unsigned long long)((SSL_OP_PKCS1_CHECK_2) | 0);  /* check that SSL_OP_PKCS1_CHECK_2 is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SINGLE_DH_USE(unsigned long long *o)
{
  int n = (SSL_OP_SINGLE_DH_USE) <= 0;
  *o = (unsigned long long)((SSL_OP_SINGLE_DH_USE) | 0);  /* check that SSL_OP_SINGLE_DH_USE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SINGLE_ECDH_USE(unsigned long long *o)
{
  int n = (SSL_OP_SINGLE_ECDH_USE) <= 0;
  *o = (unsigned long long)((SSL_OP_SINGLE_ECDH_USE) | 0);  /* check that SSL_OP_SINGLE_ECDH_USE is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SSLEAY_080_CLIENT_DH_BUG(unsigned long long *o)
{
  int n = (SSL_OP_SSLEAY_080_CLIENT_DH_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_SSLEAY_080_CLIENT_DH_BUG) | 0);  /* check that SSL_OP_SSLEAY_080_CLIENT_DH_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG(unsigned long long *o)
{
  int n = (SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG) | 0);  /* check that SSL_OP_SSLREF2_REUSE_CERT_TYPE_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_TLS_BLOCK_PADDING_BUG(unsigned long long *o)
{
  int n = (SSL_OP_TLS_BLOCK_PADDING_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_TLS_BLOCK_PADDING_BUG) | 0);  /* check that SSL_OP_TLS_BLOCK_PADDING_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_TLS_D5_BUG(unsigned long long *o)
{
  int n = (SSL_OP_TLS_D5_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_TLS_D5_BUG) | 0);  /* check that SSL_OP_TLS_D5_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_OP_TLS_ROLLBACK_BUG(unsigned long long *o)
{
  int n = (SSL_OP_TLS_ROLLBACK_BUG) <= 0;
  *o = (unsigned long long)((SSL_OP_TLS_ROLLBACK_BUG) | 0);  /* check that SSL_OP_TLS_ROLLBACK_BUG is an integer */
  return n;
}

static int _cffi_const_SSL_RECEIVED_SHUTDOWN(unsigned long long *o)
{
  int n = (SSL_RECEIVED_SHUTDOWN) <= 0;
  *o = (unsigned long long)((SSL_RECEIVED_SHUTDOWN) | 0);  /* check that SSL_RECEIVED_SHUTDOWN is an integer */
  return n;
}

static int _cffi_const_SSL_SENT_SHUTDOWN(unsigned long long *o)
{
  int n = (SSL_SENT_SHUTDOWN) <= 0;
  *o = (unsigned long long)((SSL_SENT_SHUTDOWN) | 0);  /* check that SSL_SENT_SHUTDOWN is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_BOTH(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_BOTH) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_BOTH) | 0);  /* check that SSL_SESS_CACHE_BOTH is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_CLIENT(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_CLIENT) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_CLIENT) | 0);  /* check that SSL_SESS_CACHE_CLIENT is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_AUTO_CLEAR(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_AUTO_CLEAR) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_AUTO_CLEAR) | 0);  /* check that SSL_SESS_CACHE_NO_AUTO_CLEAR is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_INTERNAL(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_INTERNAL) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_INTERNAL) | 0);  /* check that SSL_SESS_CACHE_NO_INTERNAL is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_INTERNAL_LOOKUP(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_INTERNAL_LOOKUP) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_INTERNAL_LOOKUP) | 0);  /* check that SSL_SESS_CACHE_NO_INTERNAL_LOOKUP is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_NO_INTERNAL_STORE(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_NO_INTERNAL_STORE) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_NO_INTERNAL_STORE) | 0);  /* check that SSL_SESS_CACHE_NO_INTERNAL_STORE is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_OFF(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_OFF) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_OFF) | 0);  /* check that SSL_SESS_CACHE_OFF is an integer */
  return n;
}

static int _cffi_const_SSL_SESS_CACHE_SERVER(unsigned long long *o)
{
  int n = (SSL_SESS_CACHE_SERVER) <= 0;
  *o = (unsigned long long)((SSL_SESS_CACHE_SERVER) | 0);  /* check that SSL_SESS_CACHE_SERVER is an integer */
  return n;
}

static int _cffi_const_SSL_ST_ACCEPT(unsigned long long *o)
{
  int n = (SSL_ST_ACCEPT) <= 0;
  *o = (unsigned long long)((SSL_ST_ACCEPT) | 0);  /* check that SSL_ST_ACCEPT is an integer */
  return n;
}

static int _cffi_const_SSL_ST_BEFORE(unsigned long long *o)
{
  int n = (SSL_ST_BEFORE) <= 0;
  *o = (unsigned long long)((SSL_ST_BEFORE) | 0);  /* check that SSL_ST_BEFORE is an integer */
  return n;
}

static int _cffi_const_SSL_ST_CONNECT(unsigned long long *o)
{
  int n = (SSL_ST_CONNECT) <= 0;
  *o = (unsigned long long)((SSL_ST_CONNECT) | 0);  /* check that SSL_ST_CONNECT is an integer */
  return n;
}

static int _cffi_const_SSL_ST_INIT(unsigned long long *o)
{
  int n = (SSL_ST_INIT) <= 0;
  *o = (unsigned long long)((SSL_ST_INIT) | 0);  /* check that SSL_ST_INIT is an integer */
  return n;
}

static int _cffi_const_SSL_ST_MASK(unsigned long long *o)
{
  int n = (SSL_ST_MASK) <= 0;
  *o = (unsigned long long)((SSL_ST_MASK) | 0);  /* check that SSL_ST_MASK is an integer */
  return n;
}

static int _cffi_const_SSL_ST_OK(unsigned long long *o)
{
  int n = (SSL_ST_OK) <= 0;
  *o = (unsigned long long)((SSL_ST_OK) | 0);  /* check that SSL_ST_OK is an integer */
  return n;
}

static int _cffi_const_SSL_ST_RENEGOTIATE(unsigned long long *o)
{
  int n = (SSL_ST_RENEGOTIATE) <= 0;
  *o = (unsigned long long)((SSL_ST_RENEGOTIATE) | 0);  /* check that SSL_ST_RENEGOTIATE is an integer */
  return n;
}

static int _cffi_const_SSL_TLSEXT_ERR_ALERT_FATAL(unsigned long long *o)
{
  int n = (SSL_TLSEXT_ERR_ALERT_FATAL) <= 0;
  *o = (unsigned long long)((SSL_TLSEXT_ERR_ALERT_FATAL) | 0);  /* check that SSL_TLSEXT_ERR_ALERT_FATAL is an integer */
  return n;
}

static int _cffi_const_SSL_TLSEXT_ERR_NOACK(unsigned long long *o)
{
  int n = (SSL_TLSEXT_ERR_NOACK) <= 0;
  *o = (unsigned long long)((SSL_TLSEXT_ERR_NOACK) | 0);  /* check that SSL_TLSEXT_ERR_NOACK is an integer */
  return n;
}

static int _cffi_const_SSL_TLSEXT_ERR_OK(unsigned long long *o)
{
  int n = (SSL_TLSEXT_ERR_OK) <= 0;
  *o = (unsigned long long)((SSL_TLSEXT_ERR_OK) | 0);  /* check that SSL_TLSEXT_ERR_OK is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_CLIENT_ONCE(unsigned long long *o)
{
  int n = (SSL_VERIFY_CLIENT_ONCE) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_CLIENT_ONCE) | 0);  /* check that SSL_VERIFY_CLIENT_ONCE is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_FAIL_IF_NO_PEER_CERT(unsigned long long *o)
{
  int n = (SSL_VERIFY_FAIL_IF_NO_PEER_CERT) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_FAIL_IF_NO_PEER_CERT) | 0);  /* check that SSL_VERIFY_FAIL_IF_NO_PEER_CERT is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_NONE(unsigned long long *o)
{
  int n = (SSL_VERIFY_NONE) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_NONE) | 0);  /* check that SSL_VERIFY_NONE is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_PEER(unsigned long long *o)
{
  int n = (SSL_VERIFY_PEER) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_PEER) | 0);  /* check that SSL_VERIFY_PEER is an integer */
  return n;
}

static int _cffi_const_SSL_VERIFY_POST_HANDSHAKE(unsigned long long *o)
{
  int n = (SSL_VERIFY_POST_HANDSHAKE) <= 0;
  *o = (unsigned long long)((SSL_VERIFY_POST_HANDSHAKE) | 0);  /* check that SSL_VERIFY_POST_HANDSHAKE is an integer */
  return n;
}

static int _cffi_const_TLSEXT_NAMETYPE_host_name(unsigned long long *o)
{
  int n = (TLSEXT_NAMETYPE_host_name) <= 0;
  *o = (unsigned long long)((TLSEXT_NAMETYPE_host_name) | 0);  /* check that TLSEXT_NAMETYPE_host_name is an integer */
  return n;
}

static int _cffi_const_TLSEXT_STATUSTYPE_ocsp(unsigned long long *o)
{
  int n = (TLSEXT_STATUSTYPE_ocsp) <= 0;
  *o = (unsigned long long)((TLSEXT_STATUSTYPE_ocsp) | 0);  /* check that TLSEXT_STATUSTYPE_ocsp is an integer */
  return n;
}

static int _cffi_const_TLS_ST_BEFORE(unsigned long long *o)
{
  int n = (TLS_ST_BEFORE) <= 0;
  *o = (unsigned long long)((TLS_ST_BEFORE) | 0);  /* check that TLS_ST_BEFORE is an integer */
  return n;
}

static int _cffi_const_TLS_ST_OK(unsigned long long *o)
{
  int n = (TLS_ST_OK) <= 0;
  *o = (unsigned long long)((TLS_ST_OK) | 0);  /* check that TLS_ST_OK is an integer */
  return n;
}

static int _cffi_const_V_ASN1_GENERALIZEDTIME(unsigned long long *o)
{
  int n = (V_ASN1_GENERALIZEDTIME) <= 0;
  *o = (unsigned long long)((V_ASN1_GENERALIZEDTIME) | 0);  /* check that V_ASN1_GENERALIZEDTIME is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT) | 0);  /* check that X509_CHECK_FLAG_ALWAYS_CHECK_SUBJECT is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS) | 0);  /* check that X509_CHECK_FLAG_MULTI_LABEL_WILDCARDS is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_NEVER_CHECK_SUBJECT(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_NEVER_CHECK_SUBJECT) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_NEVER_CHECK_SUBJECT) | 0);  /* check that X509_CHECK_FLAG_NEVER_CHECK_SUBJECT is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS) | 0);  /* check that X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_NO_WILDCARDS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_NO_WILDCARDS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_NO_WILDCARDS) | 0);  /* check that X509_CHECK_FLAG_NO_WILDCARDS is an integer */
  return n;
}

static int _cffi_const_X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS(unsigned long long *o)
{
  int n = (X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS) <= 0;
  *o = (unsigned long long)((X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS) | 0);  /* check that X509_CHECK_FLAG_SINGLE_LABEL_SUBDOMAINS is an integer */
  return n;
}

static int _cffi_const_X509_LU_CRL(unsigned long long *o)
{
  int n = (X509_LU_CRL) <= 0;
  *o = (unsigned long long)((X509_LU_CRL) | 0);  /* check that X509_LU_CRL is an integer */
  return n;
}

static int _cffi_const_X509_LU_X509(unsigned long long *o)
{
  int n = (X509_LU_X509) <= 0;
  *o = (unsigned long long)((X509_LU_X509) | 0);  /* check that X509_LU_X509 is an integer */
  return n;
}

static int _cffi_const_X509_R_CERT_ALREADY_IN_HASH_TABLE(unsigned long long *o)
{
  int n = (X509_R_CERT_ALREADY_IN_HASH_TABLE) <= 0;
  *o = (unsigned long long)((X509_R_CERT_ALREADY_IN_HASH_TABLE) | 0);  /* check that X509_R_CERT_ALREADY_IN_HASH_TABLE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH) | 0);  /* check that X509_V_ERR_AKID_ISSUER_SERIAL_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_AKID_SKID_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_AKID_SKID_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_AKID_SKID_MISMATCH) | 0);  /* check that X509_V_ERR_AKID_SKID_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_APPLICATION_VERIFICATION(unsigned long long *o)
{
  int n = (X509_V_ERR_APPLICATION_VERIFICATION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_APPLICATION_VERIFICATION) | 0);  /* check that X509_V_ERR_APPLICATION_VERIFICATION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_CHAIN_TOO_LONG(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_CHAIN_TOO_LONG) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_CHAIN_TOO_LONG) | 0);  /* check that X509_V_ERR_CERT_CHAIN_TOO_LONG is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_HAS_EXPIRED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_HAS_EXPIRED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_HAS_EXPIRED) | 0);  /* check that X509_V_ERR_CERT_HAS_EXPIRED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_NOT_YET_VALID(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_NOT_YET_VALID) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_NOT_YET_VALID) | 0);  /* check that X509_V_ERR_CERT_NOT_YET_VALID is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_REJECTED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_REJECTED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_REJECTED) | 0);  /* check that X509_V_ERR_CERT_REJECTED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_REVOKED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_REVOKED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_REVOKED) | 0);  /* check that X509_V_ERR_CERT_REVOKED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_SIGNATURE_FAILURE(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_SIGNATURE_FAILURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_SIGNATURE_FAILURE) | 0);  /* check that X509_V_ERR_CERT_SIGNATURE_FAILURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CERT_UNTRUSTED(unsigned long long *o)
{
  int n = (X509_V_ERR_CERT_UNTRUSTED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CERT_UNTRUSTED) | 0);  /* check that X509_V_ERR_CERT_UNTRUSTED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_HAS_EXPIRED(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_HAS_EXPIRED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_HAS_EXPIRED) | 0);  /* check that X509_V_ERR_CRL_HAS_EXPIRED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_NOT_YET_VALID(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_NOT_YET_VALID) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_NOT_YET_VALID) | 0);  /* check that X509_V_ERR_CRL_NOT_YET_VALID is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_PATH_VALIDATION_ERROR(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_PATH_VALIDATION_ERROR) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_PATH_VALIDATION_ERROR) | 0);  /* check that X509_V_ERR_CRL_PATH_VALIDATION_ERROR is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_CRL_SIGNATURE_FAILURE(unsigned long long *o)
{
  int n = (X509_V_ERR_CRL_SIGNATURE_FAILURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_CRL_SIGNATURE_FAILURE) | 0);  /* check that X509_V_ERR_CRL_SIGNATURE_FAILURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT(unsigned long long *o)
{
  int n = (X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT) <= 0;
  *o = (unsigned long long)((X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT) | 0);  /* check that X509_V_ERR_DEPTH_ZERO_SELF_SIGNED_CERT is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_DIFFERENT_CRL_SCOPE(unsigned long long *o)
{
  int n = (X509_V_ERR_DIFFERENT_CRL_SCOPE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_DIFFERENT_CRL_SCOPE) | 0);  /* check that X509_V_ERR_DIFFERENT_CRL_SCOPE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_EMAIL_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_EMAIL_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_EMAIL_MISMATCH) | 0);  /* check that X509_V_ERR_EMAIL_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD(unsigned long long *o)
{
  int n = (X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD) <= 0;
  *o = (unsigned long long)((X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD) | 0);  /* check that X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_EXCLUDED_VIOLATION(unsigned long long *o)
{
  int n = (X509_V_ERR_EXCLUDED_VIOLATION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_EXCLUDED_VIOLATION) | 0);  /* check that X509_V_ERR_EXCLUDED_VIOLATION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_HOSTNAME_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_HOSTNAME_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_HOSTNAME_MISMATCH) | 0);  /* check that X509_V_ERR_HOSTNAME_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_CA(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_CA) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_CA) | 0);  /* check that X509_V_ERR_INVALID_CA is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_EXTENSION) | 0);  /* check that X509_V_ERR_INVALID_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_NON_CA(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_NON_CA) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_NON_CA) | 0);  /* check that X509_V_ERR_INVALID_NON_CA is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_POLICY_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_POLICY_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_POLICY_EXTENSION) | 0);  /* check that X509_V_ERR_INVALID_POLICY_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_INVALID_PURPOSE(unsigned long long *o)
{
  int n = (X509_V_ERR_INVALID_PURPOSE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_INVALID_PURPOSE) | 0);  /* check that X509_V_ERR_INVALID_PURPOSE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_IP_ADDRESS_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_IP_ADDRESS_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_IP_ADDRESS_MISMATCH) | 0);  /* check that X509_V_ERR_IP_ADDRESS_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_KEYUSAGE_NO_CERTSIGN(unsigned long long *o)
{
  int n = (X509_V_ERR_KEYUSAGE_NO_CERTSIGN) <= 0;
  *o = (unsigned long long)((X509_V_ERR_KEYUSAGE_NO_CERTSIGN) | 0);  /* check that X509_V_ERR_KEYUSAGE_NO_CERTSIGN is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_KEYUSAGE_NO_CRL_SIGN(unsigned long long *o)
{
  int n = (X509_V_ERR_KEYUSAGE_NO_CRL_SIGN) <= 0;
  *o = (unsigned long long)((X509_V_ERR_KEYUSAGE_NO_CRL_SIGN) | 0);  /* check that X509_V_ERR_KEYUSAGE_NO_CRL_SIGN is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE) | 0);  /* check that X509_V_ERR_KEYUSAGE_NO_DIGITAL_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_NO_EXPLICIT_POLICY(unsigned long long *o)
{
  int n = (X509_V_ERR_NO_EXPLICIT_POLICY) <= 0;
  *o = (unsigned long long)((X509_V_ERR_NO_EXPLICIT_POLICY) | 0);  /* check that X509_V_ERR_NO_EXPLICIT_POLICY is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_OUT_OF_MEM(unsigned long long *o)
{
  int n = (X509_V_ERR_OUT_OF_MEM) <= 0;
  *o = (unsigned long long)((X509_V_ERR_OUT_OF_MEM) | 0);  /* check that X509_V_ERR_OUT_OF_MEM is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PATH_LENGTH_EXCEEDED(unsigned long long *o)
{
  int n = (X509_V_ERR_PATH_LENGTH_EXCEEDED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PATH_LENGTH_EXCEEDED) | 0);  /* check that X509_V_ERR_PATH_LENGTH_EXCEEDED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PERMITTED_VIOLATION(unsigned long long *o)
{
  int n = (X509_V_ERR_PERMITTED_VIOLATION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PERMITTED_VIOLATION) | 0);  /* check that X509_V_ERR_PERMITTED_VIOLATION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED(unsigned long long *o)
{
  int n = (X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED) | 0);  /* check that X509_V_ERR_PROXY_CERTIFICATES_NOT_ALLOWED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED(unsigned long long *o)
{
  int n = (X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED) | 0);  /* check that X509_V_ERR_PROXY_PATH_LENGTH_EXCEEDED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN(unsigned long long *o)
{
  int n = (X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN) | 0);  /* check that X509_V_ERR_SELF_SIGNED_CERT_IN_CHAIN is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUBJECT_ISSUER_MISMATCH(unsigned long long *o)
{
  int n = (X509_V_ERR_SUBJECT_ISSUER_MISMATCH) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUBJECT_ISSUER_MISMATCH) | 0);  /* check that X509_V_ERR_SUBJECT_ISSUER_MISMATCH is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUBTREE_MINMAX(unsigned long long *o)
{
  int n = (X509_V_ERR_SUBTREE_MINMAX) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUBTREE_MINMAX) | 0);  /* check that X509_V_ERR_SUBTREE_MINMAX is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_CANNOT_SIGN_P_384_WITH_P_256(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_CANNOT_SIGN_P_384_WITH_P_256) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_CANNOT_SIGN_P_384_WITH_P_256) | 0);  /* check that X509_V_ERR_SUITE_B_CANNOT_SIGN_P_384_WITH_P_256 is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_INVALID_ALGORITHM(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_INVALID_ALGORITHM) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_INVALID_ALGORITHM) | 0);  /* check that X509_V_ERR_SUITE_B_INVALID_ALGORITHM is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_INVALID_CURVE(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_INVALID_CURVE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_INVALID_CURVE) | 0);  /* check that X509_V_ERR_SUITE_B_INVALID_CURVE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_INVALID_SIGNATURE_ALGORITHM(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_INVALID_SIGNATURE_ALGORITHM) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_INVALID_SIGNATURE_ALGORITHM) | 0);  /* check that X509_V_ERR_SUITE_B_INVALID_SIGNATURE_ALGORITHM is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_INVALID_VERSION(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_INVALID_VERSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_INVALID_VERSION) | 0);  /* check that X509_V_ERR_SUITE_B_INVALID_VERSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_SUITE_B_LOS_NOT_ALLOWED(unsigned long long *o)
{
  int n = (X509_V_ERR_SUITE_B_LOS_NOT_ALLOWED) <= 0;
  *o = (unsigned long long)((X509_V_ERR_SUITE_B_LOS_NOT_ALLOWED) | 0);  /* check that X509_V_ERR_SUITE_B_LOS_NOT_ALLOWED is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY) | 0);  /* check that X509_V_ERR_UNABLE_TO_DECODE_ISSUER_PUBLIC_KEY is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE) | 0);  /* check that X509_V_ERR_UNABLE_TO_DECRYPT_CERT_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE) | 0);  /* check that X509_V_ERR_UNABLE_TO_DECRYPT_CRL_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_CRL(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_CRL) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_CRL) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_CRL is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_CRL_ISSUER is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY) | 0);  /* check that X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE) | 0);  /* check that X509_V_ERR_UNABLE_TO_VERIFY_LEAF_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION) | 0);  /* check that X509_V_ERR_UNHANDLED_CRITICAL_CRL_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION(unsigned long long *o)
{
  int n = (X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION) | 0);  /* check that X509_V_ERR_UNHANDLED_CRITICAL_EXTENSION is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNNESTED_RESOURCE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNNESTED_RESOURCE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNNESTED_RESOURCE) | 0);  /* check that X509_V_ERR_UNNESTED_RESOURCE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX) | 0);  /* check that X509_V_ERR_UNSUPPORTED_CONSTRAINT_SYNTAX is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE) | 0);  /* check that X509_V_ERR_UNSUPPORTED_CONSTRAINT_TYPE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE) | 0);  /* check that X509_V_ERR_UNSUPPORTED_EXTENSION_FEATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_ERR_UNSUPPORTED_NAME_SYNTAX(unsigned long long *o)
{
  int n = (X509_V_ERR_UNSUPPORTED_NAME_SYNTAX) <= 0;
  *o = (unsigned long long)((X509_V_ERR_UNSUPPORTED_NAME_SYNTAX) | 0);  /* check that X509_V_ERR_UNSUPPORTED_NAME_SYNTAX is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_ALLOW_PROXY_CERTS(unsigned long long *o)
{
  int n = (X509_V_FLAG_ALLOW_PROXY_CERTS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_ALLOW_PROXY_CERTS) | 0);  /* check that X509_V_FLAG_ALLOW_PROXY_CERTS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CB_ISSUER_CHECK(unsigned long long *o)
{
  int n = (X509_V_FLAG_CB_ISSUER_CHECK) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CB_ISSUER_CHECK) | 0);  /* check that X509_V_FLAG_CB_ISSUER_CHECK is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CHECK_SS_SIGNATURE(unsigned long long *o)
{
  int n = (X509_V_FLAG_CHECK_SS_SIGNATURE) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CHECK_SS_SIGNATURE) | 0);  /* check that X509_V_FLAG_CHECK_SS_SIGNATURE is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CRL_CHECK(unsigned long long *o)
{
  int n = (X509_V_FLAG_CRL_CHECK) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CRL_CHECK) | 0);  /* check that X509_V_FLAG_CRL_CHECK is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_CRL_CHECK_ALL(unsigned long long *o)
{
  int n = (X509_V_FLAG_CRL_CHECK_ALL) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_CRL_CHECK_ALL) | 0);  /* check that X509_V_FLAG_CRL_CHECK_ALL is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_EXPLICIT_POLICY(unsigned long long *o)
{
  int n = (X509_V_FLAG_EXPLICIT_POLICY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_EXPLICIT_POLICY) | 0);  /* check that X509_V_FLAG_EXPLICIT_POLICY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_EXTENDED_CRL_SUPPORT(unsigned long long *o)
{
  int n = (X509_V_FLAG_EXTENDED_CRL_SUPPORT) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_EXTENDED_CRL_SUPPORT) | 0);  /* check that X509_V_FLAG_EXTENDED_CRL_SUPPORT is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_IGNORE_CRITICAL(unsigned long long *o)
{
  int n = (X509_V_FLAG_IGNORE_CRITICAL) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_IGNORE_CRITICAL) | 0);  /* check that X509_V_FLAG_IGNORE_CRITICAL is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_INHIBIT_ANY(unsigned long long *o)
{
  int n = (X509_V_FLAG_INHIBIT_ANY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_INHIBIT_ANY) | 0);  /* check that X509_V_FLAG_INHIBIT_ANY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_INHIBIT_MAP(unsigned long long *o)
{
  int n = (X509_V_FLAG_INHIBIT_MAP) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_INHIBIT_MAP) | 0);  /* check that X509_V_FLAG_INHIBIT_MAP is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_NOTIFY_POLICY(unsigned long long *o)
{
  int n = (X509_V_FLAG_NOTIFY_POLICY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_NOTIFY_POLICY) | 0);  /* check that X509_V_FLAG_NOTIFY_POLICY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_PARTIAL_CHAIN(unsigned long long *o)
{
  int n = (X509_V_FLAG_PARTIAL_CHAIN) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_PARTIAL_CHAIN) | 0);  /* check that X509_V_FLAG_PARTIAL_CHAIN is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_POLICY_CHECK(unsigned long long *o)
{
  int n = (X509_V_FLAG_POLICY_CHECK) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_POLICY_CHECK) | 0);  /* check that X509_V_FLAG_POLICY_CHECK is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_SUITEB_128_LOS(unsigned long long *o)
{
  int n = (X509_V_FLAG_SUITEB_128_LOS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_SUITEB_128_LOS) | 0);  /* check that X509_V_FLAG_SUITEB_128_LOS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_SUITEB_128_LOS_ONLY(unsigned long long *o)
{
  int n = (X509_V_FLAG_SUITEB_128_LOS_ONLY) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_SUITEB_128_LOS_ONLY) | 0);  /* check that X509_V_FLAG_SUITEB_128_LOS_ONLY is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_SUITEB_192_LOS(unsigned long long *o)
{
  int n = (X509_V_FLAG_SUITEB_192_LOS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_SUITEB_192_LOS) | 0);  /* check that X509_V_FLAG_SUITEB_192_LOS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_TRUSTED_FIRST(unsigned long long *o)
{
  int n = (X509_V_FLAG_TRUSTED_FIRST) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_TRUSTED_FIRST) | 0);  /* check that X509_V_FLAG_TRUSTED_FIRST is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_USE_CHECK_TIME(unsigned long long *o)
{
  int n = (X509_V_FLAG_USE_CHECK_TIME) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_USE_CHECK_TIME) | 0);  /* check that X509_V_FLAG_USE_CHECK_TIME is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_USE_DELTAS(unsigned long long *o)
{
  int n = (X509_V_FLAG_USE_DELTAS) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_USE_DELTAS) | 0);  /* check that X509_V_FLAG_USE_DELTAS is an integer */
  return n;
}

static int _cffi_const_X509_V_FLAG_X509_STRICT(unsigned long long *o)
{
  int n = (X509_V_FLAG_X509_STRICT) <= 0;
  *o = (unsigned long long)((X509_V_FLAG_X509_STRICT) | 0);  /* check that X509_V_FLAG_X509_STRICT is an integer */
  return n;
}

static int _cffi_const_X509_V_OK(unsigned long long *o)
{
  int n = (X509_V_OK) <= 0;
  *o = (unsigned long long)((X509_V_OK) | 0);  /* check that X509_V_OK is an integer */
  return n;
}

static void _cffi_d_ACCESS_DESCRIPTION_free(ACCESS_DESCRIPTION * x0)
{
  ACCESS_DESCRIPTION_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ACCESS_DESCRIPTION_free(PyObject *self, PyObject *arg0)
{
  ACCESS_DESCRIPTION * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1056), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ACCESS_DESCRIPTION *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1056), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ACCESS_DESCRIPTION_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ACCESS_DESCRIPTION_free _cffi_d_ACCESS_DESCRIPTION_free
#endif

static ACCESS_DESCRIPTION * _cffi_d_ACCESS_DESCRIPTION_new(void)
{
  return ACCESS_DESCRIPTION_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ACCESS_DESCRIPTION_new(PyObject *self, PyObject *noarg)
{
  ACCESS_DESCRIPTION * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ACCESS_DESCRIPTION_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1056));
  return pyresult;
}
#else
#  define _cffi_f_ACCESS_DESCRIPTION_new _cffi_d_ACCESS_DESCRIPTION_new
#endif

static int _cffi_d_AES_unwrap_key(AES_KEY * x0, unsigned char const * x1, unsigned char * x2, unsigned char const * x3, unsigned int x4)
{
  return AES_unwrap_key(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_AES_unwrap_key(PyObject *self, PyObject *args)
{
  AES_KEY * x0;
  unsigned char const * x1;
  unsigned char * x2;
  unsigned char const * x3;
  unsigned int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "AES_unwrap_key", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(750), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (AES_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(750), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, unsigned int);
  if (x4 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = AES_unwrap_key(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_AES_unwrap_key _cffi_d_AES_unwrap_key
#endif

static int _cffi_d_AES_wrap_key(AES_KEY * x0, unsigned char const * x1, unsigned char * x2, unsigned char const * x3, unsigned int x4)
{
  return AES_wrap_key(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_AES_wrap_key(PyObject *self, PyObject *args)
{
  AES_KEY * x0;
  unsigned char const * x1;
  unsigned char * x2;
  unsigned char const * x3;
  unsigned int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "AES_wrap_key", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(750), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (AES_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(750), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, unsigned int);
  if (x4 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = AES_wrap_key(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_AES_wrap_key _cffi_d_AES_wrap_key
#endif

static void _cffi_d_ASN1_BIT_STRING_free(ASN1_OCTET_STRING * x0)
{
  ASN1_BIT_STRING_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_BIT_STRING_free(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_BIT_STRING_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_BIT_STRING_free _cffi_d_ASN1_BIT_STRING_free
#endif

static int _cffi_d_ASN1_BIT_STRING_get_bit(ASN1_OCTET_STRING * x0, int x1)
{
  return ASN1_BIT_STRING_get_bit(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_BIT_STRING_get_bit(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_BIT_STRING_get_bit", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_BIT_STRING_get_bit(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_BIT_STRING_get_bit _cffi_d_ASN1_BIT_STRING_get_bit
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_BIT_STRING_new(void)
{
  return ASN1_BIT_STRING_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_BIT_STRING_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_BIT_STRING_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_BIT_STRING_new _cffi_d_ASN1_BIT_STRING_new
#endif

static int _cffi_d_ASN1_BIT_STRING_set_bit(ASN1_OCTET_STRING * x0, int x1, int x2)
{
  return ASN1_BIT_STRING_set_bit(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_BIT_STRING_set_bit(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  int x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ASN1_BIT_STRING_set_bit", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_BIT_STRING_set_bit(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_BIT_STRING_set_bit _cffi_d_ASN1_BIT_STRING_set_bit
#endif

static void _cffi_d_ASN1_ENUMERATED_free(ASN1_ENUMERATED * x0)
{
  ASN1_ENUMERATED_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_free(PyObject *self, PyObject *arg0)
{
  ASN1_ENUMERATED * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(757), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_ENUMERATED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(757), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_ENUMERATED_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_free _cffi_d_ASN1_ENUMERATED_free
#endif

static long _cffi_d_ASN1_ENUMERATED_get(ASN1_ENUMERATED * x0)
{
  return ASN1_ENUMERATED_get(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_get(PyObject *self, PyObject *arg0)
{
  ASN1_ENUMERATED * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(757), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_ENUMERATED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(757), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_ENUMERATED_get(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_get _cffi_d_ASN1_ENUMERATED_get
#endif

static ASN1_ENUMERATED * _cffi_d_ASN1_ENUMERATED_new(void)
{
  return ASN1_ENUMERATED_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_new(PyObject *self, PyObject *noarg)
{
  ASN1_ENUMERATED * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_ENUMERATED_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(757));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_new _cffi_d_ASN1_ENUMERATED_new
#endif

static int _cffi_d_ASN1_ENUMERATED_set(ASN1_ENUMERATED * x0, long x1)
{
  return ASN1_ENUMERATED_set(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_ENUMERATED_set(PyObject *self, PyObject *args)
{
  ASN1_ENUMERATED * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_ENUMERATED_set", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(757), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_ENUMERATED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(757), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_ENUMERATED_set(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_ENUMERATED_set _cffi_d_ASN1_ENUMERATED_set
#endif

static void _cffi_d_ASN1_GENERALIZEDTIME_free(ASN1_GENERALIZEDTIME * x0)
{
  ASN1_GENERALIZEDTIME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_GENERALIZEDTIME_free(PyObject *self, PyObject *arg0)
{
  ASN1_GENERALIZEDTIME * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(9), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(9), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_GENERALIZEDTIME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_GENERALIZEDTIME_free _cffi_d_ASN1_GENERALIZEDTIME_free
#endif

static ASN1_GENERALIZEDTIME * _cffi_d_ASN1_GENERALIZEDTIME_set(ASN1_GENERALIZEDTIME * x0, time_t x1)
{
  return ASN1_GENERALIZEDTIME_set(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_GENERALIZEDTIME_set(PyObject *self, PyObject *args)
{
  ASN1_GENERALIZEDTIME * x0;
  time_t x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_GENERALIZEDTIME * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_GENERALIZEDTIME_set", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(9), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(9), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, time_t);
  if (x1 == (time_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_GENERALIZEDTIME_set(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(9));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_GENERALIZEDTIME_set _cffi_d_ASN1_GENERALIZEDTIME_set
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_IA5STRING_new(void)
{
  return ASN1_IA5STRING_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_IA5STRING_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_IA5STRING_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_IA5STRING_new _cffi_d_ASN1_IA5STRING_new
#endif

static void _cffi_d_ASN1_INTEGER_free(ASN1_INTEGER * x0)
{
  ASN1_INTEGER_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_INTEGER_free(PyObject *self, PyObject *arg0)
{
  ASN1_INTEGER * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(21), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(21), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_INTEGER_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_INTEGER_free _cffi_d_ASN1_INTEGER_free
#endif

static int _cffi_d_ASN1_INTEGER_set(ASN1_INTEGER * x0, long x1)
{
  return ASN1_INTEGER_set(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_INTEGER_set(PyObject *self, PyObject *args)
{
  ASN1_INTEGER * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_INTEGER_set", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(21), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(21), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_INTEGER_set(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_INTEGER_set _cffi_d_ASN1_INTEGER_set
#endif

static BIGNUM * _cffi_d_ASN1_INTEGER_to_BN(ASN1_INTEGER * x0, BIGNUM * x1)
{
  return ASN1_INTEGER_to_BN(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_INTEGER_to_BN(PyObject *self, PyObject *args)
{
  ASN1_INTEGER * x0;
  BIGNUM * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_INTEGER_to_BN", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(21), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(21), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_INTEGER_to_BN(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_INTEGER_to_BN _cffi_d_ASN1_INTEGER_to_BN
#endif

static ASN1_NULL * _cffi_d_ASN1_NULL_new(void)
{
  return ASN1_NULL_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_NULL_new(PyObject *self, PyObject *noarg)
{
  ASN1_NULL * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_NULL_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3013));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_NULL_new _cffi_d_ASN1_NULL_new
#endif

static void _cffi_d_ASN1_OBJECT_free(ASN1_OBJECT * x0)
{
  ASN1_OBJECT_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OBJECT_free(PyObject *self, PyObject *arg0)
{
  ASN1_OBJECT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(607), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(607), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_OBJECT_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_OBJECT_free _cffi_d_ASN1_OBJECT_free
#endif

static void _cffi_d_ASN1_OCTET_STRING_free(ASN1_OCTET_STRING * x0)
{
  ASN1_OCTET_STRING_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OCTET_STRING_free(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_OCTET_STRING_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_OCTET_STRING_free _cffi_d_ASN1_OCTET_STRING_free
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_OCTET_STRING_new(void)
{
  return ASN1_OCTET_STRING_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OCTET_STRING_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_OCTET_STRING_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_OCTET_STRING_new _cffi_d_ASN1_OCTET_STRING_new
#endif

static int _cffi_d_ASN1_OCTET_STRING_set(ASN1_OCTET_STRING * x0, unsigned char const * x1, int x2)
{
  return ASN1_OCTET_STRING_set(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_OCTET_STRING_set(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  unsigned char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ASN1_OCTET_STRING_set", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_OCTET_STRING_set(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_OCTET_STRING_set _cffi_d_ASN1_OCTET_STRING_set
#endif

static unsigned char * _cffi_d_ASN1_STRING_data(ASN1_OCTET_STRING * x0)
{
  return ASN1_STRING_data(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_data(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  unsigned char * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_data(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(752));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_data _cffi_d_ASN1_STRING_data
#endif

static int _cffi_d_ASN1_STRING_length(ASN1_OCTET_STRING * x0)
{
  return ASN1_STRING_length(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_length(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_length(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_length _cffi_d_ASN1_STRING_length
#endif

static int _cffi_d_ASN1_STRING_set(ASN1_OCTET_STRING * x0, void const * x1, int x2)
{
  return ASN1_STRING_set(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_set(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  void const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ASN1_STRING_set", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_set(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_set _cffi_d_ASN1_STRING_set
#endif

static int _cffi_d_ASN1_STRING_set_default_mask_asc(char * x0)
{
  return ASN1_STRING_set_default_mask_asc(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_set_default_mask_asc(PyObject *self, PyObject *arg0)
{
  char * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(409), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(409), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_set_default_mask_asc(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_set_default_mask_asc _cffi_d_ASN1_STRING_set_default_mask_asc
#endif

static int _cffi_d_ASN1_STRING_to_UTF8(unsigned char * * x0, ASN1_OCTET_STRING * x1)
{
  return ASN1_STRING_to_UTF8(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_to_UTF8(PyObject *self, PyObject *args)
{
  unsigned char * * x0;
  ASN1_OCTET_STRING * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_STRING_to_UTF8", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(802), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(802), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_to_UTF8(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_to_UTF8 _cffi_d_ASN1_STRING_to_UTF8
#endif

static int _cffi_d_ASN1_STRING_type(ASN1_OCTET_STRING * x0)
{
  return ASN1_STRING_type(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_STRING_type(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_STRING_type(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_STRING_type _cffi_d_ASN1_STRING_type
#endif

static void _cffi_d_ASN1_TIME_free(ASN1_OCTET_STRING * x0)
{
  ASN1_TIME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_free(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_TIME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_TIME_free _cffi_d_ASN1_TIME_free
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_TIME_new(void)
{
  return ASN1_TIME_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_new _cffi_d_ASN1_TIME_new
#endif

static int _cffi_d_ASN1_TIME_set_string(ASN1_OCTET_STRING * x0, char const * x1)
{
  return ASN1_TIME_set_string(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_set_string(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_TIME_set_string", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_set_string(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_set_string _cffi_d_ASN1_TIME_set_string
#endif

static ASN1_GENERALIZEDTIME * _cffi_d_ASN1_TIME_to_generalizedtime(ASN1_OCTET_STRING * x0, ASN1_GENERALIZEDTIME * * x1)
{
  return ASN1_TIME_to_generalizedtime(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_TIME_to_generalizedtime(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * x0;
  ASN1_GENERALIZEDTIME * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_GENERALIZEDTIME * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "ASN1_TIME_to_generalizedtime", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(14), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(14), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_TIME_to_generalizedtime(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(9));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ASN1_TIME_to_generalizedtime _cffi_d_ASN1_TIME_to_generalizedtime
#endif

static void _cffi_d_ASN1_UTF8STRING_free(ASN1_OCTET_STRING * x0)
{
  ASN1_UTF8STRING_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_UTF8STRING_free(PyObject *self, PyObject *arg0)
{
  ASN1_OCTET_STRING * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ASN1_UTF8STRING_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ASN1_UTF8STRING_free _cffi_d_ASN1_UTF8STRING_free
#endif

static ASN1_OCTET_STRING * _cffi_d_ASN1_UTF8STRING_new(void)
{
  return ASN1_UTF8STRING_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ASN1_UTF8STRING_new(PyObject *self, PyObject *noarg)
{
  ASN1_OCTET_STRING * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ASN1_UTF8STRING_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(13));
  return pyresult;
}
#else
#  define _cffi_f_ASN1_UTF8STRING_new _cffi_d_ASN1_UTF8STRING_new
#endif

static void _cffi_d_AUTHORITY_KEYID_free(AUTHORITY_KEYID * x0)
{
  AUTHORITY_KEYID_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_AUTHORITY_KEYID_free(PyObject *self, PyObject *arg0)
{
  AUTHORITY_KEYID * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2557), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (AUTHORITY_KEYID *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2557), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { AUTHORITY_KEYID_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_AUTHORITY_KEYID_free _cffi_d_AUTHORITY_KEYID_free
#endif

static AUTHORITY_KEYID * _cffi_d_AUTHORITY_KEYID_new(void)
{
  return AUTHORITY_KEYID_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_AUTHORITY_KEYID_new(PyObject *self, PyObject *noarg)
{
  AUTHORITY_KEYID * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = AUTHORITY_KEYID_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2557));
  return pyresult;
}
#else
#  define _cffi_f_AUTHORITY_KEYID_new _cffi_d_AUTHORITY_KEYID_new
#endif

static void _cffi_d_BASIC_CONSTRAINTS_free(BASIC_CONSTRAINTS * x0)
{
  BASIC_CONSTRAINTS_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BASIC_CONSTRAINTS_free(PyObject *self, PyObject *arg0)
{
  BASIC_CONSTRAINTS * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2560), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BASIC_CONSTRAINTS *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2560), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BASIC_CONSTRAINTS_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BASIC_CONSTRAINTS_free _cffi_d_BASIC_CONSTRAINTS_free
#endif

static BASIC_CONSTRAINTS * _cffi_d_BASIC_CONSTRAINTS_new(void)
{
  return BASIC_CONSTRAINTS_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BASIC_CONSTRAINTS_new(PyObject *self, PyObject *noarg)
{
  BASIC_CONSTRAINTS * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BASIC_CONSTRAINTS_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2560));
  return pyresult;
}
#else
#  define _cffi_f_BASIC_CONSTRAINTS_new _cffi_d_BASIC_CONSTRAINTS_new
#endif

static void _cffi_d_BIO_clear_retry_flags(BIO * x0)
{
  BIO_clear_retry_flags(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_clear_retry_flags(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BIO_clear_retry_flags(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BIO_clear_retry_flags _cffi_d_BIO_clear_retry_flags
#endif

static size_t _cffi_d_BIO_ctrl_pending(BIO * x0)
{
  return BIO_ctrl_pending(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_ctrl_pending(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_ctrl_pending(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_ctrl_pending _cffi_d_BIO_ctrl_pending
#endif

static int _cffi_d_BIO_free(BIO * x0)
{
  return BIO_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_free(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_free _cffi_d_BIO_free
#endif

static void _cffi_d_BIO_free_all(BIO * x0)
{
  BIO_free_all(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_free_all(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BIO_free_all(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BIO_free_all _cffi_d_BIO_free_all
#endif

static long _cffi_d_BIO_get_mem_data(BIO * x0, char * * x1)
{
  return BIO_get_mem_data(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_get_mem_data(PyObject *self, PyObject *args)
{
  BIO * x0;
  char * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_get_mem_data", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2290), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2290), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_get_mem_data(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_get_mem_data _cffi_d_BIO_get_mem_data
#endif

static int _cffi_d_BIO_gets(BIO * x0, char * x1, int x2)
{
  return BIO_gets(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_gets(PyObject *self, PyObject *args)
{
  BIO * x0;
  char * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BIO_gets", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(409), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(409), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_gets(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_gets _cffi_d_BIO_gets
#endif

static BIO * _cffi_d_BIO_new(BIO_METHOD * x0)
{
  return BIO_new(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new(PyObject *self, PyObject *arg0)
{
  BIO_METHOD * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIO * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(115), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO_METHOD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(115), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(186));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_new _cffi_d_BIO_new
#endif

static BIO * _cffi_d_BIO_new_dgram(int x0, int x1)
{
  return BIO_new_dgram(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new_dgram(PyObject *self, PyObject *args)
{
  int x0;
  int x1;
  BIO * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_new_dgram", 2, 2, &arg0, &arg1))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new_dgram(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(186));
  return pyresult;
}
#else
#  define _cffi_f_BIO_new_dgram _cffi_d_BIO_new_dgram
#endif

static BIO * _cffi_d_BIO_new_file(char const * x0, char const * x1)
{
  return BIO_new_file(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new_file(PyObject *self, PyObject *args)
{
  char const * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIO * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_new_file", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new_file(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(186));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_new_file _cffi_d_BIO_new_file
#endif

static BIO * _cffi_d_BIO_new_mem_buf(void const * x0, int x1)
{
  return BIO_new_mem_buf(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_new_mem_buf(PyObject *self, PyObject *args)
{
  void const * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIO * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_new_mem_buf", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_new_mem_buf(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(186));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_new_mem_buf _cffi_d_BIO_new_mem_buf
#endif

static int _cffi_d_BIO_read(BIO * x0, void * x1, int x2)
{
  return BIO_read(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_read(PyObject *self, PyObject *args)
{
  BIO * x0;
  void * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BIO_read", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_read(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_read _cffi_d_BIO_read
#endif

static int _cffi_d_BIO_reset(BIO * x0)
{
  return BIO_reset(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_reset(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_reset(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_reset _cffi_d_BIO_reset
#endif

static BIO_METHOD * _cffi_d_BIO_s_datagram(void)
{
  return BIO_s_datagram();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_s_datagram(PyObject *self, PyObject *noarg)
{
  BIO_METHOD * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_s_datagram(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(115));
  return pyresult;
}
#else
#  define _cffi_f_BIO_s_datagram _cffi_d_BIO_s_datagram
#endif

static BIO_METHOD * _cffi_d_BIO_s_mem(void)
{
  return BIO_s_mem();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_s_mem(PyObject *self, PyObject *noarg)
{
  BIO_METHOD * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_s_mem(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(115));
  return pyresult;
}
#else
#  define _cffi_f_BIO_s_mem _cffi_d_BIO_s_mem
#endif

static long _cffi_d_BIO_set_mem_eof_return(BIO * x0, int x1)
{
  return BIO_set_mem_eof_return(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_set_mem_eof_return(PyObject *self, PyObject *args)
{
  BIO * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BIO_set_mem_eof_return", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_set_mem_eof_return(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_set_mem_eof_return _cffi_d_BIO_set_mem_eof_return
#endif

static void _cffi_d_BIO_set_retry_read(BIO * x0)
{
  BIO_set_retry_read(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_set_retry_read(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BIO_set_retry_read(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BIO_set_retry_read _cffi_d_BIO_set_retry_read
#endif

static int _cffi_d_BIO_should_io_special(BIO * x0)
{
  return BIO_should_io_special(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_io_special(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_io_special(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_io_special _cffi_d_BIO_should_io_special
#endif

static int _cffi_d_BIO_should_read(BIO * x0)
{
  return BIO_should_read(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_read(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_read(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_read _cffi_d_BIO_should_read
#endif

static int _cffi_d_BIO_should_retry(BIO * x0)
{
  return BIO_should_retry(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_retry(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_retry(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_retry _cffi_d_BIO_should_retry
#endif

static int _cffi_d_BIO_should_write(BIO * x0)
{
  return BIO_should_write(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_should_write(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_should_write(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_should_write _cffi_d_BIO_should_write
#endif

static int _cffi_d_BIO_up_ref(BIO * x0)
{
  return BIO_up_ref(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_up_ref(PyObject *self, PyObject *arg0)
{
  BIO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_up_ref(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_up_ref _cffi_d_BIO_up_ref
#endif

static int _cffi_d_BIO_write(BIO * x0, void const * x1, int x2)
{
  return BIO_write(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BIO_write(PyObject *self, PyObject *args)
{
  BIO * x0;
  void const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BIO_write", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BIO_write(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BIO_write _cffi_d_BIO_write
#endif

static void _cffi_d_BN_CTX_end(BN_CTX * x0)
{
  BN_CTX_end(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_end(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_CTX_end(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_CTX_end _cffi_d_BN_CTX_end
#endif

static void _cffi_d_BN_CTX_free(BN_CTX * x0)
{
  BN_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_free(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_CTX_free _cffi_d_BN_CTX_free
#endif

static BIGNUM * _cffi_d_BN_CTX_get(BN_CTX * x0)
{
  return BN_CTX_get(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_get(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_CTX_get(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_CTX_get _cffi_d_BN_CTX_get
#endif

static BN_CTX * _cffi_d_BN_CTX_new(void)
{
  return BN_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_new(PyObject *self, PyObject *noarg)
{
  BN_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(94));
  return pyresult;
}
#else
#  define _cffi_f_BN_CTX_new _cffi_d_BN_CTX_new
#endif

static void _cffi_d_BN_CTX_start(BN_CTX * x0)
{
  BN_CTX_start(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_CTX_start(PyObject *self, PyObject *arg0)
{
  BN_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_CTX_start(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_CTX_start _cffi_d_BN_CTX_start
#endif

static void _cffi_d_BN_MONT_CTX_free(BN_MONT_CTX * x0)
{
  BN_MONT_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_MONT_CTX_free(PyObject *self, PyObject *arg0)
{
  BN_MONT_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(830), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(830), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_MONT_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_MONT_CTX_free _cffi_d_BN_MONT_CTX_free
#endif

static BN_MONT_CTX * _cffi_d_BN_MONT_CTX_new(void)
{
  return BN_MONT_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_MONT_CTX_new(PyObject *self, PyObject *noarg)
{
  BN_MONT_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_MONT_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(830));
  return pyresult;
}
#else
#  define _cffi_f_BN_MONT_CTX_new _cffi_d_BN_MONT_CTX_new
#endif

static int _cffi_d_BN_MONT_CTX_set(BN_MONT_CTX * x0, BIGNUM const * x1, BN_CTX * x2)
{
  return BN_MONT_CTX_set(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_MONT_CTX_set(PyObject *self, PyObject *args)
{
  BN_MONT_CTX * x0;
  BIGNUM const * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_MONT_CTX_set", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(830), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(830), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_MONT_CTX_set(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_MONT_CTX_set _cffi_d_BN_MONT_CTX_set
#endif

static int _cffi_d_BN_add(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2)
{
  return BN_add(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_add(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_add", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_add(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_add _cffi_d_BN_add
#endif

static BIGNUM * _cffi_d_BN_bin2bn(unsigned char const * x0, int x1, BIGNUM * x2)
{
  return BN_bin2bn(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_bin2bn(PyObject *self, PyObject *args)
{
  unsigned char const * x0;
  int x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_bin2bn", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_bin2bn(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_bin2bn _cffi_d_BN_bin2bn
#endif

static int _cffi_d_BN_bn2bin(BIGNUM const * x0, unsigned char * x1)
{
  return BN_bn2bin(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_bn2bin(PyObject *self, PyObject *args)
{
  BIGNUM const * x0;
  unsigned char * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_bn2bin", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_bn2bin(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_bn2bin _cffi_d_BN_bn2bin
#endif

static char * _cffi_d_BN_bn2hex(BIGNUM const * x0)
{
  return BN_bn2hex(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_bn2hex(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_bn2hex(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(409));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_bn2hex _cffi_d_BN_bn2hex
#endif

static void _cffi_d_BN_clear_free(BIGNUM * x0)
{
  BN_clear_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_clear_free(PyObject *self, PyObject *arg0)
{
  BIGNUM * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_clear_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_clear_free _cffi_d_BN_clear_free
#endif

static int _cffi_d_BN_cmp(BIGNUM const * x0, BIGNUM const * x1)
{
  return BN_cmp(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_cmp(PyObject *self, PyObject *args)
{
  BIGNUM const * x0;
  BIGNUM const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_cmp", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_cmp(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_cmp _cffi_d_BN_cmp
#endif

static BIGNUM * _cffi_d_BN_dup(BIGNUM const * x0)
{
  return BN_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_dup(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_dup _cffi_d_BN_dup
#endif

static void _cffi_d_BN_free(BIGNUM * x0)
{
  BN_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_free(PyObject *self, PyObject *arg0)
{
  BIGNUM * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_free _cffi_d_BN_free
#endif

static int _cffi_d_BN_generate_prime_ex(BIGNUM * x0, int x1, int x2, BIGNUM const * x3, BIGNUM const * x4, BN_GENCB * x5)
{
  return BN_generate_prime_ex(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_generate_prime_ex(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  int x1;
  int x2;
  BIGNUM const * x3;
  BIGNUM const * x4;
  BN_GENCB * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "BN_generate_prime_ex", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(848), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(848), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_generate_prime_ex(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_generate_prime_ex _cffi_d_BN_generate_prime_ex
#endif

static int _cffi_d_BN_hex2bn(BIGNUM * * x0, char const * x1)
{
  return BN_hex2bn(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_hex2bn(PyObject *self, PyObject *args)
{
  BIGNUM * * x0;
  char const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_hex2bn", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(805), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(805), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_hex2bn(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_hex2bn _cffi_d_BN_hex2bn
#endif

static int _cffi_d_BN_is_negative(BIGNUM const * x0)
{
  return BN_is_negative(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_is_negative(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_is_negative(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_is_negative _cffi_d_BN_is_negative
#endif

static int _cffi_d_BN_is_prime_ex(BIGNUM const * x0, int x1, BN_CTX * x2, BN_GENCB * x3)
{
  return BN_is_prime_ex(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_is_prime_ex(PyObject *self, PyObject *args)
{
  BIGNUM const * x0;
  int x1;
  BN_CTX * x2;
  BN_GENCB * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_is_prime_ex", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(848), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(848), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_is_prime_ex(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_is_prime_ex _cffi_d_BN_is_prime_ex
#endif

static int _cffi_d_BN_mod(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BN_CTX * x3)
{
  return BN_mod(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_mod", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod _cffi_d_BN_mod
#endif

static int _cffi_d_BN_mod_add(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_add(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_add(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_add", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_add(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_add _cffi_d_BN_mod_add
#endif

static int _cffi_d_BN_mod_exp(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_exp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_exp(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_exp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_exp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_exp _cffi_d_BN_mod_exp
#endif

static int _cffi_d_BN_mod_exp_mont(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4, BN_MONT_CTX * x5)
{
  return BN_mod_exp_mont(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_exp_mont(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  BN_MONT_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "BN_mod_exp_mont", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(830), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(830), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_exp_mont(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_exp_mont _cffi_d_BN_mod_exp_mont
#endif

static int _cffi_d_BN_mod_exp_mont_consttime(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4, BN_MONT_CTX * x5)
{
  return BN_mod_exp_mont_consttime(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_exp_mont_consttime(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  BN_MONT_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "BN_mod_exp_mont_consttime", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(830), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_MONT_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(830), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_exp_mont_consttime(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_exp_mont_consttime _cffi_d_BN_mod_exp_mont_consttime
#endif

static BIGNUM * _cffi_d_BN_mod_inverse(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BN_CTX * x3)
{
  return BN_mod_inverse(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_inverse(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_mod_inverse", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_inverse(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_inverse _cffi_d_BN_mod_inverse
#endif

static int _cffi_d_BN_mod_mul(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_mul(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_mul(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_mul", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_mul(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_mul _cffi_d_BN_mod_mul
#endif

static int _cffi_d_BN_mod_sub(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return BN_mod_sub(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_mod_sub(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "BN_mod_sub", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_mod_sub(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_mod_sub _cffi_d_BN_mod_sub
#endif

static BIGNUM * _cffi_d_BN_new(void)
{
  return BN_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_new(PyObject *self, PyObject *noarg)
{
  BIGNUM * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(20));
  return pyresult;
}
#else
#  define _cffi_f_BN_new _cffi_d_BN_new
#endif

static int _cffi_d_BN_nnmod(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2, BN_CTX * x3)
{
  return BN_nnmod(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_nnmod(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "BN_nnmod", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_nnmod(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_nnmod _cffi_d_BN_nnmod
#endif

static int _cffi_d_BN_num_bits(BIGNUM const * x0)
{
  return BN_num_bits(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_num_bits(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_num_bits(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_num_bits _cffi_d_BN_num_bits
#endif

static int _cffi_d_BN_num_bytes(BIGNUM const * x0)
{
  return BN_num_bytes(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_num_bytes(PyObject *self, PyObject *arg0)
{
  BIGNUM const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_num_bytes(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_num_bytes _cffi_d_BN_num_bytes
#endif

static int _cffi_d_BN_prime_checks_for_size(int x0)
{
  return BN_prime_checks_for_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_prime_checks_for_size(PyObject *self, PyObject *arg0)
{
  int x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_prime_checks_for_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_BN_prime_checks_for_size _cffi_d_BN_prime_checks_for_size
#endif

static int _cffi_d_BN_rand_range(BIGNUM * x0, BIGNUM const * x1)
{
  return BN_rand_range(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_rand_range(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_rand_range", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_rand_range(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_rand_range _cffi_d_BN_rand_range
#endif

static void _cffi_d_BN_set_flags(BIGNUM * x0, int x1)
{
  BN_set_flags(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_set_flags(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_set_flags", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { BN_set_flags(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_BN_set_flags _cffi_d_BN_set_flags
#endif

static int _cffi_d_BN_set_word(BIGNUM * x0, BN_ULONG x1)
{
  return BN_set_word(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_set_word(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BN_ULONG x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_set_word", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, BN_ULONG);
  if (x1 == (BN_ULONG)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_set_word(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_set_word _cffi_d_BN_set_word
#endif

static int _cffi_d_BN_sub(BIGNUM * x0, BIGNUM const * x1, BIGNUM const * x2)
{
  return BN_sub(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_sub(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  BIGNUM const * x1;
  BIGNUM const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "BN_sub", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_sub(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_sub _cffi_d_BN_sub
#endif

static ASN1_INTEGER * _cffi_d_BN_to_ASN1_INTEGER(BIGNUM * x0, ASN1_INTEGER * x1)
{
  return BN_to_ASN1_INTEGER(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_to_ASN1_INTEGER(PyObject *self, PyObject *args)
{
  BIGNUM * x0;
  ASN1_INTEGER * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_INTEGER * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "BN_to_ASN1_INTEGER", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(21), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(21), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_to_ASN1_INTEGER(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(21));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_BN_to_ASN1_INTEGER _cffi_d_BN_to_ASN1_INTEGER
#endif

static BIGNUM const * _cffi_d_BN_value_one(void)
{
  return BN_value_one();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_BN_value_one(PyObject *self, PyObject *noarg)
{
  BIGNUM const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = BN_value_one(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(92));
  return pyresult;
}
#else
#  define _cffi_f_BN_value_one _cffi_d_BN_value_one
#endif

static void _cffi_d_CERTIFICATEPOLICIES_free(Cryptography_STACK_OF_POLICYINFO * x0)
{
  CERTIFICATEPOLICIES_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CERTIFICATEPOLICIES_free(PyObject *self, PyObject *arg0)
{
  Cryptography_STACK_OF_POLICYINFO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(449), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_POLICYINFO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(449), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { CERTIFICATEPOLICIES_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_CERTIFICATEPOLICIES_free _cffi_d_CERTIFICATEPOLICIES_free
#endif

static int _cffi_d_CMAC_CTX_copy(CMAC_CTX * x0, CMAC_CTX const * x1)
{
  return CMAC_CTX_copy(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_CTX_copy(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  CMAC_CTX const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "CMAC_CTX_copy", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1031), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1031), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1032), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (CMAC_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1032), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_CTX_copy(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_CTX_copy _cffi_d_CMAC_CTX_copy
#endif

static void _cffi_d_CMAC_CTX_free(CMAC_CTX * x0)
{
  CMAC_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_CTX_free(PyObject *self, PyObject *arg0)
{
  CMAC_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1031), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1031), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { CMAC_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_CMAC_CTX_free _cffi_d_CMAC_CTX_free
#endif

static CMAC_CTX * _cffi_d_CMAC_CTX_new(void)
{
  return CMAC_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_CTX_new(PyObject *self, PyObject *noarg)
{
  CMAC_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1031));
  return pyresult;
}
#else
#  define _cffi_f_CMAC_CTX_new _cffi_d_CMAC_CTX_new
#endif

static int _cffi_d_CMAC_Final(CMAC_CTX * x0, unsigned char * x1, size_t * x2)
{
  return CMAC_Final(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_Final(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "CMAC_Final", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1031), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1031), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_Final(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_Final _cffi_d_CMAC_Final
#endif

static int _cffi_d_CMAC_Init(CMAC_CTX * x0, void const * x1, size_t x2, EVP_CIPHER const * x3, ENGINE * x4)
{
  return CMAC_Init(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_Init(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  void const * x1;
  size_t x2;
  EVP_CIPHER const * x3;
  ENGINE * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "CMAC_Init", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1031), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1031), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(885), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(885), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_Init(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_Init _cffi_d_CMAC_Init
#endif

static int _cffi_d_CMAC_Update(CMAC_CTX * x0, void const * x1, size_t x2)
{
  return CMAC_Update(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CMAC_Update(PyObject *self, PyObject *args)
{
  CMAC_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "CMAC_Update", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1031), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (CMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1031), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = CMAC_Update(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_CMAC_Update _cffi_d_CMAC_Update
#endif

static void _cffi_d_CRL_DIST_POINTS_free(Cryptography_STACK_OF_DIST_POINT * x0)
{
  CRL_DIST_POINTS_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_CRL_DIST_POINTS_free(PyObject *self, PyObject *arg0)
{
  Cryptography_STACK_OF_DIST_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(204), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_DIST_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(204), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { CRL_DIST_POINTS_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_CRL_DIST_POINTS_free _cffi_d_CRL_DIST_POINTS_free
#endif

static int _cffi_d_Cryptography_CRYPTO_set_mem_functions(void *(* x0)(size_t, char const *, int), void *(* x1)(void *, size_t, char const *, int), void(* x2)(void *, char const *, int))
{
  return Cryptography_CRYPTO_set_mem_functions(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_CRYPTO_set_mem_functions(PyObject *self, PyObject *args)
{
  void *(* x0)(size_t, char const *, int);
  void *(* x1)(void *, size_t, char const *, int);
  void(* x2)(void *, char const *, int);
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_CRYPTO_set_mem_functions", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  x0 = (void *(*)(size_t, char const *, int))_cffi_to_c_pointer(arg0, _cffi_type(2269));
  if (x0 == (void *(*)(size_t, char const *, int))NULL && PyErr_Occurred())
    return NULL;

  x1 = (void *(*)(void *, size_t, char const *, int))_cffi_to_c_pointer(arg1, _cffi_type(2270));
  if (x1 == (void *(*)(void *, size_t, char const *, int))NULL && PyErr_Occurred())
    return NULL;

  x2 = (void(*)(void *, char const *, int))_cffi_to_c_pointer(arg2, _cffi_type(2271));
  if (x2 == (void(*)(void *, char const *, int))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_CRYPTO_set_mem_functions(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_CRYPTO_set_mem_functions _cffi_d_Cryptography_CRYPTO_set_mem_functions
#endif

static int _cffi_d_Cryptography_DH_check(DH const * x0, int * x1)
{
  return Cryptography_DH_check(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_DH_check(PyObject *self, PyObject *args)
{
  DH const * x0;
  int * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "Cryptography_DH_check", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1165), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1165), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1169), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1169), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_DH_check(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_DH_check _cffi_d_Cryptography_DH_check
#endif

static long _cffi_d_Cryptography_DTLSv1_get_timeout(SSL * x0, time_t * x1, long * x2)
{
  return Cryptography_DTLSv1_get_timeout(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_DTLSv1_get_timeout(PyObject *self, PyObject *args)
{
  SSL * x0;
  time_t * x1;
  long * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_DTLSv1_get_timeout", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(487), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(487), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2313), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (time_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2313), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2314), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (long *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2314), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_DTLSv1_get_timeout(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_DTLSv1_get_timeout _cffi_d_Cryptography_DTLSv1_get_timeout
#endif

static void _cffi_d_Cryptography_EVP_MD_CTX_free(EVP_MD_CTX * x0)
{
  Cryptography_EVP_MD_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_EVP_MD_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_MD_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { Cryptography_EVP_MD_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_Cryptography_EVP_MD_CTX_free _cffi_d_Cryptography_EVP_MD_CTX_free
#endif

static EVP_MD_CTX * _cffi_d_Cryptography_EVP_MD_CTX_new(void)
{
  return Cryptography_EVP_MD_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_EVP_MD_CTX_new(PyObject *self, PyObject *noarg)
{
  EVP_MD_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_EVP_MD_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1351));
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_EVP_MD_CTX_new _cffi_d_Cryptography_EVP_MD_CTX_new
#endif

static int _cffi_d_Cryptography_EVP_PKEY_id(EVP_PKEY const * x0)
{
  return Cryptography_EVP_PKEY_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_EVP_PKEY_id(PyObject *self, PyObject *arg0)
{
  EVP_PKEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1445), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1445), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_EVP_PKEY_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_EVP_PKEY_id _cffi_d_Cryptography_EVP_PKEY_id
#endif

static X509_REVOKED * _cffi_d_Cryptography_X509_REVOKED_dup(X509_REVOKED * x0)
{
  return Cryptography_X509_REVOKED_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_X509_REVOKED_dup(PyObject *self, PyObject *arg0)
{
  X509_REVOKED * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_REVOKED * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(619), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (X509_REVOKED *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(619), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_X509_REVOKED_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(619));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_X509_REVOKED_dup _cffi_d_Cryptography_X509_REVOKED_dup
#endif

static int _cffi_d_Cryptography_add_osrandom_engine(void)
{
  return Cryptography_add_osrandom_engine();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_add_osrandom_engine(PyObject *self, PyObject *noarg)
{
  int result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_add_osrandom_engine(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_add_osrandom_engine _cffi_d_Cryptography_add_osrandom_engine
#endif

static DH * _cffi_d_Cryptography_d2i_DHxparams_bio(BIO * x0, DH * * x1)
{
  return Cryptography_d2i_DHxparams_bio(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_d2i_DHxparams_bio(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "Cryptography_d2i_DHxparams_bio", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(187), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(187), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_d2i_DHxparams_bio(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(196));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_d2i_DHxparams_bio _cffi_d_Cryptography_d2i_DHxparams_bio
#endif

static void _cffi_d_Cryptography_free_wrapper(void * x0, char const * x1, int x2)
{
  Cryptography_free_wrapper(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_free_wrapper(PyObject *self, PyObject *args)
{
  void * x0;
  char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_free_wrapper", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { Cryptography_free_wrapper(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_Cryptography_free_wrapper _cffi_d_Cryptography_free_wrapper
#endif

static int _cffi_d_Cryptography_i2d_DHxparams_bio(BIO * x0, DH * x1)
{
  return Cryptography_i2d_DHxparams_bio(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_i2d_DHxparams_bio(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "Cryptography_i2d_DHxparams_bio", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_i2d_DHxparams_bio(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_i2d_DHxparams_bio _cffi_d_Cryptography_i2d_DHxparams_bio
#endif

static void * _cffi_d_Cryptography_malloc_wrapper(size_t x0, char const * x1, int x2)
{
  return Cryptography_malloc_wrapper(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_malloc_wrapper(PyObject *self, PyObject *args)
{
  size_t x0;
  char const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  void * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "Cryptography_malloc_wrapper", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  x0 = _cffi_to_c_int(arg0, size_t);
  if (x0 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_malloc_wrapper(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(193));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_malloc_wrapper _cffi_d_Cryptography_malloc_wrapper
#endif

static int _cffi_d_Cryptography_pem_password_cb(char * x0, int x1, int x2, void * x3)
{
  return Cryptography_pem_password_cb(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_pem_password_cb(PyObject *self, PyObject *args)
{
  char * x0;
  int x1;
  int x2;
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "Cryptography_pem_password_cb", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(409), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(409), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_pem_password_cb(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_pem_password_cb _cffi_d_Cryptography_pem_password_cb
#endif

static void * _cffi_d_Cryptography_realloc_wrapper(void * x0, size_t x1, char const * x2, int x3)
{
  return Cryptography_realloc_wrapper(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_Cryptography_realloc_wrapper(PyObject *self, PyObject *args)
{
  void * x0;
  size_t x1;
  char const * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  void * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "Cryptography_realloc_wrapper", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = Cryptography_realloc_wrapper(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(193));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_Cryptography_realloc_wrapper _cffi_d_Cryptography_realloc_wrapper
#endif

static int _cffi_d_DH_compute_key(unsigned char * x0, BIGNUM const * x1, DH * x2)
{
  return DH_compute_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_compute_key(PyObject *self, PyObject *args)
{
  unsigned char * x0;
  BIGNUM const * x1;
  DH * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DH_compute_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_compute_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_compute_key _cffi_d_DH_compute_key
#endif

static void _cffi_d_DH_free(DH * x0)
{
  DH_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_free(PyObject *self, PyObject *arg0)
{
  DH * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DH_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DH_free _cffi_d_DH_free
#endif

static int _cffi_d_DH_generate_key(DH * x0)
{
  return DH_generate_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_generate_key(PyObject *self, PyObject *arg0)
{
  DH * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_generate_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_generate_key _cffi_d_DH_generate_key
#endif

static int _cffi_d_DH_generate_parameters_ex(DH * x0, int x1, int x2, BN_GENCB * x3)
{
  return DH_generate_parameters_ex(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_generate_parameters_ex(PyObject *self, PyObject *args)
{
  DH * x0;
  int x1;
  int x2;
  BN_GENCB * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DH_generate_parameters_ex", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(848), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(848), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_generate_parameters_ex(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_generate_parameters_ex _cffi_d_DH_generate_parameters_ex
#endif

static void _cffi_d_DH_get0_key(DH const * x0, BIGNUM const * * x1, BIGNUM const * * x2)
{
  DH_get0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_get0_key(PyObject *self, PyObject *args)
{
  DH const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DH_get0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1165), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1165), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DH_get0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DH_get0_key _cffi_d_DH_get0_key
#endif

static void _cffi_d_DH_get0_pqg(DH const * x0, BIGNUM const * * x1, BIGNUM const * * x2, BIGNUM const * * x3)
{
  DH_get0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_get0_pqg(PyObject *self, PyObject *args)
{
  DH const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  BIGNUM const * * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DH_get0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1165), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1165), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DH_get0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DH_get0_pqg _cffi_d_DH_get0_pqg
#endif

static DH * _cffi_d_DH_new(void)
{
  return DH_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_new(PyObject *self, PyObject *noarg)
{
  DH * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(196));
  return pyresult;
}
#else
#  define _cffi_f_DH_new _cffi_d_DH_new
#endif

static int _cffi_d_DH_set0_key(DH * x0, BIGNUM * x1, BIGNUM * x2)
{
  return DH_set0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_set0_key(PyObject *self, PyObject *args)
{
  DH * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DH_set0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_set0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_set0_key _cffi_d_DH_set0_key
#endif

static int _cffi_d_DH_set0_pqg(DH * x0, BIGNUM * x1, BIGNUM * x2, BIGNUM * x3)
{
  return DH_set0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_set0_pqg(PyObject *self, PyObject *args)
{
  DH * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DH_set0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_set0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_set0_pqg _cffi_d_DH_set0_pqg
#endif

static int _cffi_d_DH_size(DH const * x0)
{
  return DH_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DH_size(PyObject *self, PyObject *arg0)
{
  DH const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1165), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1165), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DH_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DH_size _cffi_d_DH_size
#endif

static DH * _cffi_d_DHparams_dup(DH * x0)
{
  return DHparams_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DHparams_dup(PyObject *self, PyObject *arg0)
{
  DH * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DHparams_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(196));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DHparams_dup _cffi_d_DHparams_dup
#endif

static void _cffi_d_DIST_POINT_NAME_free(DIST_POINT_NAME * x0)
{
  DIST_POINT_NAME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DIST_POINT_NAME_free(PyObject *self, PyObject *arg0)
{
  DIST_POINT_NAME * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2644), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DIST_POINT_NAME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2644), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DIST_POINT_NAME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DIST_POINT_NAME_free _cffi_d_DIST_POINT_NAME_free
#endif

static DIST_POINT_NAME * _cffi_d_DIST_POINT_NAME_new(void)
{
  return DIST_POINT_NAME_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DIST_POINT_NAME_new(PyObject *self, PyObject *noarg)
{
  DIST_POINT_NAME * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DIST_POINT_NAME_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2644));
  return pyresult;
}
#else
#  define _cffi_f_DIST_POINT_NAME_new _cffi_d_DIST_POINT_NAME_new
#endif

static void _cffi_d_DIST_POINT_free(DIST_POINT * x0)
{
  DIST_POINT_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DIST_POINT_free(PyObject *self, PyObject *arg0)
{
  DIST_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1077), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DIST_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1077), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DIST_POINT_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DIST_POINT_free _cffi_d_DIST_POINT_free
#endif

static DIST_POINT * _cffi_d_DIST_POINT_new(void)
{
  return DIST_POINT_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DIST_POINT_new(PyObject *self, PyObject *noarg)
{
  DIST_POINT * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DIST_POINT_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1077));
  return pyresult;
}
#else
#  define _cffi_f_DIST_POINT_new _cffi_d_DIST_POINT_new
#endif

static void _cffi_d_DSA_free(DSA * x0)
{
  DSA_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_free(PyObject *self, PyObject *arg0)
{
  DSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(212), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(212), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DSA_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DSA_free _cffi_d_DSA_free
#endif

static int _cffi_d_DSA_generate_key(DSA * x0)
{
  return DSA_generate_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_generate_key(PyObject *self, PyObject *arg0)
{
  DSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(212), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(212), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_generate_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_generate_key _cffi_d_DSA_generate_key
#endif

static int _cffi_d_DSA_generate_parameters_ex(DSA * x0, int x1, unsigned char * x2, int x3, int * x4, unsigned long * x5, BN_GENCB * x6)
{
  return DSA_generate_parameters_ex(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_generate_parameters_ex(PyObject *self, PyObject *args)
{
  DSA * x0;
  int x1;
  unsigned char * x2;
  int x3;
  int * x4;
  unsigned long * x5;
  BN_GENCB * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "DSA_generate_parameters_ex", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(212), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(212), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1169), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1169), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1191), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (unsigned long *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1191), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(848), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(848), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_generate_parameters_ex(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_generate_parameters_ex _cffi_d_DSA_generate_parameters_ex
#endif

static void _cffi_d_DSA_get0_key(DSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2)
{
  DSA_get0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_get0_key(PyObject *self, PyObject *args)
{
  DSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DSA_get0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1195), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1195), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DSA_get0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DSA_get0_key _cffi_d_DSA_get0_key
#endif

static void _cffi_d_DSA_get0_pqg(DSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2, BIGNUM const * * x3)
{
  DSA_get0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_get0_pqg(PyObject *self, PyObject *args)
{
  DSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  BIGNUM const * * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DSA_get0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1195), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1195), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { DSA_get0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_DSA_get0_pqg _cffi_d_DSA_get0_pqg
#endif

static DSA * _cffi_d_DSA_new(void)
{
  return DSA_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_new(PyObject *self, PyObject *noarg)
{
  DSA * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(212));
  return pyresult;
}
#else
#  define _cffi_f_DSA_new _cffi_d_DSA_new
#endif

static int _cffi_d_DSA_set0_key(DSA * x0, BIGNUM * x1, BIGNUM * x2)
{
  return DSA_set0_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_set0_key(PyObject *self, PyObject *args)
{
  DSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "DSA_set0_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(212), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(212), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_set0_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_set0_key _cffi_d_DSA_set0_key
#endif

static int _cffi_d_DSA_set0_pqg(DSA * x0, BIGNUM * x1, BIGNUM * x2, BIGNUM * x3)
{
  return DSA_set0_pqg(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_set0_pqg(PyObject *self, PyObject *args)
{
  DSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "DSA_set0_pqg", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(212), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(212), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_set0_pqg(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_set0_pqg _cffi_d_DSA_set0_pqg
#endif

static int _cffi_d_DSA_sign(int x0, unsigned char const * x1, int x2, unsigned char * x3, unsigned int * x4, DSA * x5)
{
  return DSA_sign(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_sign(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char * x3;
  unsigned int * x4;
  DSA * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "DSA_sign", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1390), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1390), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(212), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(212), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_sign(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_sign _cffi_d_DSA_sign
#endif

static int _cffi_d_DSA_size(DSA const * x0)
{
  return DSA_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_size(PyObject *self, PyObject *arg0)
{
  DSA const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1195), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1195), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_size _cffi_d_DSA_size
#endif

static int _cffi_d_DSA_verify(int x0, unsigned char const * x1, int x2, unsigned char const * x3, int x4, DSA * x5)
{
  return DSA_verify(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSA_verify(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char const * x3;
  int x4;
  DSA * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "DSA_verify", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(212), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(212), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSA_verify(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSA_verify _cffi_d_DSA_verify
#endif

static DSA * _cffi_d_DSAparams_dup(DSA * x0)
{
  return DSAparams_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DSAparams_dup(PyObject *self, PyObject *arg0)
{
  DSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(212), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(212), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DSAparams_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(212));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DSAparams_dup _cffi_d_DSAparams_dup
#endif

static SSL_METHOD const * _cffi_d_DTLS_client_method(void)
{
  return DTLS_client_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_client_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_client_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3114));
  return pyresult;
}
#else
#  define _cffi_f_DTLS_client_method _cffi_d_DTLS_client_method
#endif

static long _cffi_d_DTLS_get_link_min_mtu(SSL * x0)
{
  return DTLS_get_link_min_mtu(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_get_link_min_mtu(PyObject *self, PyObject *arg0)
{
  SSL * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(487), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(487), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_get_link_min_mtu(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLS_get_link_min_mtu _cffi_d_DTLS_get_link_min_mtu
#endif

static SSL_METHOD const * _cffi_d_DTLS_method(void)
{
  return DTLS_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3114));
  return pyresult;
}
#else
#  define _cffi_f_DTLS_method _cffi_d_DTLS_method
#endif

static SSL_METHOD const * _cffi_d_DTLS_server_method(void)
{
  return DTLS_server_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_server_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_server_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3114));
  return pyresult;
}
#else
#  define _cffi_f_DTLS_server_method _cffi_d_DTLS_server_method
#endif

static long _cffi_d_DTLS_set_link_mtu(SSL * x0, long x1)
{
  return DTLS_set_link_mtu(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLS_set_link_mtu(PyObject *self, PyObject *args)
{
  SSL * x0;
  long x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "DTLS_set_link_mtu", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(487), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(487), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, long);
  if (x1 == (long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLS_set_link_mtu(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLS_set_link_mtu _cffi_d_DTLS_set_link_mtu
#endif

static SSL_METHOD const * _cffi_d_DTLSv1_client_method(void)
{
  return DTLSv1_client_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_client_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_client_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3114));
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_client_method _cffi_d_DTLSv1_client_method
#endif

static long _cffi_d_DTLSv1_handle_timeout(SSL * x0)
{
  return DTLSv1_handle_timeout(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_handle_timeout(PyObject *self, PyObject *arg0)
{
  SSL * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  long result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(487), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SSL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(487), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_handle_timeout(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, long);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_handle_timeout _cffi_d_DTLSv1_handle_timeout
#endif

static SSL_METHOD const * _cffi_d_DTLSv1_method(void)
{
  return DTLSv1_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3114));
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_method _cffi_d_DTLSv1_method
#endif

static SSL_METHOD const * _cffi_d_DTLSv1_server_method(void)
{
  return DTLSv1_server_method();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_DTLSv1_server_method(PyObject *self, PyObject *noarg)
{
  SSL_METHOD const * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = DTLSv1_server_method(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3114));
  return pyresult;
}
#else
#  define _cffi_f_DTLSv1_server_method _cffi_d_DTLSv1_server_method
#endif

static int _cffi_d_ECDH_compute_key(void * x0, size_t x1, EC_POINT const * x2, EC_KEY * x3, void *(* x4)(void const *, size_t, void *, size_t *))
{
  return ECDH_compute_key(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDH_compute_key(PyObject *self, PyObject *args)
{
  void * x0;
  size_t x1;
  EC_POINT const * x2;
  EC_KEY * x3;
  void *(* x4)(void const *, size_t, void *, size_t *);
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "ECDH_compute_key", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = (void *(*)(void const *, size_t, void *, size_t *))_cffi_to_c_pointer(arg4, _cffi_type(2278));
  if (x4 == (void *(*)(void const *, size_t, void *, size_t *))NULL && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDH_compute_key(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDH_compute_key _cffi_d_ECDH_compute_key
#endif

static int _cffi_d_ECDSA_sign(int x0, unsigned char const * x1, int x2, unsigned char * x3, unsigned int * x4, EC_KEY * x5)
{
  return ECDSA_sign(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_sign(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char * x3;
  unsigned int * x4;
  EC_KEY * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "ECDSA_sign", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1390), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1390), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_sign(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_sign _cffi_d_ECDSA_sign
#endif

static int _cffi_d_ECDSA_size(EC_KEY const * x0)
{
  return ECDSA_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_size(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_size _cffi_d_ECDSA_size
#endif

static int _cffi_d_ECDSA_verify(int x0, unsigned char const * x1, int x2, unsigned char const * x3, int x4, EC_KEY * x5)
{
  return ECDSA_verify(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ECDSA_verify(PyObject *self, PyObject *args)
{
  int x0;
  unsigned char const * x1;
  int x2;
  unsigned char const * x3;
  int x4;
  EC_KEY * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "ECDSA_verify", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ECDSA_verify(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ECDSA_verify _cffi_d_ECDSA_verify
#endif

static void _cffi_d_EC_GROUP_free(EC_GROUP * x0)
{
  EC_GROUP_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_free(PyObject *self, PyObject *arg0)
{
  EC_GROUP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2661), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2661), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_GROUP_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_GROUP_free _cffi_d_EC_GROUP_free
#endif

static EC_POINT const * _cffi_d_EC_GROUP_get0_generator(EC_GROUP const * x0)
{
  return EC_GROUP_get0_generator(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get0_generator(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get0_generator(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(244));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get0_generator _cffi_d_EC_GROUP_get0_generator
#endif

static int _cffi_d_EC_GROUP_get_asn1_flag(EC_GROUP const * x0)
{
  return EC_GROUP_get_asn1_flag(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get_asn1_flag(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get_asn1_flag(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get_asn1_flag _cffi_d_EC_GROUP_get_asn1_flag
#endif

static int _cffi_d_EC_GROUP_get_curve_name(EC_GROUP const * x0)
{
  return EC_GROUP_get_curve_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get_curve_name(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get_curve_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get_curve_name _cffi_d_EC_GROUP_get_curve_name
#endif

static int _cffi_d_EC_GROUP_get_degree(EC_GROUP const * x0)
{
  return EC_GROUP_get_degree(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get_degree(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get_degree(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get_degree _cffi_d_EC_GROUP_get_degree
#endif

static int _cffi_d_EC_GROUP_get_order(EC_GROUP const * x0, BIGNUM * x1, BN_CTX * x2)
{
  return EC_GROUP_get_order(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_get_order(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  BIGNUM * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_GROUP_get_order", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_get_order(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_get_order _cffi_d_EC_GROUP_get_order
#endif

static EC_METHOD const * _cffi_d_EC_GROUP_method_of(EC_GROUP const * x0)
{
  return EC_GROUP_method_of(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_method_of(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_METHOD const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_method_of(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1298));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_method_of _cffi_d_EC_GROUP_method_of
#endif

static EC_GROUP * _cffi_d_EC_GROUP_new_by_curve_name(int x0)
{
  return EC_GROUP_new_by_curve_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_GROUP_new_by_curve_name(PyObject *self, PyObject *arg0)
{
  int x0;
  EC_GROUP * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_GROUP_new_by_curve_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2661));
  return pyresult;
}
#else
#  define _cffi_f_EC_GROUP_new_by_curve_name _cffi_d_EC_GROUP_new_by_curve_name
#endif

static void _cffi_d_EC_KEY_free(EC_KEY * x0)
{
  EC_KEY_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_free(PyObject *self, PyObject *arg0)
{
  EC_KEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_KEY_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_KEY_free _cffi_d_EC_KEY_free
#endif

static int _cffi_d_EC_KEY_generate_key(EC_KEY * x0)
{
  return EC_KEY_generate_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_generate_key(PyObject *self, PyObject *arg0)
{
  EC_KEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_generate_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_generate_key _cffi_d_EC_KEY_generate_key
#endif

static EC_GROUP const * _cffi_d_EC_KEY_get0_group(EC_KEY const * x0)
{
  return EC_KEY_get0_group(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_get0_group(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_GROUP const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_get0_group(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(238));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_get0_group _cffi_d_EC_KEY_get0_group
#endif

static BIGNUM const * _cffi_d_EC_KEY_get0_private_key(EC_KEY const * x0)
{
  return EC_KEY_get0_private_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_get0_private_key(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  BIGNUM const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_get0_private_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(92));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_get0_private_key _cffi_d_EC_KEY_get0_private_key
#endif

static EC_POINT const * _cffi_d_EC_KEY_get0_public_key(EC_KEY const * x0)
{
  return EC_KEY_get0_public_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_get0_public_key(PyObject *self, PyObject *arg0)
{
  EC_KEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(110), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(110), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_get0_public_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(244));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_get0_public_key _cffi_d_EC_KEY_get0_public_key
#endif

static EC_KEY * _cffi_d_EC_KEY_new(void)
{
  return EC_KEY_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_new(PyObject *self, PyObject *noarg)
{
  EC_KEY * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(893));
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_new _cffi_d_EC_KEY_new
#endif

static EC_KEY * _cffi_d_EC_KEY_new_by_curve_name(int x0)
{
  return EC_KEY_new_by_curve_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_new_by_curve_name(PyObject *self, PyObject *arg0)
{
  int x0;
  EC_KEY * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_new_by_curve_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(893));
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_new_by_curve_name _cffi_d_EC_KEY_new_by_curve_name
#endif

static void _cffi_d_EC_KEY_set_asn1_flag(EC_KEY * x0, int x1)
{
  EC_KEY_set_asn1_flag(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_asn1_flag(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_asn1_flag", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_KEY_set_asn1_flag(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_KEY_set_asn1_flag _cffi_d_EC_KEY_set_asn1_flag
#endif

static int _cffi_d_EC_KEY_set_group(EC_KEY * x0, EC_GROUP const * x1)
{
  return EC_KEY_set_group(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_group(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  EC_GROUP const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_group", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_group(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_group _cffi_d_EC_KEY_set_group
#endif

static int _cffi_d_EC_KEY_set_private_key(EC_KEY * x0, BIGNUM const * x1)
{
  return EC_KEY_set_private_key(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_private_key(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  BIGNUM const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_private_key", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_private_key(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_private_key _cffi_d_EC_KEY_set_private_key
#endif

static int _cffi_d_EC_KEY_set_public_key(EC_KEY * x0, EC_POINT const * x1)
{
  return EC_KEY_set_public_key(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_public_key(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  EC_POINT const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_public_key", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_public_key(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_public_key _cffi_d_EC_KEY_set_public_key
#endif

static int _cffi_d_EC_KEY_set_public_key_affine_coordinates(EC_KEY * x0, BIGNUM * x1, BIGNUM * x2)
{
  return EC_KEY_set_public_key_affine_coordinates(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_KEY_set_public_key_affine_coordinates(PyObject *self, PyObject *args)
{
  EC_KEY * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_KEY_set_public_key_affine_coordinates", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_KEY_set_public_key_affine_coordinates(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_KEY_set_public_key_affine_coordinates _cffi_d_EC_KEY_set_public_key_affine_coordinates
#endif

static int _cffi_d_EC_METHOD_get_field_type(EC_METHOD const * x0)
{
  return EC_METHOD_get_field_type(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_METHOD_get_field_type(PyObject *self, PyObject *arg0)
{
  EC_METHOD const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1298), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_METHOD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1298), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_METHOD_get_field_type(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_METHOD_get_field_type _cffi_d_EC_METHOD_get_field_type
#endif

static int _cffi_d_EC_POINT_add(EC_GROUP const * x0, EC_POINT * x1, EC_POINT const * x2, EC_POINT const * x3, BN_CTX * x4)
{
  return EC_POINT_add(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_add(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  EC_POINT const * x2;
  EC_POINT const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_add", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_add(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_add _cffi_d_EC_POINT_add
#endif

static void _cffi_d_EC_POINT_clear_free(EC_POINT * x0)
{
  EC_POINT_clear_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_clear_free(PyObject *self, PyObject *arg0)
{
  EC_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_POINT_clear_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_POINT_clear_free _cffi_d_EC_POINT_clear_free
#endif

static int _cffi_d_EC_POINT_cmp(EC_GROUP const * x0, EC_POINT const * x1, EC_POINT const * x2, BN_CTX * x3)
{
  return EC_POINT_cmp(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_cmp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  EC_POINT const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EC_POINT_cmp", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_cmp(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_cmp _cffi_d_EC_POINT_cmp
#endif

static int _cffi_d_EC_POINT_dbl(EC_GROUP const * x0, EC_POINT * x1, EC_POINT const * x2, BN_CTX * x3)
{
  return EC_POINT_dbl(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_dbl(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  EC_POINT const * x2;
  BN_CTX * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EC_POINT_dbl", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_dbl(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_dbl _cffi_d_EC_POINT_dbl
#endif

static EC_POINT * _cffi_d_EC_POINT_dup(EC_POINT const * x0, EC_GROUP const * x1)
{
  return EC_POINT_dup(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_dup(PyObject *self, PyObject *args)
{
  EC_POINT const * x0;
  EC_GROUP const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_POINT_dup", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_dup(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1207));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_dup _cffi_d_EC_POINT_dup
#endif

static void _cffi_d_EC_POINT_free(EC_POINT * x0)
{
  EC_POINT_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_free(PyObject *self, PyObject *arg0)
{
  EC_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EC_POINT_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EC_POINT_free _cffi_d_EC_POINT_free
#endif

static int _cffi_d_EC_POINT_get_affine_coordinates_GF2m(EC_GROUP const * x0, EC_POINT const * x1, BIGNUM * x2, BIGNUM * x3, BN_CTX * x4)
{
  return EC_POINT_get_affine_coordinates_GF2m(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_get_affine_coordinates_GF2m(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_get_affine_coordinates_GF2m", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_get_affine_coordinates_GF2m(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_get_affine_coordinates_GF2m _cffi_d_EC_POINT_get_affine_coordinates_GF2m
#endif

static int _cffi_d_EC_POINT_get_affine_coordinates_GFp(EC_GROUP const * x0, EC_POINT const * x1, BIGNUM * x2, BIGNUM * x3, BN_CTX * x4)
{
  return EC_POINT_get_affine_coordinates_GFp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_get_affine_coordinates_GFp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_get_affine_coordinates_GFp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_get_affine_coordinates_GFp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_get_affine_coordinates_GFp _cffi_d_EC_POINT_get_affine_coordinates_GFp
#endif

static int _cffi_d_EC_POINT_invert(EC_GROUP const * x0, EC_POINT * x1, BN_CTX * x2)
{
  return EC_POINT_invert(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_invert(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_POINT_invert", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_invert(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_invert _cffi_d_EC_POINT_invert
#endif

static int _cffi_d_EC_POINT_is_at_infinity(EC_GROUP const * x0, EC_POINT const * x1)
{
  return EC_POINT_is_at_infinity(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_is_at_infinity(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_POINT_is_at_infinity", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_is_at_infinity(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_is_at_infinity _cffi_d_EC_POINT_is_at_infinity
#endif

static int _cffi_d_EC_POINT_is_on_curve(EC_GROUP const * x0, EC_POINT const * x1, BN_CTX * x2)
{
  return EC_POINT_is_on_curve(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_is_on_curve(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  BN_CTX * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EC_POINT_is_on_curve", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_is_on_curve(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_is_on_curve _cffi_d_EC_POINT_is_on_curve
#endif

static int _cffi_d_EC_POINT_mul(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, EC_POINT const * x3, BIGNUM const * x4, BN_CTX * x5)
{
  return EC_POINT_mul(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_mul(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  EC_POINT const * x3;
  BIGNUM const * x4;
  BN_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "EC_POINT_mul", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_mul(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_mul _cffi_d_EC_POINT_mul
#endif

static EC_POINT * _cffi_d_EC_POINT_new(EC_GROUP const * x0)
{
  return EC_POINT_new(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_new(PyObject *self, PyObject *arg0)
{
  EC_GROUP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_POINT * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_new(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1207));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_new _cffi_d_EC_POINT_new
#endif

static int _cffi_d_EC_POINT_oct2point(EC_GROUP const * x0, EC_POINT * x1, unsigned char const * x2, size_t x3, BN_CTX * x4)
{
  return EC_POINT_oct2point(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_oct2point(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  unsigned char const * x2;
  size_t x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_oct2point", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_oct2point(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_oct2point _cffi_d_EC_POINT_oct2point
#endif

static size_t _cffi_d_EC_POINT_point2oct(EC_GROUP const * x0, EC_POINT const * x1, point_conversion_form_t x2, unsigned char * x3, size_t x4, BN_CTX * x5)
{
  return EC_POINT_point2oct(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_point2oct(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT const * x1;
  point_conversion_form_t x2;
  unsigned char * x3;
  size_t x4;
  BN_CTX * x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "EC_POINT_point2oct", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(244), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(244), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  if (_cffi_to_c((char *)&x2, _cffi_type(2375), arg2) < 0)
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_point2oct(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_point2oct _cffi_d_EC_POINT_point2oct
#endif

static int _cffi_d_EC_POINT_set_affine_coordinates_GF2m(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return EC_POINT_set_affine_coordinates_GF2m(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_set_affine_coordinates_GF2m(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_set_affine_coordinates_GF2m", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_set_affine_coordinates_GF2m(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_set_affine_coordinates_GF2m _cffi_d_EC_POINT_set_affine_coordinates_GF2m
#endif

static int _cffi_d_EC_POINT_set_affine_coordinates_GFp(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, BIGNUM const * x3, BN_CTX * x4)
{
  return EC_POINT_set_affine_coordinates_GFp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_set_affine_coordinates_GFp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  BIGNUM const * x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_set_affine_coordinates_GFp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_set_affine_coordinates_GFp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_set_affine_coordinates_GFp _cffi_d_EC_POINT_set_affine_coordinates_GFp
#endif

static int _cffi_d_EC_POINT_set_compressed_coordinates_GF2m(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, int x3, BN_CTX * x4)
{
  return EC_POINT_set_compressed_coordinates_GF2m(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_set_compressed_coordinates_GF2m(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  int x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_set_compressed_coordinates_GF2m", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_set_compressed_coordinates_GF2m(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_set_compressed_coordinates_GF2m _cffi_d_EC_POINT_set_compressed_coordinates_GF2m
#endif

static int _cffi_d_EC_POINT_set_compressed_coordinates_GFp(EC_GROUP const * x0, EC_POINT * x1, BIGNUM const * x2, int x3, BN_CTX * x4)
{
  return EC_POINT_set_compressed_coordinates_GFp(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_POINT_set_compressed_coordinates_GFp(PyObject *self, PyObject *args)
{
  EC_GROUP const * x0;
  EC_POINT * x1;
  BIGNUM const * x2;
  int x3;
  BN_CTX * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EC_POINT_set_compressed_coordinates_GFp", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(238), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_GROUP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(238), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1207), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1207), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(92), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(92), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_POINT_set_compressed_coordinates_GFp(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_POINT_set_compressed_coordinates_GFp _cffi_d_EC_POINT_set_compressed_coordinates_GFp
#endif

static char const * _cffi_d_EC_curve_nid2nist(int x0)
{
  return EC_curve_nid2nist(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_curve_nid2nist(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_curve_nid2nist(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  return pyresult;
}
#else
#  define _cffi_f_EC_curve_nid2nist _cffi_d_EC_curve_nid2nist
#endif

static size_t _cffi_d_EC_get_builtin_curves(EC_builtin_curve * x0, size_t x1)
{
  return EC_get_builtin_curves(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EC_get_builtin_curves(PyObject *self, PyObject *args)
{
  EC_builtin_curve * x0;
  size_t x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EC_get_builtin_curves", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2381), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EC_builtin_curve *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2381), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EC_get_builtin_curves(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EC_get_builtin_curves _cffi_d_EC_get_builtin_curves
#endif

static ENGINE * _cffi_d_ENGINE_by_id(char const * x0)
{
  return ENGINE_by_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_by_id(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ENGINE * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_by_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(282));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_by_id _cffi_d_ENGINE_by_id
#endif

static int _cffi_d_ENGINE_ctrl_cmd(ENGINE * x0, char const * x1, long x2, void * x3, void(* x4)(void), int x5)
{
  return ENGINE_ctrl_cmd(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_ctrl_cmd(PyObject *self, PyObject *args)
{
  ENGINE * x0;
  char const * x1;
  long x2;
  void * x3;
  void(* x4)(void);
  int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "ENGINE_ctrl_cmd", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, long);
  if (x2 == (long)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = (void(*)(void))_cffi_to_c_pointer(arg4, _cffi_type(1314));
  if (x4 == (void(*)(void))NULL && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_ctrl_cmd(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_ctrl_cmd _cffi_d_ENGINE_ctrl_cmd
#endif

static int _cffi_d_ENGINE_ctrl_cmd_string(ENGINE * x0, char const * x1, char const * x2, int x3)
{
  return ENGINE_ctrl_cmd_string(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_ctrl_cmd_string(PyObject *self, PyObject *args)
{
  ENGINE * x0;
  char const * x1;
  char const * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "ENGINE_ctrl_cmd_string", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_ctrl_cmd_string(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_ctrl_cmd_string _cffi_d_ENGINE_ctrl_cmd_string
#endif

static int _cffi_d_ENGINE_finish(ENGINE * x0)
{
  return ENGINE_finish(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_finish(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_finish(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_finish _cffi_d_ENGINE_finish
#endif

static int _cffi_d_ENGINE_free(ENGINE * x0)
{
  return ENGINE_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_free(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_free _cffi_d_ENGINE_free
#endif

static ENGINE * _cffi_d_ENGINE_get_default_RAND(void)
{
  return ENGINE_get_default_RAND();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_get_default_RAND(PyObject *self, PyObject *noarg)
{
  ENGINE * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_get_default_RAND(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(282));
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_get_default_RAND _cffi_d_ENGINE_get_default_RAND
#endif

static char const * _cffi_d_ENGINE_get_name(ENGINE const * x0)
{
  return ENGINE_get_name(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_get_name(PyObject *self, PyObject *arg0)
{
  ENGINE const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(723), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(723), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_get_name(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_get_name _cffi_d_ENGINE_get_name
#endif

static int _cffi_d_ENGINE_init(ENGINE * x0)
{
  return ENGINE_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_init(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_init _cffi_d_ENGINE_init
#endif

static void _cffi_d_ENGINE_load_builtin_engines(void)
{
  ENGINE_load_builtin_engines();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_load_builtin_engines(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ENGINE_load_builtin_engines(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ENGINE_load_builtin_engines _cffi_d_ENGINE_load_builtin_engines
#endif

static EVP_PKEY * _cffi_d_ENGINE_load_private_key(ENGINE * x0, char const * x1, UI_METHOD * x2, void * x3)
{
  return ENGINE_load_private_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_load_private_key(PyObject *self, PyObject *args)
{
  ENGINE * x0;
  char const * x1;
  UI_METHOD * x2;
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "ENGINE_load_private_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(284), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (UI_METHOD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(284), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_load_private_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(199));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_load_private_key _cffi_d_ENGINE_load_private_key
#endif

static EVP_PKEY * _cffi_d_ENGINE_load_public_key(ENGINE * x0, char const * x1, UI_METHOD * x2, void * x3)
{
  return ENGINE_load_public_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_load_public_key(PyObject *self, PyObject *args)
{
  ENGINE * x0;
  char const * x1;
  UI_METHOD * x2;
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "ENGINE_load_public_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(284), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (UI_METHOD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(284), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_load_public_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(199));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_load_public_key _cffi_d_ENGINE_load_public_key
#endif

static int _cffi_d_ENGINE_set_default_RAND(ENGINE * x0)
{
  return ENGINE_set_default_RAND(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_set_default_RAND(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ENGINE_set_default_RAND(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_ENGINE_set_default_RAND _cffi_d_ENGINE_set_default_RAND
#endif

static void _cffi_d_ENGINE_unregister_RAND(ENGINE * x0)
{
  ENGINE_unregister_RAND(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ENGINE_unregister_RAND(PyObject *self, PyObject *arg0)
{
  ENGINE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ENGINE_unregister_RAND(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ENGINE_unregister_RAND _cffi_d_ENGINE_unregister_RAND
#endif

static int _cffi_d_ERR_GET_FUNC(unsigned long x0)
{
  return ERR_GET_FUNC(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_GET_FUNC(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_GET_FUNC(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_ERR_GET_FUNC _cffi_d_ERR_GET_FUNC
#endif

static int _cffi_d_ERR_GET_LIB(unsigned long x0)
{
  return ERR_GET_LIB(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_GET_LIB(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_GET_LIB(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_ERR_GET_LIB _cffi_d_ERR_GET_LIB
#endif

static int _cffi_d_ERR_GET_REASON(unsigned long x0)
{
  return ERR_GET_REASON(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_GET_REASON(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_GET_REASON(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_ERR_GET_REASON _cffi_d_ERR_GET_REASON
#endif

static void _cffi_d_ERR_clear_error(void)
{
  ERR_clear_error();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_clear_error(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ERR_clear_error(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ERR_clear_error _cffi_d_ERR_clear_error
#endif

static void _cffi_d_ERR_error_string_n(unsigned long x0, char * x1, size_t x2)
{
  ERR_error_string_n(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_error_string_n(PyObject *self, PyObject *args)
{
  unsigned long x0;
  char * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "ERR_error_string_n", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(409), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(409), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ERR_error_string_n(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ERR_error_string_n _cffi_d_ERR_error_string_n
#endif

static char const * _cffi_d_ERR_func_error_string(unsigned long x0)
{
  return ERR_func_error_string(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_func_error_string(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_func_error_string(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  return pyresult;
}
#else
#  define _cffi_f_ERR_func_error_string _cffi_d_ERR_func_error_string
#endif

static unsigned long _cffi_d_ERR_get_error(void)
{
  return ERR_get_error();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_get_error(PyObject *self, PyObject *noarg)
{
  unsigned long result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_get_error(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  return pyresult;
}
#else
#  define _cffi_f_ERR_get_error _cffi_d_ERR_get_error
#endif

static char const * _cffi_d_ERR_lib_error_string(unsigned long x0)
{
  return ERR_lib_error_string(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_lib_error_string(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_lib_error_string(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  return pyresult;
}
#else
#  define _cffi_f_ERR_lib_error_string _cffi_d_ERR_lib_error_string
#endif

static void _cffi_d_ERR_load_RAND_strings(void)
{
  ERR_load_RAND_strings();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_load_RAND_strings(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ERR_load_RAND_strings(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ERR_load_RAND_strings _cffi_d_ERR_load_RAND_strings
#endif

static unsigned long _cffi_d_ERR_peek_error(void)
{
  return ERR_peek_error();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_peek_error(PyObject *self, PyObject *noarg)
{
  unsigned long result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_peek_error(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  return pyresult;
}
#else
#  define _cffi_f_ERR_peek_error _cffi_d_ERR_peek_error
#endif

static void _cffi_d_ERR_put_error(int x0, int x1, int x2, char const * x3, int x4)
{
  ERR_put_error(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_put_error(PyObject *self, PyObject *args)
{
  int x0;
  int x1;
  int x2;
  char const * x3;
  int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "ERR_put_error", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ERR_put_error(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ERR_put_error _cffi_d_ERR_put_error
#endif

static char const * _cffi_d_ERR_reason_error_string(unsigned long x0)
{
  return ERR_reason_error_string(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ERR_reason_error_string(PyObject *self, PyObject *arg0)
{
  unsigned long x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, unsigned long);
  if (x0 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ERR_reason_error_string(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  return pyresult;
}
#else
#  define _cffi_f_ERR_reason_error_string _cffi_d_ERR_reason_error_string
#endif

static EVP_CIPHER const * _cffi_d_EVP_CIPHER_CTX_cipher(EVP_CIPHER_CTX const * x0)
{
  return EVP_CIPHER_CTX_cipher(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_cipher(PyObject *self, PyObject *arg0)
{
  EVP_CIPHER_CTX const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_CIPHER const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(259), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(259), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_cipher(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(885));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_cipher _cffi_d_EVP_CIPHER_CTX_cipher
#endif

static int _cffi_d_EVP_CIPHER_CTX_cleanup(EVP_CIPHER_CTX * x0)
{
  return EVP_CIPHER_CTX_cleanup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_cleanup(PyObject *self, PyObject *arg0)
{
  EVP_CIPHER_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1318), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1318), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_cleanup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_cleanup _cffi_d_EVP_CIPHER_CTX_cleanup
#endif

static int _cffi_d_EVP_CIPHER_CTX_ctrl(EVP_CIPHER_CTX * x0, int x1, int x2, void * x3)
{
  return EVP_CIPHER_CTX_ctrl(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_ctrl(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  int x1;
  int x2;
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_CIPHER_CTX_ctrl", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1318), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1318), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_ctrl(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_ctrl _cffi_d_EVP_CIPHER_CTX_ctrl
#endif

static void _cffi_d_EVP_CIPHER_CTX_free(EVP_CIPHER_CTX * x0)
{
  EVP_CIPHER_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_CIPHER_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1318), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1318), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_CIPHER_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_free _cffi_d_EVP_CIPHER_CTX_free
#endif

static EVP_CIPHER_CTX * _cffi_d_EVP_CIPHER_CTX_new(void)
{
  return EVP_CIPHER_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_new(PyObject *self, PyObject *noarg)
{
  EVP_CIPHER_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1318));
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_new _cffi_d_EVP_CIPHER_CTX_new
#endif

static int _cffi_d_EVP_CIPHER_CTX_set_key_length(EVP_CIPHER_CTX * x0, int x1)
{
  return EVP_CIPHER_CTX_set_key_length(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_set_key_length(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_CIPHER_CTX_set_key_length", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1318), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1318), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_set_key_length(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_set_key_length _cffi_d_EVP_CIPHER_CTX_set_key_length
#endif

static int _cffi_d_EVP_CIPHER_CTX_set_padding(EVP_CIPHER_CTX * x0, int x1)
{
  return EVP_CIPHER_CTX_set_padding(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CIPHER_CTX_set_padding(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_CIPHER_CTX_set_padding", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1318), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1318), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CIPHER_CTX_set_padding(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CIPHER_CTX_set_padding _cffi_d_EVP_CIPHER_CTX_set_padding
#endif

static int _cffi_d_EVP_CipherFinal_ex(EVP_CIPHER_CTX * x0, unsigned char * x1, int * x2)
{
  return EVP_CipherFinal_ex(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CipherFinal_ex(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  unsigned char * x1;
  int * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_CipherFinal_ex", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1318), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1318), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1169), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1169), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CipherFinal_ex(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CipherFinal_ex _cffi_d_EVP_CipherFinal_ex
#endif

static int _cffi_d_EVP_CipherInit_ex(EVP_CIPHER_CTX * x0, EVP_CIPHER const * x1, ENGINE * x2, unsigned char const * x3, unsigned char const * x4, int x5)
{
  return EVP_CipherInit_ex(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CipherInit_ex(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  EVP_CIPHER const * x1;
  ENGINE * x2;
  unsigned char const * x3;
  unsigned char const * x4;
  int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "EVP_CipherInit_ex", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1318), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1318), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(885), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(885), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CipherInit_ex(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CipherInit_ex _cffi_d_EVP_CipherInit_ex
#endif

static int _cffi_d_EVP_CipherUpdate(EVP_CIPHER_CTX * x0, unsigned char * x1, int * x2, unsigned char const * x3, int x4)
{
  return EVP_CipherUpdate(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_CipherUpdate(PyObject *self, PyObject *args)
{
  EVP_CIPHER_CTX * x0;
  unsigned char * x1;
  int * x2;
  unsigned char const * x3;
  int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_CipherUpdate", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1318), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_CIPHER_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1318), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1169), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1169), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_CipherUpdate(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_CipherUpdate _cffi_d_EVP_CipherUpdate
#endif

static int _cffi_d_EVP_DigestFinalXOF(EVP_MD_CTX * x0, unsigned char * x1, size_t x2)
{
  return EVP_DigestFinalXOF(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestFinalXOF(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestFinalXOF", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestFinalXOF(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestFinalXOF _cffi_d_EVP_DigestFinalXOF
#endif

static int _cffi_d_EVP_DigestFinal_ex(EVP_MD_CTX * x0, unsigned char * x1, unsigned int * x2)
{
  return EVP_DigestFinal_ex(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestFinal_ex(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  unsigned int * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestFinal_ex", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1390), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1390), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestFinal_ex(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestFinal_ex _cffi_d_EVP_DigestFinal_ex
#endif

static int _cffi_d_EVP_DigestInit_ex(EVP_MD_CTX * x0, EVP_MD const * x1, ENGINE * x2)
{
  return EVP_DigestInit_ex(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestInit_ex(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD const * x1;
  ENGINE * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestInit_ex", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestInit_ex(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestInit_ex _cffi_d_EVP_DigestInit_ex
#endif

static int _cffi_d_EVP_DigestSign(EVP_MD_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_DigestSign(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSign(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSign", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSign(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSign _cffi_d_EVP_DigestSign
#endif

static int _cffi_d_EVP_DigestSignFinal(EVP_MD_CTX * x0, unsigned char * x1, size_t * x2)
{
  return EVP_DigestSignFinal(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSignFinal(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSignFinal", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSignFinal(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSignFinal _cffi_d_EVP_DigestSignFinal
#endif

static int _cffi_d_EVP_DigestSignInit(EVP_MD_CTX * x0, EVP_PKEY_CTX * * x1, EVP_MD const * x2, ENGINE * x3, EVP_PKEY * x4)
{
  return EVP_DigestSignInit(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSignInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_PKEY_CTX * * x1;
  EVP_MD const * x2;
  ENGINE * x3;
  EVP_PKEY * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSignInit", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1365), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1365), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSignInit(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSignInit _cffi_d_EVP_DigestSignInit
#endif

static int _cffi_d_EVP_DigestSignUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_DigestSignUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestSignUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestSignUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestSignUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestSignUpdate _cffi_d_EVP_DigestSignUpdate
#endif

static int _cffi_d_EVP_DigestUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_DigestUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_DigestUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestUpdate _cffi_d_EVP_DigestUpdate
#endif

static int _cffi_d_EVP_DigestVerify(EVP_MD_CTX * x0, unsigned char const * x1, size_t x2, unsigned char const * x3, size_t x4)
{
  return EVP_DigestVerify(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestVerify(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char const * x1;
  size_t x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestVerify", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestVerify(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestVerify _cffi_d_EVP_DigestVerify
#endif

static int _cffi_d_EVP_DigestVerifyInit(EVP_MD_CTX * x0, EVP_PKEY_CTX * * x1, EVP_MD const * x2, ENGINE * x3, EVP_PKEY * x4)
{
  return EVP_DigestVerifyInit(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_DigestVerifyInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_PKEY_CTX * * x1;
  EVP_MD const * x2;
  ENGINE * x3;
  EVP_PKEY * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_DigestVerifyInit", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1365), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1365), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_DigestVerifyInit(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_DigestVerifyInit _cffi_d_EVP_DigestVerifyInit
#endif

static int _cffi_d_EVP_MD_CTX_copy_ex(EVP_MD_CTX * x0, EVP_MD_CTX const * x1)
{
  return EVP_MD_CTX_copy_ex(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_MD_CTX_copy_ex(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD_CTX const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_MD_CTX_copy_ex", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1361), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1361), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_MD_CTX_copy_ex(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_MD_CTX_copy_ex _cffi_d_EVP_MD_CTX_copy_ex
#endif

static void _cffi_d_EVP_MD_CTX_free(EVP_MD_CTX * x0)
{
  EVP_MD_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_MD_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_MD_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_MD_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_MD_CTX_free _cffi_d_EVP_MD_CTX_free
#endif

static EVP_MD_CTX * _cffi_d_EVP_MD_CTX_new(void)
{
  return EVP_MD_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_MD_CTX_new(PyObject *self, PyObject *noarg)
{
  EVP_MD_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_MD_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1351));
  return pyresult;
}
#else
#  define _cffi_f_EVP_MD_CTX_new _cffi_d_EVP_MD_CTX_new
#endif

static int _cffi_d_EVP_PBE_scrypt(char const * x0, size_t x1, unsigned char const * x2, size_t x3, uint64_t x4, uint64_t x5, uint64_t x6, uint64_t x7, unsigned char * x8, size_t x9)
{
  return EVP_PBE_scrypt(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PBE_scrypt(PyObject *self, PyObject *args)
{
  char const * x0;
  size_t x1;
  unsigned char const * x2;
  size_t x3;
  uint64_t x4;
  uint64_t x5;
  uint64_t x6;
  uint64_t x7;
  unsigned char * x8;
  size_t x9;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;

  if (!PyArg_UnpackTuple(args, "EVP_PBE_scrypt", 10, 10, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, size_t);
  if (x1 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, uint64_t);
  if (x4 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  x5 = _cffi_to_c_int(arg5, uint64_t);
  if (x5 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, uint64_t);
  if (x6 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  x7 = _cffi_to_c_int(arg7, uint64_t);
  if (x7 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg8, (char **)&x8);
  if (datasize != 0) {
    x8 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg8, (char **)&x8,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x9 = _cffi_to_c_int(arg9, size_t);
  if (x9 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PBE_scrypt(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PBE_scrypt _cffi_d_EVP_PBE_scrypt
#endif

static EVP_PKEY_CTX * _cffi_d_EVP_PKEY_CTX_dup(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_CTX_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_dup(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY_CTX * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(309));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_dup _cffi_d_EVP_PKEY_CTX_dup
#endif

static void _cffi_d_EVP_PKEY_CTX_free(EVP_PKEY_CTX * x0)
{
  EVP_PKEY_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_free(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_PKEY_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_free _cffi_d_EVP_PKEY_CTX_free
#endif

static EVP_PKEY_CTX * _cffi_d_EVP_PKEY_CTX_new(EVP_PKEY * x0, ENGINE * x1)
{
  return EVP_PKEY_CTX_new(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_new(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  ENGINE * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY_CTX * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_new", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_new(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(309));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_new _cffi_d_EVP_PKEY_CTX_new
#endif

static EVP_PKEY_CTX * _cffi_d_EVP_PKEY_CTX_new_id(int x0, ENGINE * x1)
{
  return EVP_PKEY_CTX_new_id(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_new_id(PyObject *self, PyObject *args)
{
  int x0;
  ENGINE * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY_CTX * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_new_id", 2, 2, &arg0, &arg1))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_new_id(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(309));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_new_id _cffi_d_EVP_PKEY_CTX_new_id
#endif

static int _cffi_d_EVP_PKEY_CTX_set0_rsa_oaep_label(EVP_PKEY_CTX * x0, unsigned char * x1, int x2)
{
  return EVP_PKEY_CTX_set0_rsa_oaep_label(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set0_rsa_oaep_label(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set0_rsa_oaep_label", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set0_rsa_oaep_label(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set0_rsa_oaep_label _cffi_d_EVP_PKEY_CTX_set0_rsa_oaep_label
#endif

static int _cffi_d_EVP_PKEY_CTX_set_rsa_mgf1_md(EVP_PKEY_CTX * x0, EVP_MD * x1)
{
  return EVP_PKEY_CTX_set_rsa_mgf1_md(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_rsa_mgf1_md(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_MD * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_rsa_mgf1_md", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1461), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1461), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_rsa_mgf1_md(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_rsa_mgf1_md _cffi_d_EVP_PKEY_CTX_set_rsa_mgf1_md
#endif

static int _cffi_d_EVP_PKEY_CTX_set_rsa_oaep_md(EVP_PKEY_CTX * x0, EVP_MD * x1)
{
  return EVP_PKEY_CTX_set_rsa_oaep_md(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_rsa_oaep_md(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_MD * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_rsa_oaep_md", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1461), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1461), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_rsa_oaep_md(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_rsa_oaep_md _cffi_d_EVP_PKEY_CTX_set_rsa_oaep_md
#endif

static int _cffi_d_EVP_PKEY_CTX_set_rsa_padding(EVP_PKEY_CTX * x0, int x1)
{
  return EVP_PKEY_CTX_set_rsa_padding(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_rsa_padding(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_rsa_padding", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_rsa_padding(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_rsa_padding _cffi_d_EVP_PKEY_CTX_set_rsa_padding
#endif

static int _cffi_d_EVP_PKEY_CTX_set_rsa_pss_saltlen(EVP_PKEY_CTX * x0, int x1)
{
  return EVP_PKEY_CTX_set_rsa_pss_saltlen(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_rsa_pss_saltlen(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_rsa_pss_saltlen", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_rsa_pss_saltlen(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_rsa_pss_saltlen _cffi_d_EVP_PKEY_CTX_set_rsa_pss_saltlen
#endif

static int _cffi_d_EVP_PKEY_CTX_set_signature_md(EVP_PKEY_CTX * x0, EVP_MD const * x1)
{
  return EVP_PKEY_CTX_set_signature_md(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_CTX_set_signature_md(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_MD const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_CTX_set_signature_md", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_CTX_set_signature_md(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_CTX_set_signature_md _cffi_d_EVP_PKEY_CTX_set_signature_md
#endif

static int _cffi_d_EVP_PKEY_assign_RSA(EVP_PKEY * x0, RSA * x1)
{
  return EVP_PKEY_assign_RSA(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_assign_RSA(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  RSA * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_assign_RSA", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(476), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(476), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_assign_RSA(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_assign_RSA _cffi_d_EVP_PKEY_assign_RSA
#endif

static int _cffi_d_EVP_PKEY_bits(EVP_PKEY * x0)
{
  return EVP_PKEY_bits(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_bits(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_bits(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_bits _cffi_d_EVP_PKEY_bits
#endif

static int _cffi_d_EVP_PKEY_cmp(EVP_PKEY const * x0, EVP_PKEY const * x1)
{
  return EVP_PKEY_cmp(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_cmp(PyObject *self, PyObject *args)
{
  EVP_PKEY const * x0;
  EVP_PKEY const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_cmp", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1445), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1445), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1445), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1445), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_cmp(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_cmp _cffi_d_EVP_PKEY_cmp
#endif

static int _cffi_d_EVP_PKEY_decrypt(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_decrypt(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_decrypt(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_decrypt", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_decrypt(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_decrypt _cffi_d_EVP_PKEY_decrypt
#endif

static int _cffi_d_EVP_PKEY_decrypt_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_decrypt_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_decrypt_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_decrypt_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_decrypt_init _cffi_d_EVP_PKEY_decrypt_init
#endif

static int _cffi_d_EVP_PKEY_derive(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2)
{
  return EVP_PKEY_derive(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_derive(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_derive", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_derive(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_derive _cffi_d_EVP_PKEY_derive
#endif

static int _cffi_d_EVP_PKEY_derive_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_derive_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_derive_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_derive_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_derive_init _cffi_d_EVP_PKEY_derive_init
#endif

static int _cffi_d_EVP_PKEY_derive_set_peer(EVP_PKEY_CTX * x0, EVP_PKEY * x1)
{
  return EVP_PKEY_derive_set_peer(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_derive_set_peer(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_derive_set_peer", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_derive_set_peer(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_derive_set_peer _cffi_d_EVP_PKEY_derive_set_peer
#endif

static int _cffi_d_EVP_PKEY_encrypt(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_encrypt(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_encrypt(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_encrypt", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_encrypt(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_encrypt _cffi_d_EVP_PKEY_encrypt
#endif

static int _cffi_d_EVP_PKEY_encrypt_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_encrypt_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_encrypt_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_encrypt_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_encrypt_init _cffi_d_EVP_PKEY_encrypt_init
#endif

static void _cffi_d_EVP_PKEY_free(EVP_PKEY * x0)
{
  EVP_PKEY_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_free(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { EVP_PKEY_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_EVP_PKEY_free _cffi_d_EVP_PKEY_free
#endif

static DH * _cffi_d_EVP_PKEY_get1_DH(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_DH(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_DH(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_DH(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(196));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_DH _cffi_d_EVP_PKEY_get1_DH
#endif

static DSA * _cffi_d_EVP_PKEY_get1_DSA(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_DSA(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_DSA(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_DSA(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(212));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_DSA _cffi_d_EVP_PKEY_get1_DSA
#endif

static EC_KEY * _cffi_d_EVP_PKEY_get1_EC_KEY(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_EC_KEY(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_EC_KEY(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EC_KEY * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_EC_KEY(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(893));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_EC_KEY _cffi_d_EVP_PKEY_get1_EC_KEY
#endif

static RSA * _cffi_d_EVP_PKEY_get1_RSA(EVP_PKEY * x0)
{
  return EVP_PKEY_get1_RSA(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_RSA(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  RSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_RSA(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(476));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_RSA _cffi_d_EVP_PKEY_get1_RSA
#endif

static size_t _cffi_d_EVP_PKEY_get1_tls_encodedpoint(EVP_PKEY * x0, unsigned char * * x1)
{
  return EVP_PKEY_get1_tls_encodedpoint(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get1_tls_encodedpoint(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  unsigned char * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_get1_tls_encodedpoint", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(802), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(802), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get1_tls_encodedpoint(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get1_tls_encodedpoint _cffi_d_EVP_PKEY_get1_tls_encodedpoint
#endif

static int _cffi_d_EVP_PKEY_get_raw_private_key(EVP_PKEY const * x0, unsigned char * x1, size_t * x2)
{
  return EVP_PKEY_get_raw_private_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get_raw_private_key(PyObject *self, PyObject *args)
{
  EVP_PKEY const * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_get_raw_private_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1445), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1445), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get_raw_private_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get_raw_private_key _cffi_d_EVP_PKEY_get_raw_private_key
#endif

static int _cffi_d_EVP_PKEY_get_raw_public_key(EVP_PKEY const * x0, unsigned char * x1, size_t * x2)
{
  return EVP_PKEY_get_raw_public_key(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_get_raw_public_key(PyObject *self, PyObject *args)
{
  EVP_PKEY const * x0;
  unsigned char * x1;
  size_t * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_get_raw_public_key", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1445), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1445), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_get_raw_public_key(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_get_raw_public_key _cffi_d_EVP_PKEY_get_raw_public_key
#endif

static int _cffi_d_EVP_PKEY_id(EVP_PKEY const * x0)
{
  return EVP_PKEY_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_id(PyObject *self, PyObject *arg0)
{
  EVP_PKEY const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1445), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1445), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_id _cffi_d_EVP_PKEY_id
#endif

static int _cffi_d_EVP_PKEY_keygen(EVP_PKEY_CTX * x0, EVP_PKEY * * x1)
{
  return EVP_PKEY_keygen(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_keygen(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  EVP_PKEY * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_keygen", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(273), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(273), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_keygen(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_keygen _cffi_d_EVP_PKEY_keygen
#endif

static int _cffi_d_EVP_PKEY_keygen_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_keygen_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_keygen_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_keygen_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_keygen_init _cffi_d_EVP_PKEY_keygen_init
#endif

static EVP_PKEY * _cffi_d_EVP_PKEY_new(void)
{
  return EVP_PKEY_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_new(PyObject *self, PyObject *noarg)
{
  EVP_PKEY * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(199));
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_new _cffi_d_EVP_PKEY_new
#endif

static EVP_PKEY * _cffi_d_EVP_PKEY_new_raw_private_key(int x0, ENGINE * x1, unsigned char const * x2, size_t x3)
{
  return EVP_PKEY_new_raw_private_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_new_raw_private_key(PyObject *self, PyObject *args)
{
  int x0;
  ENGINE * x1;
  unsigned char const * x2;
  size_t x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_new_raw_private_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_new_raw_private_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(199));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_new_raw_private_key _cffi_d_EVP_PKEY_new_raw_private_key
#endif

static EVP_PKEY * _cffi_d_EVP_PKEY_new_raw_public_key(int x0, ENGINE * x1, unsigned char const * x2, size_t x3)
{
  return EVP_PKEY_new_raw_public_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_new_raw_public_key(PyObject *self, PyObject *args)
{
  int x0;
  ENGINE * x1;
  unsigned char const * x2;
  size_t x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_new_raw_public_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, size_t);
  if (x3 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_new_raw_public_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(199));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_new_raw_public_key _cffi_d_EVP_PKEY_new_raw_public_key
#endif

static int _cffi_d_EVP_PKEY_set1_DH(EVP_PKEY * x0, DH * x1)
{
  return EVP_PKEY_set1_DH(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_DH(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_DH", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_DH(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_DH _cffi_d_EVP_PKEY_set1_DH
#endif

static int _cffi_d_EVP_PKEY_set1_DSA(EVP_PKEY * x0, DSA * x1)
{
  return EVP_PKEY_set1_DSA(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_DSA(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  DSA * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_DSA", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(212), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(212), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_DSA(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_DSA _cffi_d_EVP_PKEY_set1_DSA
#endif

static int _cffi_d_EVP_PKEY_set1_EC_KEY(EVP_PKEY * x0, EC_KEY * x1)
{
  return EVP_PKEY_set1_EC_KEY(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_EC_KEY(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  EC_KEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_EC_KEY", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_EC_KEY(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_EC_KEY _cffi_d_EVP_PKEY_set1_EC_KEY
#endif

static int _cffi_d_EVP_PKEY_set1_RSA(EVP_PKEY * x0, RSA * x1)
{
  return EVP_PKEY_set1_RSA(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_RSA(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  RSA * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_RSA", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(476), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(476), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_RSA(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_RSA _cffi_d_EVP_PKEY_set1_RSA
#endif

static int _cffi_d_EVP_PKEY_set1_tls_encodedpoint(EVP_PKEY * x0, unsigned char const * x1, size_t x2)
{
  return EVP_PKEY_set1_tls_encodedpoint(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set1_tls_encodedpoint(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  unsigned char const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set1_tls_encodedpoint", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set1_tls_encodedpoint(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set1_tls_encodedpoint _cffi_d_EVP_PKEY_set1_tls_encodedpoint
#endif

static int _cffi_d_EVP_PKEY_set_type(EVP_PKEY * x0, int x1)
{
  return EVP_PKEY_set_type(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_set_type(PyObject *self, PyObject *args)
{
  EVP_PKEY * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_set_type", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_set_type(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_set_type _cffi_d_EVP_PKEY_set_type
#endif

static int _cffi_d_EVP_PKEY_sign(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_sign(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_sign(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_sign", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_sign(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_sign _cffi_d_EVP_PKEY_sign
#endif

static int _cffi_d_EVP_PKEY_sign_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_sign_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_sign_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_sign_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_sign_init _cffi_d_EVP_PKEY_sign_init
#endif

static int _cffi_d_EVP_PKEY_size(EVP_PKEY * x0)
{
  return EVP_PKEY_size(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_size(PyObject *self, PyObject *arg0)
{
  EVP_PKEY * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_size(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_size _cffi_d_EVP_PKEY_size
#endif

static int _cffi_d_EVP_PKEY_type(int x0)
{
  return EVP_PKEY_type(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_type(PyObject *self, PyObject *arg0)
{
  int x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_type(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_type _cffi_d_EVP_PKEY_type
#endif

static int _cffi_d_EVP_PKEY_verify(EVP_PKEY_CTX * x0, unsigned char const * x1, size_t x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_verify(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_verify(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char const * x1;
  size_t x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_verify", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_verify(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_verify _cffi_d_EVP_PKEY_verify
#endif

static int _cffi_d_EVP_PKEY_verify_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_verify_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_verify_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_verify_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_verify_init _cffi_d_EVP_PKEY_verify_init
#endif

static int _cffi_d_EVP_PKEY_verify_recover(EVP_PKEY_CTX * x0, unsigned char * x1, size_t * x2, unsigned char const * x3, size_t x4)
{
  return EVP_PKEY_verify_recover(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_verify_recover(PyObject *self, PyObject *args)
{
  EVP_PKEY_CTX * x0;
  unsigned char * x1;
  size_t * x2;
  unsigned char const * x3;
  size_t x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "EVP_PKEY_verify_recover", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1037), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (size_t *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1037), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, size_t);
  if (x4 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_verify_recover(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_verify_recover _cffi_d_EVP_PKEY_verify_recover
#endif

static int _cffi_d_EVP_PKEY_verify_recover_init(EVP_PKEY_CTX * x0)
{
  return EVP_PKEY_verify_recover_init(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_PKEY_verify_recover_init(PyObject *self, PyObject *arg0)
{
  EVP_PKEY_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(309), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_PKEY_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(309), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_PKEY_verify_recover_init(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_PKEY_verify_recover_init _cffi_d_EVP_PKEY_verify_recover_init
#endif

static int _cffi_d_EVP_SignFinal(EVP_MD_CTX * x0, unsigned char * x1, unsigned int * x2, EVP_PKEY * x3)
{
  return EVP_SignFinal(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_SignFinal(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char * x1;
  unsigned int * x2;
  EVP_PKEY * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_SignFinal", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1390), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1390), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_SignFinal(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_SignFinal _cffi_d_EVP_SignFinal
#endif

static int _cffi_d_EVP_SignInit(EVP_MD_CTX * x0, EVP_MD const * x1)
{
  return EVP_SignInit(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_SignInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_SignInit", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_SignInit(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_SignInit _cffi_d_EVP_SignInit
#endif

static int _cffi_d_EVP_SignUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_SignUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_SignUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_SignUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_SignUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_SignUpdate _cffi_d_EVP_SignUpdate
#endif

static int _cffi_d_EVP_VerifyFinal(EVP_MD_CTX * x0, unsigned char const * x1, unsigned int x2, EVP_PKEY * x3)
{
  return EVP_VerifyFinal(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_VerifyFinal(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  unsigned char const * x1;
  unsigned int x2;
  EVP_PKEY * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "EVP_VerifyFinal", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, unsigned int);
  if (x2 == (unsigned int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_VerifyFinal(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_VerifyFinal _cffi_d_EVP_VerifyFinal
#endif

static int _cffi_d_EVP_VerifyInit(EVP_MD_CTX * x0, EVP_MD const * x1)
{
  return EVP_VerifyInit(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_VerifyInit(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  EVP_MD const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "EVP_VerifyInit", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_VerifyInit(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_VerifyInit _cffi_d_EVP_VerifyInit
#endif

static int _cffi_d_EVP_VerifyUpdate(EVP_MD_CTX * x0, void const * x1, size_t x2)
{
  return EVP_VerifyUpdate(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_VerifyUpdate(PyObject *self, PyObject *args)
{
  EVP_MD_CTX * x0;
  void const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "EVP_VerifyUpdate", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1351), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1351), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_VerifyUpdate(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_VerifyUpdate _cffi_d_EVP_VerifyUpdate
#endif

static EVP_CIPHER const * _cffi_d_EVP_get_cipherbyname(char const * x0)
{
  return EVP_get_cipherbyname(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_get_cipherbyname(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_CIPHER const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_get_cipherbyname(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(885));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_get_cipherbyname _cffi_d_EVP_get_cipherbyname
#endif

static EVP_MD const * _cffi_d_EVP_get_digestbyname(char const * x0)
{
  return EVP_get_digestbyname(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_EVP_get_digestbyname(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_MD const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = EVP_get_digestbyname(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(354));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_EVP_get_digestbyname _cffi_d_EVP_get_digestbyname
#endif

static int _cffi_d_FIPS_mode(void)
{
  return FIPS_mode();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_FIPS_mode(PyObject *self, PyObject *noarg)
{
  int result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = FIPS_mode(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_FIPS_mode _cffi_d_FIPS_mode
#endif

static int _cffi_d_FIPS_mode_set(int x0)
{
  return FIPS_mode_set(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_FIPS_mode_set(PyObject *self, PyObject *arg0)
{
  int x0;
  int result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = FIPS_mode_set(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_FIPS_mode_set _cffi_d_FIPS_mode_set
#endif

static void _cffi_d_GENERAL_NAMES_free(GENERAL_NAMES * x0)
{
  GENERAL_NAMES_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAMES_free(PyObject *self, PyObject *arg0)
{
  GENERAL_NAMES * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(316), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (GENERAL_NAMES *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(316), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { GENERAL_NAMES_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_GENERAL_NAMES_free _cffi_d_GENERAL_NAMES_free
#endif

static GENERAL_NAMES * _cffi_d_GENERAL_NAMES_new(void)
{
  return GENERAL_NAMES_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAMES_new(PyObject *self, PyObject *noarg)
{
  GENERAL_NAMES * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = GENERAL_NAMES_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(316));
  return pyresult;
}
#else
#  define _cffi_f_GENERAL_NAMES_new _cffi_d_GENERAL_NAMES_new
#endif

static void _cffi_d_GENERAL_NAME_free(GENERAL_NAME * x0)
{
  GENERAL_NAME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAME_free(PyObject *self, PyObject *arg0)
{
  GENERAL_NAME * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(928), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (GENERAL_NAME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(928), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { GENERAL_NAME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_GENERAL_NAME_free _cffi_d_GENERAL_NAME_free
#endif

static GENERAL_NAME * _cffi_d_GENERAL_NAME_new(void)
{
  return GENERAL_NAME_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAME_new(PyObject *self, PyObject *noarg)
{
  GENERAL_NAME * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = GENERAL_NAME_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(928));
  return pyresult;
}
#else
#  define _cffi_f_GENERAL_NAME_new _cffi_d_GENERAL_NAME_new
#endif

static int _cffi_d_GENERAL_NAME_print(BIO * x0, GENERAL_NAME * x1)
{
  return GENERAL_NAME_print(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_NAME_print(PyObject *self, PyObject *args)
{
  BIO * x0;
  GENERAL_NAME * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "GENERAL_NAME_print", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(928), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (GENERAL_NAME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(928), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = GENERAL_NAME_print(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_GENERAL_NAME_print _cffi_d_GENERAL_NAME_print
#endif

static GENERAL_SUBTREE * _cffi_d_GENERAL_SUBTREE_new(void)
{
  return GENERAL_SUBTREE_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_GENERAL_SUBTREE_new(PyObject *self, PyObject *noarg)
{
  GENERAL_SUBTREE * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = GENERAL_SUBTREE_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1084));
  return pyresult;
}
#else
#  define _cffi_f_GENERAL_SUBTREE_new _cffi_d_GENERAL_SUBTREE_new
#endif

static int _cffi_d_HMAC_CTX_copy(HMAC_CTX * x0, HMAC_CTX * x1)
{
  return HMAC_CTX_copy(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_CTX_copy(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  HMAC_CTX * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "HMAC_CTX_copy", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1515), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1515), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1515), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1515), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_CTX_copy(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_CTX_copy _cffi_d_HMAC_CTX_copy
#endif

static void _cffi_d_HMAC_CTX_free(HMAC_CTX * x0)
{
  HMAC_CTX_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_CTX_free(PyObject *self, PyObject *arg0)
{
  HMAC_CTX * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1515), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1515), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { HMAC_CTX_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_HMAC_CTX_free _cffi_d_HMAC_CTX_free
#endif

static HMAC_CTX * _cffi_d_HMAC_CTX_new(void)
{
  return HMAC_CTX_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_CTX_new(PyObject *self, PyObject *noarg)
{
  HMAC_CTX * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_CTX_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1515));
  return pyresult;
}
#else
#  define _cffi_f_HMAC_CTX_new _cffi_d_HMAC_CTX_new
#endif

static int _cffi_d_HMAC_Final(HMAC_CTX * x0, unsigned char * x1, unsigned int * x2)
{
  return HMAC_Final(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_Final(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  unsigned char * x1;
  unsigned int * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "HMAC_Final", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1515), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1515), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1390), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1390), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_Final(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_Final _cffi_d_HMAC_Final
#endif

static int _cffi_d_HMAC_Init_ex(HMAC_CTX * x0, void const * x1, int x2, EVP_MD const * x3, ENGINE * x4)
{
  return HMAC_Init_ex(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_Init_ex(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  void const * x1;
  int x2;
  EVP_MD const * x3;
  ENGINE * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "HMAC_Init_ex", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1515), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1515), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(282), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (ENGINE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(282), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_Init_ex(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_Init_ex _cffi_d_HMAC_Init_ex
#endif

static int _cffi_d_HMAC_Update(HMAC_CTX * x0, unsigned char const * x1, size_t x2)
{
  return HMAC_Update(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_HMAC_Update(PyObject *self, PyObject *args)
{
  HMAC_CTX * x0;
  unsigned char const * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "HMAC_Update", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1515), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (HMAC_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1515), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = HMAC_Update(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_HMAC_Update _cffi_d_HMAC_Update
#endif

static void _cffi_d_ISSUING_DIST_POINT_free(ISSUING_DIST_POINT * x0)
{
  ISSUING_DIST_POINT_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ISSUING_DIST_POINT_free(PyObject *self, PyObject *arg0)
{
  ISSUING_DIST_POINT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2702), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ISSUING_DIST_POINT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2702), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { ISSUING_DIST_POINT_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_ISSUING_DIST_POINT_free _cffi_d_ISSUING_DIST_POINT_free
#endif

static ISSUING_DIST_POINT * _cffi_d_ISSUING_DIST_POINT_new(void)
{
  return ISSUING_DIST_POINT_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_ISSUING_DIST_POINT_new(PyObject *self, PyObject *noarg)
{
  ISSUING_DIST_POINT * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = ISSUING_DIST_POINT_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2702));
  return pyresult;
}
#else
#  define _cffi_f_ISSUING_DIST_POINT_new _cffi_d_ISSUING_DIST_POINT_new
#endif

static void _cffi_d_NAME_CONSTRAINTS_free(NAME_CONSTRAINTS * x0)
{
  NAME_CONSTRAINTS_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NAME_CONSTRAINTS_free(PyObject *self, PyObject *arg0)
{
  NAME_CONSTRAINTS * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2705), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NAME_CONSTRAINTS *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2705), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { NAME_CONSTRAINTS_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_NAME_CONSTRAINTS_free _cffi_d_NAME_CONSTRAINTS_free
#endif

static NAME_CONSTRAINTS * _cffi_d_NAME_CONSTRAINTS_new(void)
{
  return NAME_CONSTRAINTS_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NAME_CONSTRAINTS_new(PyObject *self, PyObject *noarg)
{
  NAME_CONSTRAINTS * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NAME_CONSTRAINTS_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2705));
  return pyresult;
}
#else
#  define _cffi_f_NAME_CONSTRAINTS_new _cffi_d_NAME_CONSTRAINTS_new
#endif

static NETSCAPE_SPKI * _cffi_d_NETSCAPE_SPKI_b64_decode(char const * x0, int x1)
{
  return NETSCAPE_SPKI_b64_decode(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_b64_decode(PyObject *self, PyObject *args)
{
  char const * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  NETSCAPE_SPKI * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_b64_decode", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_b64_decode(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(288));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_b64_decode _cffi_d_NETSCAPE_SPKI_b64_decode
#endif

static char * _cffi_d_NETSCAPE_SPKI_b64_encode(NETSCAPE_SPKI * x0)
{
  return NETSCAPE_SPKI_b64_encode(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_b64_encode(PyObject *self, PyObject *arg0)
{
  NETSCAPE_SPKI * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  char * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(288), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(288), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_b64_encode(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(409));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_b64_encode _cffi_d_NETSCAPE_SPKI_b64_encode
#endif

static void _cffi_d_NETSCAPE_SPKI_free(NETSCAPE_SPKI * x0)
{
  NETSCAPE_SPKI_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_free(PyObject *self, PyObject *arg0)
{
  NETSCAPE_SPKI * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(288), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(288), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { NETSCAPE_SPKI_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_free _cffi_d_NETSCAPE_SPKI_free
#endif

static EVP_PKEY * _cffi_d_NETSCAPE_SPKI_get_pubkey(NETSCAPE_SPKI * x0)
{
  return NETSCAPE_SPKI_get_pubkey(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_get_pubkey(PyObject *self, PyObject *arg0)
{
  NETSCAPE_SPKI * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(288), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(288), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_get_pubkey(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(199));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_get_pubkey _cffi_d_NETSCAPE_SPKI_get_pubkey
#endif

static NETSCAPE_SPKI * _cffi_d_NETSCAPE_SPKI_new(void)
{
  return NETSCAPE_SPKI_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_new(PyObject *self, PyObject *noarg)
{
  NETSCAPE_SPKI * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(288));
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_new _cffi_d_NETSCAPE_SPKI_new
#endif

static int _cffi_d_NETSCAPE_SPKI_set_pubkey(NETSCAPE_SPKI * x0, EVP_PKEY * x1)
{
  return NETSCAPE_SPKI_set_pubkey(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_set_pubkey(PyObject *self, PyObject *args)
{
  NETSCAPE_SPKI * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_set_pubkey", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(288), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(288), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_set_pubkey(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_set_pubkey _cffi_d_NETSCAPE_SPKI_set_pubkey
#endif

static int _cffi_d_NETSCAPE_SPKI_sign(NETSCAPE_SPKI * x0, EVP_PKEY * x1, EVP_MD const * x2)
{
  return NETSCAPE_SPKI_sign(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_sign(PyObject *self, PyObject *args)
{
  NETSCAPE_SPKI * x0;
  EVP_PKEY * x1;
  EVP_MD const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_sign", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(288), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(288), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_sign(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_sign _cffi_d_NETSCAPE_SPKI_sign
#endif

static int _cffi_d_NETSCAPE_SPKI_verify(NETSCAPE_SPKI * x0, EVP_PKEY * x1)
{
  return NETSCAPE_SPKI_verify(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NETSCAPE_SPKI_verify(PyObject *self, PyObject *args)
{
  NETSCAPE_SPKI * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "NETSCAPE_SPKI_verify", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(288), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NETSCAPE_SPKI *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(288), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NETSCAPE_SPKI_verify(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_NETSCAPE_SPKI_verify _cffi_d_NETSCAPE_SPKI_verify
#endif

static void _cffi_d_NOTICEREF_free(NOTICEREF * x0)
{
  NOTICEREF_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NOTICEREF_free(PyObject *self, PyObject *arg0)
{
  NOTICEREF * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2711), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (NOTICEREF *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2711), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { NOTICEREF_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_NOTICEREF_free _cffi_d_NOTICEREF_free
#endif

static NOTICEREF * _cffi_d_NOTICEREF_new(void)
{
  return NOTICEREF_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_NOTICEREF_new(PyObject *self, PyObject *noarg)
{
  NOTICEREF * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = NOTICEREF_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2711));
  return pyresult;
}
#else
#  define _cffi_f_NOTICEREF_new _cffi_d_NOTICEREF_new
#endif

static char const * _cffi_d_OBJ_nid2ln(int x0)
{
  return OBJ_nid2ln(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_nid2ln(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_nid2ln(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  return pyresult;
}
#else
#  define _cffi_f_OBJ_nid2ln _cffi_d_OBJ_nid2ln
#endif

static char const * _cffi_d_OBJ_nid2sn(int x0)
{
  return OBJ_nid2sn(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_nid2sn(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_nid2sn(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  return pyresult;
}
#else
#  define _cffi_f_OBJ_nid2sn _cffi_d_OBJ_nid2sn
#endif

static int _cffi_d_OBJ_obj2nid(ASN1_OBJECT const * x0)
{
  return OBJ_obj2nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_obj2nid(PyObject *self, PyObject *arg0)
{
  ASN1_OBJECT const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(765), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(765), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_obj2nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_obj2nid _cffi_d_OBJ_obj2nid
#endif

static int _cffi_d_OBJ_obj2txt(char * x0, int x1, ASN1_OBJECT const * x2, int x3)
{
  return OBJ_obj2txt(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_obj2txt(PyObject *self, PyObject *args)
{
  char * x0;
  int x1;
  ASN1_OBJECT const * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "OBJ_obj2txt", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(409), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(409), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(765), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(765), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_obj2txt(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_obj2txt _cffi_d_OBJ_obj2txt
#endif

static int _cffi_d_OBJ_sn2nid(char const * x0)
{
  return OBJ_sn2nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_sn2nid(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_sn2nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_sn2nid _cffi_d_OBJ_sn2nid
#endif

static int _cffi_d_OBJ_txt2nid(char const * x0)
{
  return OBJ_txt2nid(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_txt2nid(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_txt2nid(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_txt2nid _cffi_d_OBJ_txt2nid
#endif

static ASN1_OBJECT * _cffi_d_OBJ_txt2obj(char const * x0, int x1)
{
  return OBJ_txt2obj(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OBJ_txt2obj(PyObject *self, PyObject *args)
{
  char const * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_OBJECT * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OBJ_txt2obj", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OBJ_txt2obj(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(607));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OBJ_txt2obj _cffi_d_OBJ_txt2obj
#endif

static int _cffi_d_OCSP_BASICRESP_add_ext(OCSP_BASICRESP * x0, X509_EXTENSION * x1, int x2)
{
  return OCSP_BASICRESP_add_ext(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_BASICRESP_add_ext(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  X509_EXTENSION * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OCSP_BASICRESP_add_ext", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(387), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(387), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(40), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_EXTENSION *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(40), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_BASICRESP_add_ext(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_BASICRESP_add_ext _cffi_d_OCSP_BASICRESP_add_ext
#endif

static void _cffi_d_OCSP_BASICRESP_free(OCSP_BASICRESP * x0)
{
  OCSP_BASICRESP_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_BASICRESP_free(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(387), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(387), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OCSP_BASICRESP_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OCSP_BASICRESP_free _cffi_d_OCSP_BASICRESP_free
#endif

static X509_EXTENSION * _cffi_d_OCSP_BASICRESP_get_ext(OCSP_BASICRESP * x0, int x1)
{
  return OCSP_BASICRESP_get_ext(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_BASICRESP_get_ext(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_EXTENSION * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_BASICRESP_get_ext", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(387), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(387), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_BASICRESP_get_ext(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(40));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_BASICRESP_get_ext _cffi_d_OCSP_BASICRESP_get_ext
#endif

static int _cffi_d_OCSP_BASICRESP_get_ext_count(OCSP_BASICRESP * x0)
{
  return OCSP_BASICRESP_get_ext_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_BASICRESP_get_ext_count(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(387), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(387), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_BASICRESP_get_ext_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_BASICRESP_get_ext_count _cffi_d_OCSP_BASICRESP_get_ext_count
#endif

static OCSP_BASICRESP * _cffi_d_OCSP_BASICRESP_new(void)
{
  return OCSP_BASICRESP_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_BASICRESP_new(PyObject *self, PyObject *noarg)
{
  OCSP_BASICRESP * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_BASICRESP_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(387));
  return pyresult;
}
#else
#  define _cffi_f_OCSP_BASICRESP_new _cffi_d_OCSP_BASICRESP_new
#endif

static void _cffi_d_OCSP_CERTID_free(OCSP_CERTID * x0)
{
  OCSP_CERTID_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_CERTID_free(PyObject *self, PyObject *arg0)
{
  OCSP_CERTID * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(366), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_CERTID *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(366), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OCSP_CERTID_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OCSP_CERTID_free _cffi_d_OCSP_CERTID_free
#endif

static int _cffi_d_OCSP_REQUEST_add_ext(OCSP_REQUEST * x0, X509_EXTENSION * x1, int x2)
{
  return OCSP_REQUEST_add_ext(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_REQUEST_add_ext(PyObject *self, PyObject *args)
{
  OCSP_REQUEST * x0;
  X509_EXTENSION * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OCSP_REQUEST_add_ext", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(365), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(365), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(40), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_EXTENSION *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(40), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_REQUEST_add_ext(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_REQUEST_add_ext _cffi_d_OCSP_REQUEST_add_ext
#endif

static void _cffi_d_OCSP_REQUEST_free(OCSP_REQUEST * x0)
{
  OCSP_REQUEST_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_REQUEST_free(PyObject *self, PyObject *arg0)
{
  OCSP_REQUEST * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(365), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(365), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OCSP_REQUEST_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OCSP_REQUEST_free _cffi_d_OCSP_REQUEST_free
#endif

static X509_EXTENSION * _cffi_d_OCSP_REQUEST_get_ext(OCSP_REQUEST * x0, int x1)
{
  return OCSP_REQUEST_get_ext(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_REQUEST_get_ext(PyObject *self, PyObject *args)
{
  OCSP_REQUEST * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_EXTENSION * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_REQUEST_get_ext", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(365), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(365), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_REQUEST_get_ext(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(40));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_REQUEST_get_ext _cffi_d_OCSP_REQUEST_get_ext
#endif

static int _cffi_d_OCSP_REQUEST_get_ext_count(OCSP_REQUEST * x0)
{
  return OCSP_REQUEST_get_ext_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_REQUEST_get_ext_count(PyObject *self, PyObject *arg0)
{
  OCSP_REQUEST * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(365), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(365), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_REQUEST_get_ext_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_REQUEST_get_ext_count _cffi_d_OCSP_REQUEST_get_ext_count
#endif

static OCSP_REQUEST * _cffi_d_OCSP_REQUEST_new(void)
{
  return OCSP_REQUEST_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_REQUEST_new(PyObject *self, PyObject *noarg)
{
  OCSP_REQUEST * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_REQUEST_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(365));
  return pyresult;
}
#else
#  define _cffi_f_OCSP_REQUEST_new _cffi_d_OCSP_REQUEST_new
#endif

static void _cffi_d_OCSP_RESPONSE_free(OCSP_RESPONSE * x0)
{
  OCSP_RESPONSE_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_RESPONSE_free(PyObject *self, PyObject *arg0)
{
  OCSP_RESPONSE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(349), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_RESPONSE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(349), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OCSP_RESPONSE_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OCSP_RESPONSE_free _cffi_d_OCSP_RESPONSE_free
#endif

static OCSP_CERTID const * _cffi_d_OCSP_SINGLERESP_get0_id(OCSP_SINGLERESP const * x0)
{
  return OCSP_SINGLERESP_get0_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_SINGLERESP_get0_id(PyObject *self, PyObject *arg0)
{
  OCSP_SINGLERESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_CERTID const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(362), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_SINGLERESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(362), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_SINGLERESP_get0_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3079));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_SINGLERESP_get0_id _cffi_d_OCSP_SINGLERESP_get0_id
#endif

static X509_EXTENSION * _cffi_d_OCSP_SINGLERESP_get_ext(OCSP_SINGLERESP * x0, int x1)
{
  return OCSP_SINGLERESP_get_ext(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_SINGLERESP_get_ext(PyObject *self, PyObject *args)
{
  OCSP_SINGLERESP * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_EXTENSION * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_SINGLERESP_get_ext", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(594), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_SINGLERESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(594), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_SINGLERESP_get_ext(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(40));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_SINGLERESP_get_ext _cffi_d_OCSP_SINGLERESP_get_ext
#endif

static int _cffi_d_OCSP_SINGLERESP_get_ext_count(OCSP_SINGLERESP * x0)
{
  return OCSP_SINGLERESP_get_ext_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_SINGLERESP_get_ext_count(PyObject *self, PyObject *arg0)
{
  OCSP_SINGLERESP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(594), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_SINGLERESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(594), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_SINGLERESP_get_ext_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_SINGLERESP_get_ext_count _cffi_d_OCSP_SINGLERESP_get_ext_count
#endif

static int _cffi_d_OCSP_basic_add1_cert(OCSP_BASICRESP * x0, X509 * x1)
{
  return OCSP_basic_add1_cert(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_basic_add1_cert(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  X509 * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_basic_add1_cert", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(387), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(387), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(28), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(28), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_basic_add1_cert(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_basic_add1_cert _cffi_d_OCSP_basic_add1_cert
#endif

static OCSP_SINGLERESP * _cffi_d_OCSP_basic_add1_status(OCSP_BASICRESP * x0, OCSP_CERTID * x1, int x2, int x3, ASN1_OCTET_STRING * x4, ASN1_OCTET_STRING * x5, ASN1_OCTET_STRING * x6)
{
  return OCSP_basic_add1_status(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_basic_add1_status(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  OCSP_CERTID * x1;
  int x2;
  int x3;
  ASN1_OCTET_STRING * x4;
  ASN1_OCTET_STRING * x5;
  ASN1_OCTET_STRING * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_SINGLERESP * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "OCSP_basic_add1_status", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(387), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(387), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(366), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (OCSP_CERTID *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(366), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(13), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(13), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_basic_add1_status(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(594));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_basic_add1_status _cffi_d_OCSP_basic_add1_status
#endif

static int _cffi_d_OCSP_basic_sign(OCSP_BASICRESP * x0, X509 * x1, EVP_PKEY * x2, EVP_MD const * x3, Cryptography_STACK_OF_X509 * x4, unsigned long x5)
{
  return OCSP_basic_sign(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_basic_sign(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  X509 * x1;
  EVP_PKEY * x2;
  EVP_MD const * x3;
  Cryptography_STACK_OF_X509 * x4;
  unsigned long x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "OCSP_basic_sign", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(387), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(387), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(28), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(28), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(413), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(413), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, unsigned long);
  if (x5 == (unsigned long)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_basic_sign(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_basic_sign _cffi_d_OCSP_basic_sign
#endif

static OCSP_CERTID * _cffi_d_OCSP_cert_to_id(EVP_MD const * x0, X509 const * x1, X509 const * x2)
{
  return OCSP_cert_to_id(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_cert_to_id(PyObject *self, PyObject *args)
{
  EVP_MD const * x0;
  X509 const * x1;
  X509 const * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_CERTID * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OCSP_cert_to_id", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(355), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(355), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(355), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (X509 const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(355), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_cert_to_id(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(366));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_cert_to_id _cffi_d_OCSP_cert_to_id
#endif

static int _cffi_d_OCSP_id_get0_info(ASN1_OCTET_STRING * * x0, ASN1_OBJECT * * x1, ASN1_OCTET_STRING * * x2, ASN1_INTEGER * * x3, OCSP_CERTID * x4)
{
  return OCSP_id_get0_info(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_id_get0_info(PyObject *self, PyObject *args)
{
  ASN1_OCTET_STRING * * x0;
  ASN1_OBJECT * * x1;
  ASN1_OCTET_STRING * * x2;
  ASN1_INTEGER * * x3;
  OCSP_CERTID * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "OCSP_id_get0_info", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(768), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(768), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(769), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_OBJECT * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(769), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(768), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(768), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(771), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ASN1_INTEGER * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(771), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(366), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (OCSP_CERTID *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(366), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_id_get0_info(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_id_get0_info _cffi_d_OCSP_id_get0_info
#endif

static OCSP_CERTID * _cffi_d_OCSP_onereq_get0_id(OCSP_ONEREQ * x0)
{
  return OCSP_onereq_get0_id(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_onereq_get0_id(PyObject *self, PyObject *arg0)
{
  OCSP_ONEREQ * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_CERTID * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(359), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_ONEREQ *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(359), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_onereq_get0_id(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(366));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_onereq_get0_id _cffi_d_OCSP_onereq_get0_id
#endif

static OCSP_ONEREQ * _cffi_d_OCSP_request_add0_id(OCSP_REQUEST * x0, OCSP_CERTID * x1)
{
  return OCSP_request_add0_id(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_request_add0_id(PyObject *self, PyObject *args)
{
  OCSP_REQUEST * x0;
  OCSP_CERTID * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_ONEREQ * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_request_add0_id", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(365), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(365), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(366), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (OCSP_CERTID *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(366), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_request_add0_id(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(359));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_request_add0_id _cffi_d_OCSP_request_add0_id
#endif

static int _cffi_d_OCSP_request_onereq_count(OCSP_REQUEST * x0)
{
  return OCSP_request_onereq_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_request_onereq_count(PyObject *self, PyObject *arg0)
{
  OCSP_REQUEST * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(365), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(365), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_request_onereq_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_request_onereq_count _cffi_d_OCSP_request_onereq_count
#endif

static OCSP_ONEREQ * _cffi_d_OCSP_request_onereq_get0(OCSP_REQUEST * x0, int x1)
{
  return OCSP_request_onereq_get0(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_request_onereq_get0(PyObject *self, PyObject *args)
{
  OCSP_REQUEST * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_ONEREQ * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_request_onereq_get0", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(365), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_REQUEST *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(365), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_request_onereq_get0(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(359));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_request_onereq_get0 _cffi_d_OCSP_request_onereq_get0
#endif

static int _cffi_d_OCSP_resp_count(OCSP_BASICRESP * x0)
{
  return OCSP_resp_count(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_count(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(387), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(387), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_count(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_count _cffi_d_OCSP_resp_count
#endif

static OCSP_SINGLERESP * _cffi_d_OCSP_resp_get0(OCSP_BASICRESP * x0, int x1)
{
  return OCSP_resp_get0(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_SINGLERESP * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_resp_get0", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(387), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(387), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(594));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0 _cffi_d_OCSP_resp_get0
#endif

static Cryptography_STACK_OF_X509 * _cffi_d_OCSP_resp_get0_certs(OCSP_BASICRESP const * x0)
{
  return OCSP_resp_get0_certs(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_certs(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  Cryptography_STACK_OF_X509 * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_certs(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(413));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_certs _cffi_d_OCSP_resp_get0_certs
#endif

static int _cffi_d_OCSP_resp_get0_id(OCSP_BASICRESP const * x0, ASN1_OCTET_STRING const * * x1, X509_NAME const * * x2)
{
  return OCSP_resp_get0_id(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_id(PyObject *self, PyObject *args)
{
  OCSP_BASICRESP const * x0;
  ASN1_OCTET_STRING const * * x1;
  X509_NAME const * * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "OCSP_resp_get0_id", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1566), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (ASN1_OCTET_STRING const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1566), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1567), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (X509_NAME const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1567), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_id(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_id _cffi_d_OCSP_resp_get0_id
#endif

static ASN1_GENERALIZEDTIME const * _cffi_d_OCSP_resp_get0_produced_at(OCSP_BASICRESP const * x0)
{
  return OCSP_resp_get0_produced_at(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_produced_at(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_GENERALIZEDTIME const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_produced_at(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3009));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_produced_at _cffi_d_OCSP_resp_get0_produced_at
#endif

static OCSP_RESPDATA const * _cffi_d_OCSP_resp_get0_respdata(OCSP_BASICRESP const * x0)
{
  return OCSP_resp_get0_respdata(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_respdata(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_RESPDATA const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_respdata(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3083));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_respdata _cffi_d_OCSP_resp_get0_respdata
#endif

static ASN1_OCTET_STRING const * _cffi_d_OCSP_resp_get0_signature(OCSP_BASICRESP const * x0)
{
  return OCSP_resp_get0_signature(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_signature(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ASN1_OCTET_STRING const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_signature(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3016));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_signature _cffi_d_OCSP_resp_get0_signature
#endif

static X509_ALGOR const * _cffi_d_OCSP_resp_get0_tbs_sigalg(OCSP_BASICRESP const * x0)
{
  return OCSP_resp_get0_tbs_sigalg(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_resp_get0_tbs_sigalg(PyObject *self, PyObject *arg0)
{
  OCSP_BASICRESP const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_ALGOR const * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(17), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(17), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_resp_get0_tbs_sigalg(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3121));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_resp_get0_tbs_sigalg _cffi_d_OCSP_resp_get0_tbs_sigalg
#endif

static OCSP_RESPONSE * _cffi_d_OCSP_response_create(int x0, OCSP_BASICRESP * x1)
{
  return OCSP_response_create(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_response_create(PyObject *self, PyObject *args)
{
  int x0;
  OCSP_BASICRESP * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_RESPONSE * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "OCSP_response_create", 2, 2, &arg0, &arg1))
    return NULL;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(387), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (OCSP_BASICRESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(387), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_response_create(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(349));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_response_create _cffi_d_OCSP_response_create
#endif

static OCSP_BASICRESP * _cffi_d_OCSP_response_get1_basic(OCSP_RESPONSE * x0)
{
  return OCSP_response_get1_basic(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_response_get1_basic(PyObject *self, PyObject *arg0)
{
  OCSP_RESPONSE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  OCSP_BASICRESP * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(349), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_RESPONSE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(349), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_response_get1_basic(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(387));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_response_get1_basic _cffi_d_OCSP_response_get1_basic
#endif

static int _cffi_d_OCSP_response_status(OCSP_RESPONSE * x0)
{
  return OCSP_response_status(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_response_status(PyObject *self, PyObject *arg0)
{
  OCSP_RESPONSE * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(349), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_RESPONSE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(349), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_response_status(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_response_status _cffi_d_OCSP_response_status
#endif

static int _cffi_d_OCSP_single_get0_status(OCSP_SINGLERESP * x0, int * x1, ASN1_GENERALIZEDTIME * * x2, ASN1_GENERALIZEDTIME * * x3, ASN1_GENERALIZEDTIME * * x4)
{
  return OCSP_single_get0_status(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OCSP_single_get0_status(PyObject *self, PyObject *args)
{
  OCSP_SINGLERESP * x0;
  int * x1;
  ASN1_GENERALIZEDTIME * * x2;
  ASN1_GENERALIZEDTIME * * x3;
  ASN1_GENERALIZEDTIME * * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "OCSP_single_get0_status", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(594), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OCSP_SINGLERESP *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(594), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1169), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (int *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1169), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(14), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(14), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(14), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(14), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(14), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (ASN1_GENERALIZEDTIME * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(14), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OCSP_single_get0_status(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_OCSP_single_get0_status _cffi_d_OCSP_single_get0_status
#endif

static void _cffi_d_OPENSSL_cleanup(void)
{
  OPENSSL_cleanup();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_cleanup(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_cleanup(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_cleanup _cffi_d_OPENSSL_cleanup
#endif

static void _cffi_d_OPENSSL_config(char const * x0)
{
  OPENSSL_config(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_config(PyObject *self, PyObject *arg0)
{
  char const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_config(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_config _cffi_d_OPENSSL_config
#endif

static void _cffi_d_OPENSSL_free(void * x0)
{
  OPENSSL_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_free(PyObject *self, PyObject *arg0)
{
  void * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_free _cffi_d_OPENSSL_free
#endif

static void * _cffi_d_OPENSSL_malloc(size_t x0)
{
  return OPENSSL_malloc(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_malloc(PyObject *self, PyObject *arg0)
{
  size_t x0;
  void * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, size_t);
  if (x0 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OPENSSL_malloc(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(193));
  return pyresult;
}
#else
#  define _cffi_f_OPENSSL_malloc _cffi_d_OPENSSL_malloc
#endif

static void _cffi_d_OPENSSL_no_config(void)
{
  OPENSSL_no_config();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OPENSSL_no_config(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OPENSSL_no_config(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OPENSSL_no_config _cffi_d_OPENSSL_no_config
#endif

static void _cffi_d_OTHERNAME_free(OTHERNAME * x0)
{
  OTHERNAME_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OTHERNAME_free(PyObject *self, PyObject *arg0)
{
  OTHERNAME * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2726), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (OTHERNAME *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2726), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OTHERNAME_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OTHERNAME_free _cffi_d_OTHERNAME_free
#endif

static OTHERNAME * _cffi_d_OTHERNAME_new(void)
{
  return OTHERNAME_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OTHERNAME_new(PyObject *self, PyObject *noarg)
{
  OTHERNAME * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OTHERNAME_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2726));
  return pyresult;
}
#else
#  define _cffi_f_OTHERNAME_new _cffi_d_OTHERNAME_new
#endif

static void _cffi_d_OpenSSL_add_all_algorithms(void)
{
  OpenSSL_add_all_algorithms();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OpenSSL_add_all_algorithms(PyObject *self, PyObject *noarg)
{

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { OpenSSL_add_all_algorithms(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_OpenSSL_add_all_algorithms _cffi_d_OpenSSL_add_all_algorithms
#endif

static char const * _cffi_d_OpenSSL_version(int x0)
{
  return OpenSSL_version(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OpenSSL_version(PyObject *self, PyObject *arg0)
{
  int x0;
  char const * result;
  PyObject *pyresult;

  x0 = _cffi_to_c_int(arg0, int);
  if (x0 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OpenSSL_version(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(46));
  return pyresult;
}
#else
#  define _cffi_f_OpenSSL_version _cffi_d_OpenSSL_version
#endif

static unsigned long _cffi_d_OpenSSL_version_num(void)
{
  return OpenSSL_version_num();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_OpenSSL_version_num(PyObject *self, PyObject *noarg)
{
  unsigned long result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = OpenSSL_version_num(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, unsigned long);
  return pyresult;
}
#else
#  define _cffi_f_OpenSSL_version_num _cffi_d_OpenSSL_version_num
#endif

static DH * _cffi_d_PEM_read_bio_DHparams(BIO * x0, DH * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_DHparams(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_DHparams(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  DH * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_DHparams", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(187), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(187), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(192));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_DHparams(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(196));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_DHparams _cffi_d_PEM_read_bio_DHparams
#endif

static PKCS7 * _cffi_d_PEM_read_bio_PKCS7(BIO * x0, PKCS7 * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_PKCS7(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_PKCS7(PyObject *self, PyObject *args)
{
  BIO * x0;
  PKCS7 * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PKCS7 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_PKCS7", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(426), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (PKCS7 * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(426), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(192));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_PKCS7(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(442));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_PKCS7 _cffi_d_PEM_read_bio_PKCS7
#endif

static EVP_PKEY * _cffi_d_PEM_read_bio_PUBKEY(BIO * x0, EVP_PKEY * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_PUBKEY(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_PUBKEY(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_PUBKEY", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(273), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(273), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(192));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_PUBKEY(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(199));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_PUBKEY _cffi_d_PEM_read_bio_PUBKEY
#endif

static EVP_PKEY * _cffi_d_PEM_read_bio_PrivateKey(BIO * x0, EVP_PKEY * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_PrivateKey(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_PrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  EVP_PKEY * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_PrivateKey", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(273), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(273), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(192));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_PrivateKey(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(199));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_PrivateKey _cffi_d_PEM_read_bio_PrivateKey
#endif

static RSA * _cffi_d_PEM_read_bio_RSAPublicKey(BIO * x0, RSA * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_RSAPublicKey(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_RSAPublicKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  RSA * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  RSA * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_RSAPublicKey", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(464), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(464), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(192));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_RSAPublicKey(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(476));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_RSAPublicKey _cffi_d_PEM_read_bio_RSAPublicKey
#endif

static X509 * _cffi_d_PEM_read_bio_X509(BIO * x0, X509 * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_X509(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_X509(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509 * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_X509", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(518), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(518), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(192));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_X509(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(28));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_X509 _cffi_d_PEM_read_bio_X509
#endif

static X509_CRL * _cffi_d_PEM_read_bio_X509_CRL(BIO * x0, X509_CRL * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_X509_CRL(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_X509_CRL(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509_CRL * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_CRL * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_X509_CRL", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(556), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_CRL * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(556), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(192));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_X509_CRL(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(57));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_X509_CRL _cffi_d_PEM_read_bio_X509_CRL
#endif

static X509_REQ * _cffi_d_PEM_read_bio_X509_REQ(BIO * x0, X509_REQ * * x1, int(* x2)(char *, int, int, void *), void * x3)
{
  return PEM_read_bio_X509_REQ(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_read_bio_X509_REQ(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509_REQ * * x1;
  int(* x2)(char *, int, int, void *);
  void * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  X509_REQ * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_read_bio_X509_REQ", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(671), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_REQ * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(671), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg2, _cffi_type(192));
  if (x2 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_read_bio_X509_REQ(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(294));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_read_bio_X509_REQ _cffi_d_PEM_read_bio_X509_REQ
#endif

static int _cffi_d_PEM_write_bio_DHparams(BIO * x0, DH * x1)
{
  return PEM_write_bio_DHparams(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_DHparams(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_DHparams", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_DHparams(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_DHparams _cffi_d_PEM_write_bio_DHparams
#endif

static int _cffi_d_PEM_write_bio_DHxparams(BIO * x0, DH * x1)
{
  return PEM_write_bio_DHxparams(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_DHxparams(PyObject *self, PyObject *args)
{
  BIO * x0;
  DH * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_DHxparams", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(196), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DH *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(196), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_DHxparams(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_DHxparams _cffi_d_PEM_write_bio_DHxparams
#endif

static int _cffi_d_PEM_write_bio_DSAPrivateKey(BIO * x0, DSA * x1, EVP_CIPHER const * x2, unsigned char * x3, int x4, int(* x5)(char *, int, int, void *), void * x6)
{
  return PEM_write_bio_DSAPrivateKey(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_DSAPrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  DSA * x1;
  EVP_CIPHER const * x2;
  unsigned char * x3;
  int x4;
  int(* x5)(char *, int, int, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_DSAPrivateKey", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(212), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (DSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(212), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(885), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(885), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg5, _cffi_type(192));
  if (x5 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_DSAPrivateKey(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_DSAPrivateKey _cffi_d_PEM_write_bio_DSAPrivateKey
#endif

static int _cffi_d_PEM_write_bio_ECPrivateKey(BIO * x0, EC_KEY * x1, EVP_CIPHER const * x2, unsigned char * x3, int x4, int(* x5)(char *, int, int, void *), void * x6)
{
  return PEM_write_bio_ECPrivateKey(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_ECPrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  EC_KEY * x1;
  EVP_CIPHER const * x2;
  unsigned char * x3;
  int x4;
  int(* x5)(char *, int, int, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_ECPrivateKey", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(893), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EC_KEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(893), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(885), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(885), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg5, _cffi_type(192));
  if (x5 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_ECPrivateKey(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_ECPrivateKey _cffi_d_PEM_write_bio_ECPrivateKey
#endif

static int _cffi_d_PEM_write_bio_PKCS7(BIO * x0, PKCS7 * x1)
{
  return PEM_write_bio_PKCS7(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_PKCS7(PyObject *self, PyObject *args)
{
  BIO * x0;
  PKCS7 * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_PKCS7", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(442), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(442), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_PKCS7(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_PKCS7 _cffi_d_PEM_write_bio_PKCS7
#endif

static int _cffi_d_PEM_write_bio_PKCS7_stream(BIO * x0, PKCS7 * x1, BIO * x2, int x3)
{
  return PEM_write_bio_PKCS7_stream(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_PKCS7_stream(PyObject *self, PyObject *args)
{
  BIO * x0;
  PKCS7 * x1;
  BIO * x2;
  int x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_PKCS7_stream", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(442), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(442), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_PKCS7_stream(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_PKCS7_stream _cffi_d_PEM_write_bio_PKCS7_stream
#endif

static int _cffi_d_PEM_write_bio_PKCS8PrivateKey(BIO * x0, EVP_PKEY * x1, EVP_CIPHER const * x2, char * x3, int x4, int(* x5)(char *, int, int, void *), void * x6)
{
  return PEM_write_bio_PKCS8PrivateKey(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_PKCS8PrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * x1;
  EVP_CIPHER const * x2;
  char * x3;
  int x4;
  int(* x5)(char *, int, int, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_PKCS8PrivateKey", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(885), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(885), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(409), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(409), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg5, _cffi_type(192));
  if (x5 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_PKCS8PrivateKey(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_PKCS8PrivateKey _cffi_d_PEM_write_bio_PKCS8PrivateKey
#endif

static int _cffi_d_PEM_write_bio_PUBKEY(BIO * x0, EVP_PKEY * x1)
{
  return PEM_write_bio_PUBKEY(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_PUBKEY(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_PUBKEY", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_PUBKEY(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_PUBKEY _cffi_d_PEM_write_bio_PUBKEY
#endif

static int _cffi_d_PEM_write_bio_PrivateKey(BIO * x0, EVP_PKEY * x1, EVP_CIPHER const * x2, unsigned char * x3, int x4, int(* x5)(char *, int, int, void *), void * x6)
{
  return PEM_write_bio_PrivateKey(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_PrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  EVP_PKEY * x1;
  EVP_CIPHER const * x2;
  unsigned char * x3;
  int x4;
  int(* x5)(char *, int, int, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_PrivateKey", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(885), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(885), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg5, _cffi_type(192));
  if (x5 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_PrivateKey(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_PrivateKey _cffi_d_PEM_write_bio_PrivateKey
#endif

static int _cffi_d_PEM_write_bio_RSAPrivateKey(BIO * x0, RSA * x1, EVP_CIPHER const * x2, unsigned char * x3, int x4, int(* x5)(char *, int, int, void *), void * x6)
{
  return PEM_write_bio_RSAPrivateKey(x0, x1, x2, x3, x4, x5, x6);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_RSAPrivateKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  RSA * x1;
  EVP_CIPHER const * x2;
  unsigned char * x3;
  int x4;
  int(* x5)(char *, int, int, void *);
  void * x6;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_RSAPrivateKey", 7, 7, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(476), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(476), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(885), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_CIPHER const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(885), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  x5 = (int(*)(char *, int, int, void *))_cffi_to_c_pointer(arg5, _cffi_type(192));
  if (x5 == (int(*)(char *, int, int, void *))NULL && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(193), arg6, (char **)&x6);
  if (datasize != 0) {
    x6 = ((size_t)datasize) <= 640 ? (void *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(193), arg6, (char **)&x6,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_RSAPrivateKey(x0, x1, x2, x3, x4, x5, x6); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_RSAPrivateKey _cffi_d_PEM_write_bio_RSAPrivateKey
#endif

static int _cffi_d_PEM_write_bio_RSAPublicKey(BIO * x0, RSA const * x1)
{
  return PEM_write_bio_RSAPublicKey(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_RSAPublicKey(PyObject *self, PyObject *args)
{
  BIO * x0;
  RSA const * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_RSAPublicKey", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(967), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(967), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_RSAPublicKey(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_RSAPublicKey _cffi_d_PEM_write_bio_RSAPublicKey
#endif

static int _cffi_d_PEM_write_bio_X509(BIO * x0, X509 * x1)
{
  return PEM_write_bio_X509(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_X509(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509 * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_X509", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(28), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(28), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_X509(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_X509 _cffi_d_PEM_write_bio_X509
#endif

static int _cffi_d_PEM_write_bio_X509_CRL(BIO * x0, X509_CRL * x1)
{
  return PEM_write_bio_X509_CRL(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_X509_CRL(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509_CRL * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_X509_CRL", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(57), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_CRL *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(57), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_X509_CRL(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_X509_CRL _cffi_d_PEM_write_bio_X509_CRL
#endif

static int _cffi_d_PEM_write_bio_X509_REQ(BIO * x0, X509_REQ * x1)
{
  return PEM_write_bio_X509_REQ(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PEM_write_bio_X509_REQ(PyObject *self, PyObject *args)
{
  BIO * x0;
  X509_REQ * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "PEM_write_bio_X509_REQ", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(294), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509_REQ *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(294), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PEM_write_bio_X509_REQ(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PEM_write_bio_X509_REQ _cffi_d_PEM_write_bio_X509_REQ
#endif

static PKCS12 * _cffi_d_PKCS12_create(char * x0, char * x1, EVP_PKEY * x2, X509 * x3, Cryptography_STACK_OF_X509 * x4, int x5, int x6, int x7, int x8, int x9)
{
  return PKCS12_create(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS12_create(PyObject *self, PyObject *args)
{
  char * x0;
  char * x1;
  EVP_PKEY * x2;
  X509 * x3;
  Cryptography_STACK_OF_X509 * x4;
  int x5;
  int x6;
  int x7;
  int x8;
  int x9;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PKCS12 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;
  PyObject *arg8;
  PyObject *arg9;

  if (!PyArg_UnpackTuple(args, "PKCS12_create", 10, 10, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7, &arg8, &arg9))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(409), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(409), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(409), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(409), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(28), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(28), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(413), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(413), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  x7 = _cffi_to_c_int(arg7, int);
  if (x7 == (int)-1 && PyErr_Occurred())
    return NULL;

  x8 = _cffi_to_c_int(arg8, int);
  if (x8 == (int)-1 && PyErr_Occurred())
    return NULL;

  x9 = _cffi_to_c_int(arg9, int);
  if (x9 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS12_create(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(940));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS12_create _cffi_d_PKCS12_create
#endif

static void _cffi_d_PKCS12_free(PKCS12 * x0)
{
  PKCS12_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS12_free(PyObject *self, PyObject *arg0)
{
  PKCS12 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(940), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS12 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(940), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { PKCS12_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_PKCS12_free _cffi_d_PKCS12_free
#endif

static int _cffi_d_PKCS12_parse(PKCS12 * x0, char const * x1, EVP_PKEY * * x2, X509 * * x3, Cryptography_STACK_OF_X509 * * x4)
{
  return PKCS12_parse(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS12_parse(PyObject *self, PyObject *args)
{
  PKCS12 * x0;
  char const * x1;
  EVP_PKEY * * x2;
  X509 * * x3;
  Cryptography_STACK_OF_X509 * * x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "PKCS12_parse", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(940), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS12 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(940), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(273), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_PKEY * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(273), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(518), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (X509 * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(518), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1599), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1599), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS12_parse(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS12_parse _cffi_d_PKCS12_parse
#endif

static int _cffi_d_PKCS5_PBKDF2_HMAC(char const * x0, int x1, unsigned char const * x2, int x3, int x4, EVP_MD const * x5, int x6, unsigned char * x7)
{
  return PKCS5_PBKDF2_HMAC(x0, x1, x2, x3, x4, x5, x6, x7);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS5_PBKDF2_HMAC(PyObject *self, PyObject *args)
{
  char const * x0;
  int x1;
  unsigned char const * x2;
  int x3;
  int x4;
  EVP_MD const * x5;
  int x6;
  unsigned char * x7;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;
  PyObject *arg6;
  PyObject *arg7;

  if (!PyArg_UnpackTuple(args, "PKCS5_PBKDF2_HMAC", 8, 8, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5, &arg6, &arg7))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(46), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(46), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(103), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (unsigned char const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(103), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x3 = _cffi_to_c_int(arg3, int);
  if (x3 == (int)-1 && PyErr_Occurred())
    return NULL;

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg5, (char **)&x5);
  if (datasize != 0) {
    x5 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg5, (char **)&x5,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x6 = _cffi_to_c_int(arg6, int);
  if (x6 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg7, (char **)&x7);
  if (datasize != 0) {
    x7 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg7, (char **)&x7,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS5_PBKDF2_HMAC(x0, x1, x2, x3, x4, x5, x6, x7); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS5_PBKDF2_HMAC _cffi_d_PKCS5_PBKDF2_HMAC
#endif

static int _cffi_d_PKCS7_final(PKCS7 * x0, BIO * x1, int x2)
{
  return PKCS7_final(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_final(PyObject *self, PyObject *args)
{
  PKCS7 * x0;
  BIO * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "PKCS7_final", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(442), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(442), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_final(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_final _cffi_d_PKCS7_final
#endif

static void _cffi_d_PKCS7_free(PKCS7 * x0)
{
  PKCS7_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_free(PyObject *self, PyObject *arg0)
{
  PKCS7 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(442), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(442), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { PKCS7_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_PKCS7_free _cffi_d_PKCS7_free
#endif

static PKCS7 * _cffi_d_PKCS7_sign(X509 * x0, EVP_PKEY * x1, Cryptography_STACK_OF_X509 * x2, BIO * x3, int x4)
{
  return PKCS7_sign(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_sign(PyObject *self, PyObject *args)
{
  X509 * x0;
  EVP_PKEY * x1;
  Cryptography_STACK_OF_X509 * x2;
  BIO * x3;
  int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PKCS7 * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "PKCS7_sign", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(28), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(28), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(413), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(413), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_sign(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(442));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_sign _cffi_d_PKCS7_sign
#endif

static PKCS7_SIGNER_INFO * _cffi_d_PKCS7_sign_add_signer(PKCS7 * x0, X509 * x1, EVP_PKEY * x2, EVP_MD const * x3, int x4)
{
  return PKCS7_sign_add_signer(x0, x1, x2, x3, x4);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_sign_add_signer(PyObject *self, PyObject *args)
{
  PKCS7 * x0;
  X509 * x1;
  EVP_PKEY * x2;
  EVP_MD const * x3;
  int x4;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PKCS7_SIGNER_INFO * result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;

  if (!PyArg_UnpackTuple(args, "PKCS7_sign_add_signer", 5, 5, &arg0, &arg1, &arg2, &arg3, &arg4))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(442), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(442), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(28), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(28), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(199), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (EVP_PKEY *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(199), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(354), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (EVP_MD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(354), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x4 = _cffi_to_c_int(arg4, int);
  if (x4 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_sign_add_signer(x0, x1, x2, x3, x4); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(3098));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_sign_add_signer _cffi_d_PKCS7_sign_add_signer
#endif

static int _cffi_d_PKCS7_type_is_data(PKCS7 * x0)
{
  return PKCS7_type_is_data(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_type_is_data(PyObject *self, PyObject *arg0)
{
  PKCS7 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(442), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(442), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_type_is_data(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_type_is_data _cffi_d_PKCS7_type_is_data
#endif

static int _cffi_d_PKCS7_type_is_enveloped(PKCS7 * x0)
{
  return PKCS7_type_is_enveloped(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_type_is_enveloped(PyObject *self, PyObject *arg0)
{
  PKCS7 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(442), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(442), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_type_is_enveloped(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_type_is_enveloped _cffi_d_PKCS7_type_is_enveloped
#endif

static int _cffi_d_PKCS7_type_is_signed(PKCS7 * x0)
{
  return PKCS7_type_is_signed(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_type_is_signed(PyObject *self, PyObject *arg0)
{
  PKCS7 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(442), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(442), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_type_is_signed(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_type_is_signed _cffi_d_PKCS7_type_is_signed
#endif

static int _cffi_d_PKCS7_type_is_signedAndEnveloped(PKCS7 * x0)
{
  return PKCS7_type_is_signedAndEnveloped(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_type_is_signedAndEnveloped(PyObject *self, PyObject *arg0)
{
  PKCS7 * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(442), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(442), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_type_is_signedAndEnveloped(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_type_is_signedAndEnveloped _cffi_d_PKCS7_type_is_signedAndEnveloped
#endif

static int _cffi_d_PKCS7_verify(PKCS7 * x0, Cryptography_STACK_OF_X509 * x1, X509_STORE * x2, BIO * x3, BIO * x4, int x5)
{
  return PKCS7_verify(x0, x1, x2, x3, x4, x5);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_PKCS7_verify(PyObject *self, PyObject *args)
{
  PKCS7 * x0;
  Cryptography_STACK_OF_X509 * x1;
  X509_STORE * x2;
  BIO * x3;
  BIO * x4;
  int x5;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;
  PyObject *arg4;
  PyObject *arg5;

  if (!PyArg_UnpackTuple(args, "PKCS7_verify", 6, 6, &arg0, &arg1, &arg2, &arg3, &arg4, &arg5))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(442), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (PKCS7 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(442), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(413), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_X509 *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(413), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(180), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (X509_STORE *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(180), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg4, (char **)&x4);
  if (datasize != 0) {
    x4 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg4, (char **)&x4,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x5 = _cffi_to_c_int(arg5, int);
  if (x5 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = PKCS7_verify(x0, x1, x2, x3, x4, x5); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_PKCS7_verify _cffi_d_PKCS7_verify
#endif

static void _cffi_d_POLICYINFO_free(POLICYINFO * x0)
{
  POLICYINFO_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_POLICYINFO_free(PyObject *self, PyObject *arg0)
{
  POLICYINFO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1091), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (POLICYINFO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1091), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { POLICYINFO_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_POLICYINFO_free _cffi_d_POLICYINFO_free
#endif

static POLICYINFO * _cffi_d_POLICYINFO_new(void)
{
  return POLICYINFO_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_POLICYINFO_new(PyObject *self, PyObject *noarg)
{
  POLICYINFO * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = POLICYINFO_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1091));
  return pyresult;
}
#else
#  define _cffi_f_POLICYINFO_new _cffi_d_POLICYINFO_new
#endif

static void _cffi_d_POLICYQUALINFO_free(POLICYQUALINFO * x0)
{
  POLICYQUALINFO_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_POLICYQUALINFO_free(PyObject *self, PyObject *arg0)
{
  POLICYQUALINFO * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1098), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (POLICYQUALINFO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1098), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { POLICYQUALINFO_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_POLICYQUALINFO_free _cffi_d_POLICYQUALINFO_free
#endif

static POLICYQUALINFO * _cffi_d_POLICYQUALINFO_new(void)
{
  return POLICYQUALINFO_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_POLICYQUALINFO_new(PyObject *self, PyObject *noarg)
{
  POLICYQUALINFO * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = POLICYQUALINFO_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1098));
  return pyresult;
}
#else
#  define _cffi_f_POLICYQUALINFO_new _cffi_d_POLICYQUALINFO_new
#endif

static void _cffi_d_POLICY_CONSTRAINTS_free(POLICY_CONSTRAINTS * x0)
{
  POLICY_CONSTRAINTS_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_POLICY_CONSTRAINTS_free(PyObject *self, PyObject *arg0)
{
  POLICY_CONSTRAINTS * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2741), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (POLICY_CONSTRAINTS *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2741), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { POLICY_CONSTRAINTS_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_POLICY_CONSTRAINTS_free _cffi_d_POLICY_CONSTRAINTS_free
#endif

static POLICY_CONSTRAINTS * _cffi_d_POLICY_CONSTRAINTS_new(void)
{
  return POLICY_CONSTRAINTS_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_POLICY_CONSTRAINTS_new(PyObject *self, PyObject *noarg)
{
  POLICY_CONSTRAINTS * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = POLICY_CONSTRAINTS_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(2741));
  return pyresult;
}
#else
#  define _cffi_f_POLICY_CONSTRAINTS_new _cffi_d_POLICY_CONSTRAINTS_new
#endif

static void _cffi_d_RAND_add(void const * x0, int x1, double x2)
{
  RAND_add(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RAND_add(PyObject *self, PyObject *args)
{
  void const * x0;
  int x1;
  double x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "RAND_add", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(126), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (void const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(126), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  x2 = (double)_cffi_to_c_double(arg2);
  if (x2 == (double)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { RAND_add(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_RAND_add _cffi_d_RAND_add
#endif

static int _cffi_d_RAND_bytes(unsigned char * x0, int x1)
{
  return RAND_bytes(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RAND_bytes(PyObject *self, PyObject *args)
{
  unsigned char * x0;
  int x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "RAND_bytes", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RAND_bytes(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RAND_bytes _cffi_d_RAND_bytes
#endif

static int _cffi_d_RAND_set_rand_method(RAND_METHOD const * x0)
{
  return RAND_set_rand_method(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RAND_set_rand_method(PyObject *self, PyObject *arg0)
{
  RAND_METHOD const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1618), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RAND_METHOD const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1618), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RAND_set_rand_method(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RAND_set_rand_method _cffi_d_RAND_set_rand_method
#endif

static int _cffi_d_RAND_status(void)
{
  return RAND_status();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RAND_status(PyObject *self, PyObject *noarg)
{
  int result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RAND_status(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  return pyresult;
}
#else
#  define _cffi_f_RAND_status _cffi_d_RAND_status
#endif

static RSA * _cffi_d_RSAPublicKey_dup(RSA * x0)
{
  return RSAPublicKey_dup(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSAPublicKey_dup(PyObject *self, PyObject *arg0)
{
  RSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  RSA * result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(476), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(476), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSAPublicKey_dup(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(476));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSAPublicKey_dup _cffi_d_RSAPublicKey_dup
#endif

static int _cffi_d_RSA_blinding_on(RSA * x0, BN_CTX * x1)
{
  return RSA_blinding_on(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_blinding_on(PyObject *self, PyObject *args)
{
  RSA * x0;
  BN_CTX * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "RSA_blinding_on", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(476), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(476), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(94), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BN_CTX *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(94), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_blinding_on(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_blinding_on _cffi_d_RSA_blinding_on
#endif

static int _cffi_d_RSA_check_key(RSA const * x0)
{
  return RSA_check_key(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_check_key(PyObject *self, PyObject *arg0)
{
  RSA const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(967), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(967), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_check_key(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_check_key _cffi_d_RSA_check_key
#endif

static void _cffi_d_RSA_free(RSA * x0)
{
  RSA_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_free(PyObject *self, PyObject *arg0)
{
  RSA * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(476), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(476), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { RSA_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_RSA_free _cffi_d_RSA_free
#endif

static int _cffi_d_RSA_generate_key_ex(RSA * x0, int x1, BIGNUM * x2, BN_GENCB * x3)
{
  return RSA_generate_key_ex(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_generate_key_ex(PyObject *self, PyObject *args)
{
  RSA * x0;
  int x1;
  BIGNUM * x2;
  BN_GENCB * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "RSA_generate_key_ex", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(476), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(476), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, int);
  if (x1 == (int)-1 && PyErr_Occurred())
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(848), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BN_GENCB *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(848), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_generate_key_ex(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_generate_key_ex _cffi_d_RSA_generate_key_ex
#endif

static void _cffi_d_RSA_get0_crt_params(RSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2, BIGNUM const * * x3)
{
  RSA_get0_crt_params(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_get0_crt_params(PyObject *self, PyObject *args)
{
  RSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  BIGNUM const * * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "RSA_get0_crt_params", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(967), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(967), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { RSA_get0_crt_params(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_RSA_get0_crt_params _cffi_d_RSA_get0_crt_params
#endif

static void _cffi_d_RSA_get0_factors(RSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2)
{
  RSA_get0_factors(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_get0_factors(PyObject *self, PyObject *args)
{
  RSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "RSA_get0_factors", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(967), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(967), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { RSA_get0_factors(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_RSA_get0_factors _cffi_d_RSA_get0_factors
#endif

static void _cffi_d_RSA_get0_key(RSA const * x0, BIGNUM const * * x1, BIGNUM const * * x2, BIGNUM const * * x3)
{
  RSA_get0_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_get0_key(PyObject *self, PyObject *args)
{
  RSA const * x0;
  BIGNUM const * * x1;
  BIGNUM const * * x2;
  BIGNUM const * * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "RSA_get0_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(967), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(967), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(2631), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM const * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(2631), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { RSA_get0_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_RSA_get0_key _cffi_d_RSA_get0_key
#endif

static RSA * _cffi_d_RSA_new(void)
{
  return RSA_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_new(PyObject *self, PyObject *noarg)
{
  RSA * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(476));
  return pyresult;
}
#else
#  define _cffi_f_RSA_new _cffi_d_RSA_new
#endif

static int _cffi_d_RSA_print(BIO * x0, RSA const * x1, int x2)
{
  return RSA_print(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_print(PyObject *self, PyObject *args)
{
  BIO * x0;
  RSA const * x1;
  int x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "RSA_print", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(186), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (BIO *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(186), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(967), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (RSA const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(967), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, int);
  if (x2 == (int)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_print(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_print _cffi_d_RSA_print
#endif

static int _cffi_d_RSA_set0_crt_params(RSA * x0, BIGNUM * x1, BIGNUM * x2, BIGNUM * x3)
{
  return RSA_set0_crt_params(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_set0_crt_params(PyObject *self, PyObject *args)
{
  RSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "RSA_set0_crt_params", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(476), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(476), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_set0_crt_params(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_set0_crt_params _cffi_d_RSA_set0_crt_params
#endif

static int _cffi_d_RSA_set0_factors(RSA * x0, BIGNUM * x1, BIGNUM * x2)
{
  return RSA_set0_factors(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_set0_factors(PyObject *self, PyObject *args)
{
  RSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "RSA_set0_factors", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(476), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(476), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_set0_factors(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_set0_factors _cffi_d_RSA_set0_factors
#endif

static int _cffi_d_RSA_set0_key(RSA * x0, BIGNUM * x1, BIGNUM * x2, BIGNUM * x3)
{
  return RSA_set0_key(x0, x1, x2, x3);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_RSA_set0_key(PyObject *self, PyObject *args)
{
  RSA * x0;
  BIGNUM * x1;
  BIGNUM * x2;
  BIGNUM * x3;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;
  PyObject *arg3;

  if (!PyArg_UnpackTuple(args, "RSA_set0_key", 4, 4, &arg0, &arg1, &arg2, &arg3))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(476), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (RSA *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(476), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg2, (char **)&x2);
  if (datasize != 0) {
    x2 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg2, (char **)&x2,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(20), arg3, (char **)&x3);
  if (datasize != 0) {
    x3 = ((size_t)datasize) <= 640 ? (BIGNUM *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(20), arg3, (char **)&x3,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = RSA_set0_key(x0, x1, x2, x3); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_RSA_set0_key _cffi_d_RSA_set0_key
#endif

static void _cffi_d_SCT_LIST_free(Cryptography_STACK_OF_SCT * x0)
{
  SCT_LIST_free(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_LIST_free(PyObject *self, PyObject *arg0)
{
  Cryptography_STACK_OF_SCT * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1101), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (Cryptography_STACK_OF_SCT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1101), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SCT_LIST_free(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SCT_LIST_free _cffi_d_SCT_LIST_free
#endif

static size_t _cffi_d_SCT_get0_log_id(SCT const * x0, unsigned char * * x1)
{
  return SCT_get0_log_id(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_get0_log_id(PyObject *self, PyObject *args)
{
  SCT const * x0;
  unsigned char * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SCT_get0_log_id", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(747), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SCT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(747), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(802), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(802), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SCT_get0_log_id(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SCT_get0_log_id _cffi_d_SCT_get0_log_id
#endif

static size_t _cffi_d_SCT_get0_signature(SCT const * x0, unsigned char * * x1)
{
  return SCT_get0_signature(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_get0_signature(PyObject *self, PyObject *args)
{
  SCT const * x0;
  unsigned char * * x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  size_t result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SCT_get0_signature", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(747), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SCT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(747), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(802), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char * *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(802), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SCT_get0_signature(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, size_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SCT_get0_signature _cffi_d_SCT_get0_signature
#endif

static ct_log_entry_type_t _cffi_d_SCT_get_log_entry_type(SCT const * x0)
{
  return SCT_get_log_entry_type(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_get_log_entry_type(PyObject *self, PyObject *arg0)
{
  SCT const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  ct_log_entry_type_t result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(747), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SCT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(747), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SCT_get_log_entry_type(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_deref((char *)&result, _cffi_type(1646));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SCT_get_log_entry_type _cffi_d_SCT_get_log_entry_type
#endif

static uint64_t _cffi_d_SCT_get_timestamp(SCT const * x0)
{
  return SCT_get_timestamp(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_get_timestamp(PyObject *self, PyObject *arg0)
{
  SCT const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  uint64_t result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(747), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SCT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(747), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SCT_get_timestamp(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, uint64_t);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SCT_get_timestamp _cffi_d_SCT_get_timestamp
#endif

static sct_version_t _cffi_d_SCT_get_version(SCT const * x0)
{
  return SCT_get_version(x0);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_get_version(PyObject *self, PyObject *arg0)
{
  SCT const * x0;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  sct_version_t result;
  PyObject *pyresult;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(747), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SCT const *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(747), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SCT_get_version(x0); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_deref((char *)&result, _cffi_type(1654));
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SCT_get_version _cffi_d_SCT_get_version
#endif

static SCT * _cffi_d_SCT_new(void)
{
  return SCT_new();
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_new(PyObject *self, PyObject *noarg)
{
  SCT * result;
  PyObject *pyresult;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SCT_new(); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  (void)noarg; /* unused */
  pyresult = _cffi_from_c_pointer((char *)result, _cffi_type(1102));
  return pyresult;
}
#else
#  define _cffi_f_SCT_new _cffi_d_SCT_new
#endif

static int _cffi_d_SCT_set1_log_id(SCT * x0, unsigned char * x1, size_t x2)
{
  return SCT_set1_log_id(x0, x1, x2);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_set1_log_id(PyObject *self, PyObject *args)
{
  SCT * x0;
  unsigned char * x1;
  size_t x2;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;
  PyObject *arg2;

  if (!PyArg_UnpackTuple(args, "SCT_set1_log_id", 3, 3, &arg0, &arg1, &arg2))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1102), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SCT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1102), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(752), arg1, (char **)&x1);
  if (datasize != 0) {
    x1 = ((size_t)datasize) <= 640 ? (unsigned char *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(752), arg1, (char **)&x1,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x2 = _cffi_to_c_int(arg2, size_t);
  if (x2 == (size_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SCT_set1_log_id(x0, x1, x2); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SCT_set1_log_id _cffi_d_SCT_set1_log_id
#endif

static int _cffi_d_SCT_set_log_entry_type(SCT * x0, ct_log_entry_type_t x1)
{
  return SCT_set_log_entry_type(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_set_log_entry_type(PyObject *self, PyObject *args)
{
  SCT * x0;
  ct_log_entry_type_t x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SCT_set_log_entry_type", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1102), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SCT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1102), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  if (_cffi_to_c((char *)&x1, _cffi_type(1646), arg1) < 0)
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SCT_set_log_entry_type(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SCT_set_log_entry_type _cffi_d_SCT_set_log_entry_type
#endif

static int _cffi_d_SCT_set_source(SCT * x0, sct_source_t x1)
{
  return SCT_set_source(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_set_source(PyObject *self, PyObject *args)
{
  SCT * x0;
  sct_source_t x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  int result;
  PyObject *pyresult;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SCT_set_source", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1102), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SCT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1102), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  if (_cffi_to_c((char *)&x1, _cffi_type(1650), arg1) < 0)
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { result = SCT_set_source(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  pyresult = _cffi_from_c_int(result, int);
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  return pyresult;
}
#else
#  define _cffi_f_SCT_set_source _cffi_d_SCT_set_source
#endif

static void _cffi_d_SCT_set_timestamp(SCT * x0, uint64_t x1)
{
  SCT_set_timestamp(x0, x1);
}
#ifndef PYPY_VERSION
static PyObject *
_cffi_f_SCT_set_timestamp(PyObject *self, PyObject *args)
{
  SCT * x0;
  uint64_t x1;
  Py_ssize_t datasize;
  struct _cffi_freeme_s *large_args_free = NULL;
  PyObject *arg0;
  PyObject *arg1;

  if (!PyArg_UnpackTuple(args, "SCT_set_timestamp", 2, 2, &arg0, &arg1))
    return NULL;

  datasize = _cffi_prepare_pointer_call_argument(
      _cffi_type(1102), arg0, (char **)&x0);
  if (datasize != 0) {
    x0 = ((size_t)datasize) <= 640 ? (SCT *)alloca((size_t)datasize) : NULL;
    if (_cffi_convert_array_argument(_cffi_type(1102), arg0, (char **)&x0,
            datasize, &large_args_free) < 0)
      return NULL;
  }

  x1 = _cffi_to_c_int(arg1, uint64_t);
  if (x1 == (uint64_t)-1 && PyErr_Occurred())
    return NULL;

  Py_BEGIN_ALLOW_THREADS
  _cffi_restore_errno();
  { SCT_set_timestamp(x0, x1); }
  _cffi_save_errno();
  Py_END_ALLOW_THREADS

  (void)self; /* unused */
  if (large_args_free != NULL) _cffi_free_array_arguments(large_args_free);
  Py_INCREF(Py_None);
  return Py_None;
}
#else
#  define _cffi_f_SCT_set_timestamp _cffi_d_SCT_set_timesta