﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm-quicksetup/SSMQuickSetupRequest.h>
#include <aws/ssm-quicksetup/SSMQuickSetup_EXPORTS.h>
#include <aws/ssm-quicksetup/model/Filter.h>

#include <utility>

namespace Aws {
namespace SSMQuickSetup {
namespace Model {

/**
 */
class ListConfigurationManagersRequest : public SSMQuickSetupRequest {
 public:
  AWS_SSMQUICKSETUP_API ListConfigurationManagersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListConfigurationManagers"; }

  AWS_SSMQUICKSETUP_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Filters the results returned by the request.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  ListConfigurationManagersRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  ListConfigurationManagersRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the maximum number of configuration managers that are returned by
   * the request.</p>
   */
  inline int GetMaxItems() const { return m_maxItems; }
  inline bool MaxItemsHasBeenSet() const { return m_maxItemsHasBeenSet; }
  inline void SetMaxItems(int value) {
    m_maxItemsHasBeenSet = true;
    m_maxItems = value;
  }
  inline ListConfigurationManagersRequest& WithMaxItems(int value) {
    SetMaxItems(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token to use when requesting a specific set of items from a list.</p>
   */
  inline const Aws::String& GetStartingToken() const { return m_startingToken; }
  inline bool StartingTokenHasBeenSet() const { return m_startingTokenHasBeenSet; }
  template <typename StartingTokenT = Aws::String>
  void SetStartingToken(StartingTokenT&& value) {
    m_startingTokenHasBeenSet = true;
    m_startingToken = std::forward<StartingTokenT>(value);
  }
  template <typename StartingTokenT = Aws::String>
  ListConfigurationManagersRequest& WithStartingToken(StartingTokenT&& value) {
    SetStartingToken(std::forward<StartingTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Filter> m_filters;

  int m_maxItems{0};

  Aws::String m_startingToken;
  bool m_filtersHasBeenSet = false;
  bool m_maxItemsHasBeenSet = false;
  bool m_startingTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSMQuickSetup
}  // namespace Aws
