/*
 * // Copyright (c) Radzivon Bartoshyk 7/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
Y0 zeros and extremums.
Generated by SageMath:
```python
R120 = RealField(120)

zeros = []

mp.prec = 150

step = mpf("0.1")
epsilon = mpf("1e-35")
x = mpf("1.25")

previous_zero = R120(0)
y0_zeros = []

while x < mpf("76.0"):
    f1 = bessely(0, x)
    f2 = bessely(0, x + step)
    if f1 * f2 < 0:
        zero = findroot(lambda t: bessely(0, t), (x, x + step), solver='secant', tol=mp.mpf("1e-41"))
        previous_zero = zero
        y0_zeros.append(zero)
    if previous_zero is not None and abs(x - mpf(f'{round(x)}')) < epsilon:
        zeros.append(previous_zero)
    x += step

y0_extrema = []

x = mpf("1.25")
while x < mpf("76.0"):
    d1 = mp.diff(lambda t: bessely(0, t), x)
    d2 = mp.diff(lambda t: bessely(0, t), x + step)
    if d1 * d2 < 0:
        extremum = findroot(lambda t: mp.diff(lambda u: bessely(0, u), t), (x, x + step), solver='secant', tol=mp.mpf("1e-41"))
        y0_extrema.append(extremum)
    x += step

# Print results
for i, z in enumerate(j0_zeros):
    print(f"Zero {i+1}: x ≈ {z}")

print("Extrema (peaks/valleys) of Y0(x):")
for e in y0_extrema:
    print(f"nExtrema: {e}")

y0_zeros.extend(y0_extrema)

y0_zeros = sorted(y0_zeros)

# Print results
for i, z in enumerate(y0_zeros):
    print(f"Peak or zero {i+1}: x ≈ {z}")

print("")

print("pub(crate) static Y0_ZEROS: [(u64, u64); 48] = [")
print(f"(0x0, 0x0),")
for z in y0_zeros:
    k = split_double_double(z)
    hi = double_to_hex(k[1])
    lo = double_to_hex(k[0])
    print(f"({lo}, {hi}),")

print("];")
```
**/
pub(crate) static Y0_ZEROS: [(u64, u64); 48] = [
    (0x0, 0x0), // not really used, just a stab to avoid messing with indices
    (0xbc8bd1e50d219bfd, 0x400193bed4dff243),
    (0xbc9f06ae7804384e, 0x400fa9534d98569c),
    (0x3cbdfe7bac228e8c, 0x4015b7fe4e87b02e),
    (0xbc99774a495f56cf, 0x401c581dc4e72103),
    (0x3cb479cc068d9046, 0x40213127ae6169b4),
    (0xbcccb49ff791c495, 0x402471d735a47d58),
    (0x3c80fc786ce06080, 0x40277f9138d43206),
    (0xbcc7df81de86f24d, 0x402ab8e1c4a1e74a),
    (0xbcc5e091a50f8e05, 0x402dcb7d88de848b),
    (0x3cd25a237d12159b, 0x4030803c74003214),
    (0xbcda1ee4c5487ede, 0x40320b1c695f1e3b),
    (0xbcd8bf92d51fbaeb, 0x4033a42cdf5febd7),
    (0x3cd391b14410528f, 0x40353025492188cd),
    (0x3cdca75080cf53a8, 0x4036c832fd77ac07),
    (0x3cb52f75f025b205, 0x403854fa303820ca),
    (0xbcb03e052bd9c0af, 0x4039ec46f3e80146),
    (0xbcbcf130fbea3b24, 0x403b79acee8cfb7d),
    (0x3cd0aab17eca74b9, 0x403d106449616c4f),
    (0xbc9e7a77047d6166, 0x403e9e480605283c),
    (0x3ced2f18aa8a8f2f, 0x40401a4420e4abee),
    (0xbce96beabef7ecf4, 0x4040e16907f8fb56),
    (0xbca9dd1578036d11, 0x4041ac588c944279),
    (0x3cd2481e87adfe57, 0x404273a7b35a7aff),
    (0xbce9c3dd43e59158, 0x40433e6ecf5cb221),
    (0x3cda8ffacaac8461, 0x404405e18393afb5),
    (0xbce753b7fcd5250c, 0x4044d0867ec213f3),
    (0x3cbfe463face2c1c, 0x4045981787d668db),
    (0x3cec2eb6ee3e4c70, 0x4046629f4e1e0321),
    (0xbce26390f25f01cb, 0x40472a4a85cc317e),
    (0xbcaafdee84ced526, 0x4047f4b904dc9a53),
    (0xbcdcc667e557a177, 0x4048bc7b10ed3960),
    (0x3cebbca9a96dc1aa, 0x404986d378522b70),
    (0x3cb377717d2f36f7, 0x404a4ea9997b5eaa),
    (0xbcedf185c89a4066, 0x404b18ee87b4e700),
    (0xbcc68a841a2af000, 0x404be0d6766d13de),
    (0x3ced8a878e90eda4, 0x404cab0a1968b221),
    (0x3cddcc8dd083c434, 0x404d7301ec2bf16e),
    (0xbceb5c58b22794af, 0x404e3d2619226875),
    (0xbce5919b54336657, 0x404f052c3146d1da),
    (0xbc87b54cc765fd11, 0x404fcf4276983013),
    (0xbcf389f975d9dfa2, 0x40504baab8e42f8f),
    (0xbce195534d06b08a, 0x4050b0af92483bf5),
    (0xbcd4e0ffa0b9d2e9, 0x405114bee8d0f4bd),
    (0xbcf1c076cb371565, 0x405179be0c18f0e1),
    (0x3cf5d0de43102daa, 0x4051ddd2b7398636),
    (0xbce8a4f19e47d0be, 0x405242cca44048f8),
    (0x3ceb379dbe48b8f5, 0x4052a6e6306f1d08),
];

/**
Zeros and extremums solved values for Y0

Generated by MPFR:
```text
let mut arr = vec![];
for zeros in Y0_ZEROS.iter() {
    if zeros.1 == 0 {
        arr.push(0);
    } else {
        let mpfr = Float::with_val(107, f64::from_bits(zeros.1)).y0();
        arr.push(mpfr.to_f64().to_bits());
    }
}
println!(
    "arr: [{}]",
    arr.iter()
        .map(|x| format!("0x{:016x}", x))
        .collect::<Vec<_>>()
        .join(", ")
);
```
**/
pub(crate) static Y0_ZEROS_VALUES: [u64; 48] = [
    0x0000000000000000,
    0x3fe0aa48442f014b,
    0xbc88fa8956b4b481,
    0xbfd5c7c556f0c19a,
    0x3c7e91b198d39ce2,
    0x3fd15f993fceab5c,
    0xbcacabd7c823bf21,
    0xbfcdc14ea14e89f9,
    0x3ca4d9fe48a91cef,
    0x3fca7022be084d99,
    0x3cacd827f6c074a2,
    0xbfc80781c32422e7,
    0x3cb1d336c8b0a153,
    0x3fc62d94d97e859c,
    0x3cb3296749aaf800,
    0xbfc4b2a38f1ab9b4,
    0x3c845d52c7c075b2,
    0x3fc37aaceac987b9,
    0x3ca3bc249963d4c1,
    0xbfc2740819f1caaa,
    0xbcc069eacea8ff0b,
    0x3fc192f2627a74e3,
    0xbc7bc4fd81e2b2c5,
    0xbfc0cf3ee98f769b,
    0x3cba82b3a8997d82,
    0x3fc0230ba90f2871,
    0xbcb71414f52d813b,
    0xbfbf13fb0c0e6fcd,
    0xbcbae2d399eafcc6,
    0x3fbe018dac1c17e3,
    0xbc78e44442565953,
    0xbfbd09b21e36c0bd,
    0xbcb8c779e4075b10,
    0x3fbc2861347b1b39,
    0xbcb9f6a4816aab2e,
    0xbfbb5a622198a72c,
    0xbcb8e727b71a083d,
    0x3fba9d183bc04545,
    0xbcb67533047d83fb,
    0xbfb9ee5eee1a97c6,
    0x3c52f9257b962770,
    0x3fb94c6f5898708b,
    0xbcab7916830bd3b8,
    0xbfb8b5ccb03d459b,
    0x3cbb1b180f8e9062,
    0x3fb829356c2fb67c,
    0xbcb267f22477b55d,
    0xbfb7a597eb76a5e3,
];

/**
Taylor series at different zeros and extremums for Y0
Generated by SageMath:
```python
def compute_intervals(zeros):
    intervals = []
    for i in range(0, len(zeros)):
        if i == 0:
            a = 2 - zeros[i]
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.05 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        elif i + 1 > len(zeros) - 1:
            a = (zeros[i - 1] + zeros[i]) / 2 - 0.05 - zeros[i]
            b = (zeros[i]) + 0.83 + 0.05 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        else:
            a = (zeros[i - 1] + zeros[i]) / 2 - zeros[i] - 0.05
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.05  - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
    return intervals

intervals = compute_intervals(y0_zeros)
print(intervals)

def build_sollya_script(a, b, zero, deg):
    return f"""
prec = 500;
bessel_y0 = library("./notes/bessel_sollya/cmake-build-release/libbessel_sollya.dylib");
f = bessel_y0(x + {zero});
d = [{a}, {b}];
pf = remez(f, {deg}, d);
for i from 0 to degree(pf) do {{
    write(coeff(pf, i)) >> "coefficients.txt";
    write("\\n") >> "coefficients.txt";
}};
"""

def load_coefficients(filename):
    with open(filename, "r") as f:
        return [RR(line.strip()) for line in f if line.strip()]

def call_sollya_on_interval(a, b, zero, degree=12):
    sollya_script = build_sollya_script(a, b, zero, degree)
    with open("tmp_interval.sollya", "w") as f:
        f.write(sollya_script)
    import subprocess
    if os.path.exists("coefficients.txt"):
        os.remove("coefficients.txt")
    try:
        result = subprocess.run(
            ["sollya", "tmp_interval.sollya"],
            check=True,
            capture_output=True,
            text=True
        )
    except subprocess.CalledProcessError as e:
        return

degree = 17

print(f"pub(crate) static Y0F_COEFFS: [[u64;{degree + 1}]; {len(intervals)}] = [")
for i in range(0, len(intervals)):
    interval = intervals[i]
    call_sollya_on_interval(interval[0], interval[1], interval[2], degree)
    coeffs = load_coefficients(f"coefficients.txt")
    print("[")
    for c in coeffs:
        print(double_to_hex(c) + ",")
    print("],")
print("];")
```
**/
pub(crate) static Y0F_COEFFS: [[u64; 18]; 47] = [
    [
        0x3fe0aa48442f014b,
        0xbc009e915c2e5f95,
        0xbfd0aa48442f0143,
        0x3fa439fac165257c,
        0x3f80d2af4e929791,
        0x3f4f71646c0dc646,
        0xbf5444bda7f2ee4c,
        0x3f384c21f1d0de42,
        0xbf217ab4d69ab0db,
        0x3f0dafc989c8ea40,
        0xbef8bbefbac0d893,
        0x3ee48ac7a622784f,
        0xbed1245593770ad0,
        0x3ebc28878607b687,
        0xbea585c089718cc9,
        0x3e8b7ad75bd333d8,
        0xbe68a20d8d3d57a5,
        0x3e362c70b36d12a4,
    ],
    [
        0x3bf75cf88f1b7237,
        0xbfd9c34256a12a0c,
        0x3faa09c9290367ec,
        0x3fadf6d59bf50e4b,
        0xbf7c116fdc597e2f,
        0xbf61e32bc4eef0b0,
        0x3f29982765136b27,
        0x3f0ab2c1fd5871c0,
        0xbed486371d6e71eb,
        0xbe93b212b9208b3e,
        0xbe577700c20e01fd,
        0x3e545234fb052f6a,
        0xbe2f6a5f736d8c82,
        0x3e0a8134a3ddd47f,
        0xbde9666a74f1504a,
        0x3dc63b75b90c0da3,
        0xbdaa991122c61028,
        0x3d930f1a31e30006,
    ],
    [
        0xbfd5c7c556f0c19a,
        0xbbc44b457ab6a440,
        0x3fc5c7c556f0c19a,
        0xbf8564d4b1ed0d7e,
        0xbf8a15d92dfe3e28,
        0x3f4b438430469f6e,
        0x3f37a8924cc2fcab,
        0xbef5f69b4a89930d,
        0xbed85b940ebd0c07,
        0x3e955ac0f4e7f8ab,
        0x3e6d563e284b0072,
        0xbe23eac267e6cf3c,
        0xbe00077161c32b55,
        0x3dc04d965b688847,
        0xbd45e4f4886f7d51,
        0x3d5501605dba787a,
        0xbd363e6119f047ec,
        0x3d096861e7e8004b,
    ],
    [
        0xbb236a0ad5bec7c3,
        0x3fd334cca0697a5b,
        0xbf95aef611fc4d57,
        0xbfa8969c64cbf452,
        0x3f6b2f14a95527cb,
        0x3f61d35e85fde2a4,
        0xbf226dd71e391c65,
        0xbf08177e4fe524db,
        0x3ec6a9227352b13b,
        0x3ea34aa757412911,
        0xbe60a2814d2e238a,
        0xbe3442a3d67717ae,
        0x3defa6c448c9759c,
        0x3dbf14eacd6e3a21,
        0xbd771434b676ae83,
        0xbd406a106470261f,
        0x3cf2c5324aed291b,
        0x3cc47f208256e17f,
    ],
    [
        0x3fd15f993fceab5c,
        0xbb597aaa5d4472ea,
        0xbfc15f993fceab5c,
        0x3f758ef6efbed797,
        0x3f86395dfe49fcd4,
        0xbf3fb15104a41c03,
        0xbf35f88a11d55641,
        0x3eef37d226a827fd,
        0x3ed6f7bab104d878,
        0xbe8f0c45a3868f32,
        0xbe6dfe11ded99f62,
        0x3e2311adc6046d1a,
        0x3dfad34dd085ee0f,
        0xbdafdc85833041dc,
        0xbd819463fb6a5c22,
        0x3d33a4c369438250,
        0x3d014560b8776112,
        0xbcb2ae15d3a41e47,
    ],
    [
        0x3b12ba7db6ce5807,
        0xbfcff635cc72b9f1,
        0x3f89036451ff57c5,
        0x3fa4e667a71556af,
        0xbf60325ee41e910c,
        0xbf5fe23914fb9128,
        0x3f17f84d7c50e49e,
        0x3f06afdd57be1e75,
        0xbec04053abf415e2,
        0xbea2aea9ec4954d7,
        0x3e593eb9f19579b5,
        0x3e3428a3a4917f21,
        0xbde99d8be9e4cd5a,
        0xbdbec9815ee61f19,
        0x3d7260f619cb637c,
        0x3d418f963ce687db,
        0xbcf38a6e7195daa0,
        0xbcbf332d386dfb1b,
    ],
    [
        0xbfcdc14ea14e89f9,
        0x3b52ef4e10970973,
        0x3fbdc14ea14e89f9,
        0xbf6b037fe9cf2a52,
        0xbf8367d7d608e4ba,
        0x3f34abef5636e4f9,
        0x3f33d8a66122924a,
        0xbee5cfe92a1a2ead,
        0xbed571814a1a8669,
        0x3e87414e33ccd6d7,
        0x3e6ca7069e37c5b4,
        0xbe1e0b3a74e3f35e,
        0xbdfa15dd188a976e,
        0x3daa33983e3b34b5,
        0x3d8141656fbf9e02,
        0xbd3089bd5153b08a,
        0xbd01345e91fc8d5c,
        0x3cb05566655c2807,
    ],
    [
        0xbb10bed1f7905ee9,
        0x3fcbf32a27594007,
        0xbf80bc2d84e65214,
        0xbfa26cab38a8b368,
        0x3f55f03e47165d72,
        0x3f5caaa76e34992d,
        0xbf10c5f18c46d00f,
        0xbf04f0af7d46cd82,
        0x3eb7d1e28094a46b,
        0x3ea1ad073122cd33,
        0xbe536021c88affe4,
        0xbe3371ae6cd96849,
        0x3de46ce028e7c109,
        0x3dbe235dc856ef66,
        0xbd6e401d592ca47a,
        0xbd415f37788a98a4,
        0x3cf078a1b34fa822,
        0x3cbefbc3ca8cda9f,
    ],
    [
        0x3fca7022be084d99,
        0xbb4ddca3665cb95a,
        0xbfba7022be084d99,
        0x3f62ee079d020b12,
        0x3f8163191c30aa62,
        0xbf2d7806ea72fc7a,
        0xbf320f95702b1d3e,
        0x3ee006108822968d,
        0x3ed3e398cbc455aa,
        0xbe81bbe181c8bfbe,
        0xbe6b0f89b78908bf,
        0x3e17c3f85c062f0a,
        0x3df902361639d835,
        0xbda56494e7b59641,
        0xbd80bde23236fbba,
        0x3d2bbbd3bb25cf30,
        0x3d00d7f96ac5af92,
        0xbcabe3861f527a48,
    ],
    [
        0x3b0d39acdbc38af2,
        0xbfc925c35988ee29,
        0x3f7862549367591e,
        0x3fa0a4512039d6a2,
        0xbf5013b38cfb9292,
        0xbf5a24a1215f6684,
        0x3f08f91421377f72,
        0x3f035d17cec01753,
        0xbeb2283a93110ac0,
        0xbea099e713932038,
        0x3e4e5de01dbe7d10,
        0x3e3288585522eec4,
        0xbde0730be51a22c7,
        0xbdbd174416e1daeb,
        0x3d68f810d9b70d7c,
        0x3d40f177553ea68b,
        0xbcebc58c79d9f1a6,
        0xbcbe6305e9d630f0,
    ],
    [
        0xbfc80781c32422e7,
        0x3b483f69e003254a,
        0x3fb80781c32422e7,
        0xbf5c6923374d561f,
        0xbf7fbe6df840847f,
        0x3f264f4711a85f1d,
        0x3f309ff42b0d7a58,
        0xbed8a60685a59ae7,
        0xbed282d26a74a669,
        0x3e7bdb57a88d9298,
        0x3e697d9c12695952,
        0xbe131aa86bece7b9,
        0xbdf7d486a4891297,
        0x3da197f4a31bb9d7,
        0x3d801da2d1e74508,
        0xbd2749f32e6b29e4,
        0xbd005a3044bf379a,
        0x3ca7cdb1346a2761,
    ],
    [
        0xbb095b0f9cce8be6,
        0x3fc70c4f66cab47f,
        0xbf72c6731071e936,
        0xbf9e924b85a17361,
        0x3f48d6c364d92082,
        0x3f58247b02d6b0f6,
        0xbf037612593855eb,
        0xbf0206da232a2b7e,
        0x3eacaf84db86b79e,
        0x3e9f367117733794,
        0xbe48654affaeff8a,
        0xbe3199ec88cf0125,
        0x3ddaeb1a0e170f3e,
        0x3dbbe67c38767c0a,
        0xbd64cda59f058a2d,
        0xbd406490f1401c35,
        0x3ce7873ddbd2ecb8,
        0x3cbd9254c745332c,
    ],
    [
        0x3fc62d94d97e859c,
        0xbb4423d84fd9daf1,
        0xbfb62d94d97e859c,
        0x3f565481b55eaefb,
        0x3f7d5f857a2a6107,
        0xbf21a015a2ccb791,
        0xbf2eeb4eafd8612c,
        0x3ed3a7b79e4d8384,
        0x3ed154ed4598b648,
        0xbe767f762ea5d48d,
        0xbe680ec1e0075865,
        0x3e0f52965769de53,
        0x3df6ad7b894b88a3,
        0xbd9d4e5bf81e66fd,
        0xbd7eeb07104d446e,
        0x3d23b37b411dc13d,
        0x3cff991c92fdc4d8,
        0xbca4681dcbe602f7,
    ],
    [
        0x3b0608e1eb7299ca,
        0xbfc5664d37c37d7b,
        0x3f6e0ee8ec84659a,
        0x3f9c6c415c971b4b,
        0xbf43ec49cb941f55,
        0xbf56853b2d047885,
        0x3eff632a9396fcf6,
        0x3f00e6afa01af56c,
        0xbea7541247ac8e20,
        0xbe9d735e8beba89d,
        0x3e440e6a7c5b76f7,
        0x3e30b9cfae36d52a,
        0xbdd66808178f0472,
        0xbdbab498e9ad024c,
        0x3d618b0d40031fb2,
        0x3d3f99bd1527877f,
        0xbce4186fe1076e16,
        0xbcbca6e4a9f85609,
    ],
    [
        0xbfc4b2a38f1ab9b4,
        0x3b4106370f2ef866,
        0x3fb4b2a38f1ab9b4,
        0xbf5225a5c73f2233,
        0xbf7b750d89a9b35f,
        0x3f1cbdb4f1d5dbc0,
        0x3f2cfe933fc6d09a,
        0xbed01ef10d839ddf,
        0xbed05375a5888b31,
        0x3e729afe7eaa78e7,
        0x3e66c8fe01244a3c,
        0xbe0a2a01eba23e6a,
        0xbdf59b32be188f85,
        0x3d98c0a83d3ee202,
        0x3d7da2dd5a1ff0d5,
        0xbd20d5047e87a8a0,
        0xbcfe76ab94a96cf5,
        0x3ca19e4f3f768bb8,
    ],
    [
        0xbb033ddc20ef994b,
        0x3fc40f8ffdf09a5f,
        0xbf68c37a29c4586f,
        0xbf9aab099314b209,
        0x3f406f735cc0f6a6,
        0x3f552d29a06802e0,
        0xbef9fc04c675c0a0,
        0xbeffe48825ed3c99,
        0x3ea36bd2d58fad6c,
        0x3e9be87e2cad35c1,
        0xbe40d1fded7d33ea,
        0xbe2fdbeb107f2787,
        0x3dd2f44ca7722117,
        0x3db992136419242c,
        0xbd5df6388aabd242,
        0xbd3e6bd4176ef817,
        0x3ce15380b88a0004,
        0x3cbbb4210e2ade39,
    ],
    [
        0x3fc37aaceac987b9,
        0xbb3d355a175fbb67,
        0xbfb37aaceac987b9,
        0x3f4e3fdbfd65014a,
        0x3f79de7a33bc3a97,
        0xbf1801d911fbd06f,
        0xbf2b605a5ade3a62,
        0x3ecb0a2608144de7,
        0x3eceeceb341aa1c8,
        0xbe6f623fc7c35376,
        0xbe65a9ca94685dee,
        0x3e0638ba32c100f7,
        0x3df4a1b3dc97031d,
        0xbd9530c9cb91ec68,
        0xbd7c6deff6c29a1e,
        0x3d1d10f27ed0d91c,
        0x3cfd5b5110a971f2,
        0xbc9eac069b193638,
    ],
    [
        0x3b00e83df01ac491,
        0xbfc2f206e49909c7,
        0x3f64dbf6a9fb80d7,
        0x3f99336443318ed1,
        0xbf3bb6aa3d4e9e78,
        0xbf540aaa5d94bd8d,
        0x3ef5f61b66612962,
        0x3efe4158391f2c2d,
        0xbea07a7a0745b8f1,
        0xbe9a8ea97b66fe4d,
        0x3e3cb10b3a7e93b8,
        0x3e2e6d788791546c,
        0xbdd044442a2af39d,
        0xbdb8850c7d8525ad,
        0x3d59e48a6e2fcf83,
        0x3d3d4af5e70dabe3,
        0xbcde2a65241d7e39,
        0xbcbac578a90cdb0d,
    ],
    [
        0xbfc2740819f1caaa,
        0x3b395de3b03fa760,
        0x3fb2740819f1caaa,
        0xbf49b6f37d0a00f9,
        0xbf78868d7401bf2e,
        0x3f1470a7cbcb436d,
        0x3f29fe94ce3d3e66,
        0xbec71660e61f123b,
        0xbecd6dfcdb022b53,
        0x3e6ae79c35d4303c,
        0x3e64ac50be35d108,
        0xbe0325840c0fab26,
        0xbdf3c09642eaf47d,
        0x3d925d87f4bac822,
        0x3d7b50337a12ad98,
        0xbd195a5cf071931d,
        0xbcfc4e3bdd8deb9e,
        0x3c9aec36c7cd8057,
    ],
    [
        0xbafde3378387cd4f,
        0x3fc1ff5ebddd3c3a,
        0xbf61e2035324643c,
        0xbf97f3506d4a1231,
        0x3f37c65c9302c53a,
        0x3f53117816335151,
        0xbef2df9afa521258,
        0xbefcd5d4a9d78a10,
        0x3e9c672d4d6f090d,
        0x3e995e4b10218b56,
        0xbe38d22635a19a7c,
        0xbe2d24fc943fc6e5,
        0x3dcc462d12216cc2,
        0x3db78eaf8cc86453,
        0xbd569efa1b265a8d,
        0xbd3c3c11b66258b3,
        0x3cda808e4f387067,
        0x3cb9e0f8deb75911,
    ],
    [
        0x3fc192f2627a74e3,
        0xbb3645a3987074b0,
        0xbfb192f2627a74e3,
        0x3f4635b38affe698,
        0x3f775eceaabf7f86,
        0xbf11ac9e0164f7c3,
        0xbf28cc464a35b093,
        0x3ec4014d9bf38c8d,
        0x3ecc1f05a2d81e1e,
        0xbe6761d18ebe9900,
        0xbe63cb9af0ceb4cf,
        0x3e00b3cb584e2702,
        0x3df2f5c722e3bf8d,
        0xbd9017792eaec1aa,
        0xbd7a4a02574df6fb,
        0x3d1652dad940d7a4,
        0x3cfb525fce2f20bd,
        0xbc97d2bda2bb98bd,
    ],
    [
        0x3afa988afe4c49f9,
        0xbfc12dd55d4be2b3,
        0x3f5f1aee31818d19,
        0x3f96de64242a8310,
        0xbf34afdf89fca61a,
        0xbf5238cfc13ac771,
        0x3ef0719d13e00e1d,
        0x3efb974781a526ac,
        0xbe98cc82a70d12de,
        0xbe9850ae878c16d4,
        0x3e35bba73dc2363a,
        0x3e2bfe1396c7d24f,
        0xbdc8d7db68570364,
        0xbdb6adfbad375234,
        0x3d53f4c573239e3d,
        0x3d3b40924b9cec60,
        0xbcd77b88ddf3cd06,
        0xbcb909671eb81963,
    ],
    [
        0xbfc0cf3ee98f769b,
        0x3b33bcff5c943527,
        0x3fb0cf3ee98f769b,
        0xbf436f451f6e14fb,
        0xbf765d05948a946a,
        0x3f0ef55c5a0d16d1,
        0x3f27bfec9d15d01e,
        0xbec18c549f28e7a1,
        0xbecaf7544eea95a1,
        0x3e648e81edf6b6c0,
        0x3e630341e759775f,
        0xbdfd73d2cdd0aab5,
        0xbdf23ec233be72a4,
        0x3d8c78e7f536eeff,
        0x3d795a1608a24a32,
        0xbd13d34748616c7a,
        0xbcfa685ede39fb91,
        0x3c953d75ba0ff04b,
    ],
    [
        0xbaf7d336e374344e,
        0x3fc0768257dad56a,
        0xbf5b602a7beaaa48,
        0xbf95ebc22efd092c,
        0x3f3236a604142e61,
        0x3f517a482faa8d85,
        0xbeecfc008907874e,
        0xbefa7d7b128ac52d,
        0x3e95e419f1b51819,
        0x3e97603cea233f99,
        0xbe333a0b43111627,
        0xbe2af464175ecac5,
        0x3dc609ac1d6253eb,
        0x3db5e11b59984046,
        0xbd51c2262e85ed55,
        0xbd3a58213539aed8,
        0x3cd4f80a679b93ee,
        0x3cb83fa375444bd0,
    ],
    [
        0x3fc0230ba90f2871,
        0xbb31a4207958473d,
        0xbfb0230ba90f2871,
        0x3f413164a0864cde,
        0x3f7579c1bdbcfc99,
        0xbf0b67e1913c668f,
        0xbf26d26de4fd8c43,
        0x3ebf1b520b063cbf,
        0x3ec9f01e7c18d94a,
        0xbe624071b17c353c,
        0xbe624f8e936aca3d,
        0x3dfa356639a676b4,
        0x3df19911d0412ad5,
        0xbd89677525aa079f,
        0xbd787e875d392b57,
        0x3d11be245f9c21d5,
        0x3cf98faabb17d8d4,
        0xbc93119ed524d1f5,
    ],
    [
        0x3af57b2a92bacca4,
        0xbfbfa8b3f9ae4375,
        0x3f5856073b7fa2cd,
        0x3f9514e652eb2e96,
        0xbf3032298718ea1a,
        0xbf50d1153fde431b,
        0x3ee9cb455c29d651,
        0x3ef982167b1c52b5,
        0xbe938191ef689b27,
        0xbe968865345b1bd6,
        0x3e31296000bb210d,
        0x3e2a0403677632fc,
        0xbdc3b67c5c27ec16,
        0xbdb526035aeb726c,
        0x3d4fd977f75a677c,
        0x3d398199969ac0ac,
        0xbcd2db2724a8c540,
        0xbcb7838b8a136198,
    ],
    [
        0xbfbf13fb0c0e6fcd,
        0x3b2fb8a657120d32,
        0x3faf13fb0c0e6fcd,
        0xbf3eb3e6fcc47c00,
        0xbf74af74cbd77bef,
        0x3f087bb1ebeaec78,
        0x3f25fe62920314f5,
        0xbebbd0f2a6556264,
        0xbec9040de83035a2,
        0x3e6057f7a76c1f7c,
        0x3e61ad6dd4dfaab9,
        0xbdf782a6f91b13a7,
        0xbdf1027d851ec0eb,
        0x3d86d65e028f8e8e,
        0x3d77b54a61df8d64,
        0xbd0ffa6793d8746b,
        0xbcf8c72862ef4d7d,
        0x3c913a3849824cc1,
    ],
    [
        0xbaf37825e4b1711a,
        0x3fbe8727c572a2c2,
        0xbf55d1ef092ab395,
        0xbf9454e7a7395636,
        0x3f2d0cc3a7fa6d3d,
        0x3f50398d2cbd02df,
        0xbee725b0909f9c17,
        0xbef8a022f5fee439,
        0x3e9185b1ea975d21,
        0x3e95c5711673ea6a,
        0xbe2edff650b86504,
        0xbe29298da3993db9,
        0x3dc1c346aaab062e,
        0x3db47ab8b9f6e94a,
        0xbd4cc1c87cd4af9d,
        0xbd38bb86b43ec34d,
        0x3cd1104df3fa9e99,
        0x3cb6d47888429c17,
    ],
    [
        0x3fbe018dac1c17e3,
        0xbb2cbf68b5ed7271,
        0xbfae018dac1c17e3,
        0x3f3ba21bd15d02a4,
        0x3f73f9e0db07e7ef,
        0xbf060b77c5e27625,
        0xbf253f9b1a5d2273,
        0x3eb910b38812c5ec,
        0x3ec82ee6dfdfc034,
        0xbe5d7cc2a9aa9fae,
        0xbe611a57d834f490,
        0x3df53d401868b2b6,
        0x3df07915b5d7b1b9,
        0xbd84aabf5e11aa97,
        0xbd76fc66f73ce419,
        0x3d0cff2a096a9861,
        0x3cf80d87c580ea09,
        0xbc8f4ef03ad4ec66,
    ],
    [
        0x3af1bbdf10c72747,
        0xbfbd82939ab62339,
        0x3f53b5a54845670f,
        0x3f93a7ff1622def8,
        0xbf2a3ebc476a606d,
        0xbf4f61adde3a8c61,
        0x3ee4ec45da047869,
        0x3ef7d3b28159f22c,
        0xbe8fb3f21bccfc28,
        0xbe95145a876d09d8,
        0x3e2bf64cdef37a7c,
        0x3e2862204a96b8ac,
        0xbdc01c0cf54f6b8d,
        0xbdb3dd6c7e6818d8,
        0x3d4a1faa7e189d99,
        0x3d3804640c4d999d,
        0xbccf0f5eda690a07,
        0xbcb63189b3dc27d0,
    ],
    [
        0xbfbd09b21e36c0bd,
        0x3b2a335791a61ca6,
        0x3fad09b21e36c0bd,
        0xbf390b0e6a475e45,
        0xbf7355b904fbf7ee,
        0x3f03fc459d1e25a7,
        0x3f2492cc61d19de6,
        0xbeb6bcf110a02e19,
        0xbec76d44f6a8089e,
        0x3e5ac61efcb7f75f,
        0x3e609436fc7c1966,
        0xbdf34eb60c63b887,
        0xbdeff662240b32ed,
        0x3d82d05e647447d5,
        0x3d76520ffbea05d6,
        0xbd0a70b3d6b77e3f,
        0xbcf7617161d2567e,
        0x3c8c991bfcf38c32,
    ],
    [
        0xbaf03b145709e9fb,
        0x3fbc96700031f601,
        0xbf51eb2a07d0f09e,
        0xbf930b36eddaa234,
        0x3f27dd2dde84b73c,
        0x3f4e696553e0b8a5,
        0xbee3085aa35a1606,
        0xbef7199f24bccad0,
        0x3e8cdbab6a75e8e1,
        0x3e9472a7cffbdf8d,
        0xbe2979f5209f2f88,
        0xbe27ab4b017a68c3,
        0x3dbd638fb5c45254,
        0x3db34c83c71aa648,
        0xbd47dc91892a3240,
        0xbd375abc2b119f77,
        0x3ccc6a16f48d942f,
        0x3cb599c2db65d50c,
    ],
    [
        0x3fbc2861347b1b39,
        0xbb27f93aa95d7811,
        0xbfac2861347b1b39,
        0x3f36d57bffb37fd5,
        0x3f72c060ef553f18,
        0xbf023a407b722d31,
        0xbf23f55581ec6826,
        0x3eb4bfb4daa30262,
        0x3ec6bc69099ac792,
        0xbe587334098baabf,
        0xbe601951f000356a,
        0x3df1a61752a7a337,
        0x3def0ec9ee45cf6f,
        0xbd8137cd564b9118,
        0xbd75b4aa3b6c601b,
        0x3d083b040a6d9f05,
        0x3cf6c19bd64bc967,
        0xbc8a3eded96f75bf,
    ],
    [
        0x3aedd70a3afed493,
        0xbfbbbf246019c0d4,
        0x3f506224199140d8,
        0x3f927c3416c09898,
        0xbf25d279dc87cf68,
        0xbf4d86a5f5adbdac,
        0x3ee169041634c20c,
        0x3ef66f5bd1bedad0,
        0xbe8a69b2c7176656,
        0xbe93de4eab455284,
        0x3e2755eabb6b4563,
        0x3e2702fe7b7aa32d,
        0xbdbaf1fff2894365,
        0xbdb2c697124bdb4b,
        0x3d45e717125b6112,
        0x3d36bd366d374341,
        0xbcca1d9ce8759323,
        0xbcb50c32fba6f689,
    ],
    [
        0xbfbb5a622198a72c,
        0x3b261140492792be,
        0x3fab5a622198a72c,
        0xbf34ee71cf67c243,
        0xbf7237c02b462f6f,
        0x3f00b67b6fdfec64,
        0x3f2365167d8bc319,
        0xbeb3086b7a29712d,
        0xbec61a141425984e,
        0x3e56712b239ca861,
        0x3e5f507367e2287b,
        0xbdf0366d05430f3b,
        0xbdee38f952d755dd,
        0x3d7faa412f355ab7,
        0x3d7522cbd2af359c,
        0xbd064e7a6cf56505,
        0xbcf62cd7c73270a1,
        0x3c883112ff8933cd,
    ],
    [
        0xbaeb8a30d224cee7,
        0x3fbaf9cb42cd08a7,
        0xbf4e1c66d7616e37,
        0xbf91f90fd1013589,
        0x3f240e3eb09b7d6d,
        0x3f4cb682ff471274,
        0xbee0016819a61da0,
        0xbef5d2d1c420645b,
        0x3e884b0fe8a7ecd7,
        0x3e93559d96bedaf4,
        0xbe257a0be6df375d,
        0xbe26677c8f23f9e7,
        0x3db8d1ada000f3d8,
        0x3db24a6de676e9f8,
        0xbd4431a53c25281e,
        0xbd362a9b74747797,
        0x3cc81ad277664852,
        0x3cb487f213f494c1,
    ],
    [
        0x3fba9d183bc04545,
        0xbb24612b590e8e50,
        0xbfaa9d183bc04545,
        0x3f334779874010dc,
        0x3f71ba2299ab88a8,
        0xbefecb19f5bdc64d,
        0xbf22e052707cc843,
        0x3eb18a5da77ee010,
        0x3ec5846b622f2fd0,
        0xbe54b17e97b8ccc9,
        0xbe5e7f7666f84ced,
        0x3dedeb50538393d5,
        0x3ded72e1f6ccd5a2,
        0xbd7d3e0b2d7b598f,
        0xbd749b384c86d390,
        0x3d049ebe7a682354,
        0x3cf5a2123b57dec6,
        0xbc8662ce26072b62,
    ],
    [
        0x3ae97c92b0497970,
        0xbfba4407dac72297,
        0x3f4bcba4dec1da44,
        0x3f91803c65cafdfb,
        0xbf2283df2b3e4a6d,
        0xbf4bf695e89259cd,
        0x3edd8f3759122cea,
        0x3ef54246c8e04206,
        0xbe867111bc8e08a2,
        0xbe92d72a1f62d131,
        0x3e23d9ca3e4f40f9,
        0x3e25d74a6df26826,
        0xbdb6f4b3a86a00b8,
        0xbdb1d6f951aed310,
        0x3d42b1828de6ffa0,
        0x3d35a1d54468858d,
        0xbcc655ad620318bd,
        0xbcb40c27591e7a1a,
    ],
    [
        0xbfb9ee5eee1a97c6,
        0x3b22e5c966133b1c,
        0x3fa9ee5eee1a97c6,
        0xbf31d55d93e59bd7,
        0xbf7146219394a99c,
        0x3efc7d32bb646e2e,
        0x3f226599aed4f1b4,
        0xbeb03ba58ec85e6f,
        0xbec4f9e3961b9170,
        0x3e5328d4dc364807,
        0x3e5dbda7ac1d3f8b,
        0xbdebb7c2f5dfce50,
        0xbdecbac1e6849bc9,
        0x3d7b1c39328db702,
        0x3d741cdba002ff38,
        0xbd0321f9e5a1aec7,
        0xbcf520545c68186a,
        0x3c84ca30f2d8fe34,
    ],
    [
        0xbae7aaec85700fa6,
        0x3fb99be73fa3efcc,
        0xbf49c3248da75775,
        0xbf91107147eda800,
        0x3f212980be9d8520,
        0x3f4b44e1221e6051,
        0xbedb67b78c1dd966,
        0xbef4bc49e9b5bceb,
        0x3e84d02ffb8b7772,
        0x3e9261c31456c932,
        0xbe226b4006e91d9d,
        0xbe25512519fb5144,
        0x3db5500f2545f2d3,
        0x3db16b4e5f84ad48,
        0xbd415e211f9e7a0f,
        0xbd3521ed4cbd2b27,
        0x3cc4c4800c50a30c,
        0x3cb39810921b4be9,
    ],
    [
        0x3fb94c6f5898708b,
        0xbb219499b2c197ca,
        0xbfa94c6f5898708b,
        0x3f308f4f1b8fb0dc,
        0x3f70da931c776c71,
        0xbefa7550d2940f2e,
        0xbf21f3b978a47cde,
        0x3eae28ea0247325a,
        0x3ec47930aec7652a,
        0xbe51ce32c4f61fb7,
        0xbe5d0950e94a9be9,
        0x3de9c5a164496f38,
        0x3dec0f17151563cb,
        0xbd79390150c5d3eb,
        0xbd73a6c5157c0de8,
        0x3d01d041a270134c,
        0x3cf4a6c2b9e9b4e9,
        0xbc835f05ebdf3e12,
    ],
    [
        0x3ae60bdc6a36e0e5,
        0xbfb8ffc9b9a131f6,
        0x3f47f724314bb99d,
        0x3f90a89c5d1074ba,
        0xbf1feeaeb33465e7,
        0xbf4a9fba36a7a08e,
        0x3ed980034770f358,
        0x3ef43fa4bb8cba45,
        0xbe835f40f3a33db9,
        0xbe91f465ba2910f8,
        0x3e212688f6f9e1b4,
        0x3e24d3f7e3ce0c1a,
        0xbdb3daee33c4651d,
        0xbdb106a0ff5e7529,
        0x3d40309d2f92ac42,
        0x3d34aa09772c258a,
        0xbcc35f72aaef208c,
        0xbcb32afd4e360d62,
    ],
    [
        0xbfb8b5ccb03d459b,
        0x3b206ac337f71f68,
        0x3fa8b5ccb03d459b,
        0xbf2edc98b92fedf3,
        0xbf70767d524cea88,
        0x3ef8a85a847c89c6,
        0x3f2189afaa2bffb7,
        0xbeac1d231a69bf1e,
        0xbec40139bdbc767b,
        0x3e509a6859cb8509,
        0x3e5c60fc5f4b173c,
        0xbde80ad395bd719c,
        0xbdeb6e94ec333a76,
        0x3d778adf7d6d4ffb,
        0x3d7338227f2ea5d9,
        0xbd00a328a2899a4a,
        0xbcf4349a4535eb0d,
        0x3c821ac1fe027675,
    ],
    [
        0xbae4990a795d52df,
        0x3fb86e51bb2ee24d,
        0xbf465e3bad214eb8,
        0xbf9047d6ed159c31,
        0x3f1dce4a381ce24d,
        0x3f4a05b93842da9a,
        0xbed7ce3df60b0e4e,
        0xbef3cb50138cce39,
        0x3e8216e84a5d9f52,
        0x3e918e354c013832,
        0xbe2005492fe9c63e,
        0xbe245ed4a343a749,
        0x3db28e2e71c56d3c,
        0x3db0a83f8589bb9d,
        0xbd3e46baa33f452c,
        0xbd3439690ccae646,
        0x3cc2201c8a30e9e6,
        0x3cb2c44f88828341,
    ],
    [
        0x3fb829356c2fb67c,
        0xbb1ebe6fec859455,
        0xbfa829356c2fb67c,
        0x3f2cd964e5e4caa0,
        0x3f70190ce66b97c3,
        0xbef70d5fd8bad7b8,
        0xbf2126a150469af1,
        0x3eaa49ea858153dd,
        0x3ec3910f5c157962,
        0xbe4f0f4bdbe6f9e5,
        0xbe5bc36977db17fb,
        0x3de67f295d55fdea,
        0x3dead81ba8b0c52f,
        0xbd760a13803ef42a,
        0xbd72d03bf820c54e,
        0x3cff2ad7c1434050,
        0x3cf3c92e07a03306,
        0xbc80f7d055471199,
    ],
    [
        0x3ae34bfa6ed120b4,
        0xbfb7e656ed57a0d1,
        0x3f44f0cfc62eb61c,
        0x3f8fdaba63e9c655,
        0xbf1be7bad1c50546,
        0xbf4975ac0770c98d,
        0x3ed64a6af83ba812,
        0x3ef35e6b41b6d769,
        0xbe80f12d050da9bf,
        0xbe912e744222d16c,
        0x3e1e04a8c0124324,
        0x3e23f0ed5e3ac1ab,
        0xbdb163fd1599c398,
        0xbdb04f8ecba977f7,
        0x3d3c639575076eb9,
        0x3d33cf61a84aa254,
        0xbcc101368b088743,
        0xbcb26377b4c35c24,
    ],
    [
        0xbfb7a597eb76a5e3,
        0x3b53aefe167f1294,
        0x3fa7a597eb76a5e3,
        0xbf2b0bd9eb615315,
        0xbf6f831b9629acd2,
        0x3ef59d1c40c95bc4,
        0x3f20c9d35d4be179,
        0xbea8a711bec6b4b0,
        0xbec327e42f38db9e,
        0x3e4d225aa909ad2f,
        0x3e5b2f83a8f0f85c,
        0xbde51bef24cad6b5,
        0xbdea4ab13337518e,
        0x3d74b0547a699854,
        0x3d726e6cf385d626,
        0xbcfd56db39ddf26a,
        0xbcf362be01af446b,
        0x3c82edb1a6e2242c,
    ],
];
