// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package users

import (
	"context"
	"net/http"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/render"
	"github.com/drone/drone/logger"

	"github.com/go-chi/chi"
)

// HandleDelete returns an http.HandlerFunc that processes an http.Request
// to delete the named user account from the system.
func HandleDelete(
	users core.UserStore,
	transferer core.Transferer,
	sender core.WebhookSender,
) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		login := chi.URLParam(r, "user")
		user, err := users.FindLogin(r.Context(), login)
		if err != nil {
			render.NotFound(w, err)
			logger.FromRequest(r).WithError(err).
				Debugln("api: cannot find user")
			return
		}

		err = transferer.Transfer(context.Background(), user)
		if err != nil {
			logger.FromRequest(r).WithError(err).
				Warnln("api: cannot transfer repository ownership")
		}

		err = users.Delete(r.Context(), user)
		if err != nil {
			render.InternalError(w, err)
			logger.FromRequest(r).WithError(err).
				Warnln("api: cannot delete user")
			return
		}

		err = sender.Send(r.Context(), &core.WebhookData{
			Event:  core.WebhookEventUser,
			Action: core.WebhookActionDeleted,
			User:   user,
		})
		if err != nil {
			logger.FromRequest(r).WithError(err).
				Warnln("api: cannot send webhook")
		}

		w.WriteHeader(http.StatusNoContent)
	}
}
