/*************************************************************************
 *
 * Copyright (C) 2018-2025 Ruilin Peng (Nick) <pymumu@gmail.com>.
 *
 * smartdns is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * smartdns is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "qpack.h"

static const uint8_t _qpack_huffman_bit[64] = {
	249, 50,  115, 39,  38,  79, 147, 39, 38,  100, 249, 50,  114, 100, 242, 100, 228, 228, 206, 77,  52, 228,
	204, 203, 202, 114, 102, 79, 147, 39, 76,  156, 153, 62,  76,  156, 156, 153, 62,  76,  156, 153, 60, 154,
	100, 242, 102, 79,  147, 39, 38,  83, 228, 201, 201, 147, 211, 39,  38,  79,  38,  78,  76,  178};

static const uint8_t _qpack_huffman_val[512] = {
	/* 0: |  X: 44 */ 44,
	/* 1: |0  X: 17 */ 16,
	/* 2: |00  X: 10 */ 8,
	/* 3: |000  X: 7 */ 4,
	/* 4: |0000  X: 6 */ 2,
	/* 5: |00000  V: 48 */ 48,
	/* 6: |00001  V: 49 */ 49,
	/* 7: |0001  X: 9 */ 2,
	/* 8: |00010  V: 50 */ 50,
	/* 9: |00011  V: 97 */ 97,
	/* 10: |001  X: 14 */ 4,
	/* 11: |0010  X: 13 */ 2,
	/* 12: |00100  V: 99 */ 99,
	/* 13: |00101  V: 101 */ 101,
	/* 14: |0011  X: 16 */ 2,
	/* 15: |00110  V: 105 */ 105,
	/* 16: |00111  V: 111 */ 111,
	/* 17: |01  X: 29 */ 12,
	/* 18: |010  X: 22 */ 4,
	/* 19: |0100  X: 21 */ 2,
	/* 20: |01000  V: 115 */ 115,
	/* 21: |01001  V: 116 */ 116,
	/* 22: |0101  X: 26 */ 4,
	/* 23: |01010  X: 25 */ 2,
	/* 24: |010100  V: 32 */ 32,
	/* 25: |010101  V: 37 */ 37,
	/* 26: |01011  X: 28 */ 2,
	/* 27: |010110  V: 45 */ 45,
	/* 28: |010111  V: 46 */ 46,
	/* 29: |011  X: 37 */ 8,
	/* 30: |0110  X: 34 */ 4,
	/* 31: |01100  X: 33 */ 2,
	/* 32: |011000  V: 47 */ 47,
	/* 33: |011001  V: 51 */ 51,
	/* 34: |01101  X: 36 */ 2,
	/* 35: |011010  V: 52 */ 52,
	/* 36: |011011  V: 53 */ 53,
	/* 37: |0111  X: 41 */ 4,
	/* 38: |01110  X: 40 */ 2,
	/* 39: |011100  V: 54 */ 54,
	/* 40: |011101  V: 55 */ 55,
	/* 41: |01111  X: 43 */ 2,
	/* 42: |011110  V: 56 */ 56,
	/* 43: |011111  V: 57 */ 57,
	/* 44: |1  X: 80 */ 36,
	/* 45: |10  X: 61 */ 16,
	/* 46: |100  X: 54 */ 8,
	/* 47: |1000  X: 51 */ 4,
	/* 48: |10000  X: 50 */ 2,
	/* 49: |100000  V: 61 */ 61,
	/* 50: |100001  V: 65 */ 65,
	/* 51: |10001  X: 53 */ 2,
	/* 52: |100010  V: 95 */ 95,
	/* 53: |100011  V: 98 */ 98,
	/* 54: |1001  X: 58 */ 4,
	/* 55: |10010  X: 57 */ 2,
	/* 56: |100100  V: 100 */ 100,
	/* 57: |100101  V: 102 */ 102,
	/* 58: |10011  X: 60 */ 2,
	/* 59: |100110  V: 103 */ 103,
	/* 60: |100111  V: 104 */ 104,
	/* 61: |101  X: 69 */ 8,
	/* 62: |1010  X: 66 */ 4,
	/* 63: |10100  X: 65 */ 2,
	/* 64: |101000  V: 108 */ 108,
	/* 65: |101001  V: 109 */ 109,
	/* 66: |10101  X: 68 */ 2,
	/* 67: |101010  V: 110 */ 110,
	/* 68: |101011  V: 112 */ 112,
	/* 69: |1011  X: 73 */ 4,
	/* 70: |10110  X: 72 */ 2,
	/* 71: |101100  V: 114 */ 114,
	/* 72: |101101  V: 117 */ 117,
	/* 73: |10111  X: 77 */ 4,
	/* 74: |101110  X: 76 */ 2,
	/* 75: |1011100  V: 58 */ 58,
	/* 76: |1011101  V: 66 */ 66,
	/* 77: |101111  X: 79 */ 2,
	/* 78: |1011110  V: 67 */ 67,
	/* 79: |1011111  V: 68 */ 68,
	/* 80: |11  X: 112 */ 32,
	/* 81: |110  X: 97 */ 16,
	/* 82: |1100  X: 90 */ 8,
	/* 83: |11000  X: 87 */ 4,
	/* 84: |110000  X: 86 */ 2,
	/* 85: |1100000  V: 69 */ 69,
	/* 86: |1100001  V: 70 */ 70,
	/* 87: |110001  X: 89 */ 2,
	/* 88: |1100010  V: 71 */ 71,
	/* 89: |1100011  V: 72 */ 72,
	/* 90: |11001  X: 94 */ 4,
	/* 91: |110010  X: 93 */ 2,
	/* 92: |1100100  V: 73 */ 73,
	/* 93: |1100101  V: 74 */ 74,
	/* 94: |110011  X: 96 */ 2,
	/* 95: |1100110  V: 75 */ 75,
	/* 96: |1100111  V: 76 */ 76,
	/* 97: |1101  X: 105 */ 8,
	/* 98: |11010  X: 102 */ 4,
	/* 99: |110100  X: 101 */ 2,
	/* 100: |1101000  V: 77 */ 77,
	/* 101: |1101001  V: 78 */ 78,
	/* 102: |110101  X: 104 */ 2,
	/* 103: |1101010  V: 79 */ 79,
	/* 104: |1101011  V: 80 */ 80,
	/* 105: |11011  X: 109 */ 4,
	/* 106: |110110  X: 108 */ 2,
	/* 107: |1101100  V: 81 */ 81,
	/* 108: |1101101  V: 82 */ 82,
	/* 109: |110111  X: 111 */ 2,
	/* 110: |1101110  V: 83 */ 83,
	/* 111: |1101111  V: 84 */ 84,
	/* 112: |111  X: 128 */ 16,
	/* 113: |1110  X: 121 */ 8,
	/* 114: |11100  X: 118 */ 4,
	/* 115: |111000  X: 117 */ 2,
	/* 116: |1110000  V: 85 */ 85,
	/* 117: |1110001  V: 86 */ 86,
	/* 118: |111001  X: 120 */ 2,
	/* 119: |1110010  V: 87 */ 87,
	/* 120: |1110011  V: 89 */ 89,
	/* 121: |11101  X: 125 */ 4,
	/* 122: |111010  X: 124 */ 2,
	/* 123: |1110100  V: 106 */ 106,
	/* 124: |1110101  V: 107 */ 107,
	/* 125: |111011  X: 127 */ 2,
	/* 126: |1110110  V: 113 */ 113,
	/* 127: |1110111  V: 118 */ 118,
	/* 128: |1111  X: 136 */ 8,
	/* 129: |11110  X: 133 */ 4,
	/* 130: |111100  X: 132 */ 2,
	/* 131: |1111000  V: 119 */ 119,
	/* 132: |1111001  V: 120 */ 120,
	/* 133: |111101  X: 135 */ 2,
	/* 134: |1111010  V: 121 */ 121,
	/* 135: |1111011  V: 122 */ 122,
	/* 136: |11111  X: 144 */ 8,
	/* 137: |111110  X: 141 */ 4,
	/* 138: |1111100  X: 140 */ 2,
	/* 139: |11111000  V: 38 */ 38,
	/* 140: |11111001  V: 42 */ 42,
	/* 141: |1111101  X: 143 */ 2,
	/* 142: |11111010  V: 44 */ 44,
	/* 143: |11111011  V: 59 */ 59,
	/* 144: |111111  X: 148 */ 4,
	/* 145: |1111110  X: 147 */ 2,
	/* 146: |11111100  V: 88 */ 88,
	/* 147: |11111101  V: 90 */ 90,
	/* 148: |1111111  X: 156 */ 8,
	/* 149: |11111110  X: 153 */ 4,
	/* 150: |11111110|0  X: 152 */ 2,
	/* 151: |11111110|00  V: 33 */ 33,
	/* 152: |11111110|01  V: 34 */ 34,
	/* 153: |11111110|1  X: 155 */ 2,
	/* 154: |11111110|10  V: 40 */ 40,
	/* 155: |11111110|11  V: 41 */ 41,
	/* 156: |11111111|  X: 162 */ 6,
	/* 157: |11111111|0  X: 159 */ 2,
	/* 158: |11111111|00  V: 63 */ 63,
	/* 159: |11111111|01  X: 161 */ 2,
	/* 160: |11111111|010  V: 39 */ 39,
	/* 161: |11111111|011  V: 43 */ 43,
	/* 162: |11111111|1  X: 168 */ 6,
	/* 163: |11111111|10  X: 165 */ 2,
	/* 164: |11111111|100  V: 124 */ 124,
	/* 165: |11111111|101  X: 167 */ 2,
	/* 166: |11111111|1010  V: 35 */ 35,
	/* 167: |11111111|1011  V: 62 */ 62,
	/* 168: |11111111|11  X: 176 */ 8,
	/* 169: |11111111|110  X: 173 */ 4,
	/* 170: |11111111|1100  X: 172 */ 2,
	/* 171: |11111111|11000  V: 0 */ 0,
	/* 172: |11111111|11001  V: 36 */ 36,
	/* 173: |11111111|1101  X: 175 */ 2,
	/* 174: |11111111|11010  V: 64 */ 64,
	/* 175: |11111111|11011  V: 91 */ 91,
	/* 176: |11111111|111  X: 180 */ 4,
	/* 177: |11111111|1110  X: 179 */ 2,
	/* 178: |11111111|11100  V: 93 */ 93,
	/* 179: |11111111|11101  V: 126 */ 126,
	/* 180: |11111111|1111  X: 184 */ 4,
	/* 181: |11111111|11110  X: 183 */ 2,
	/* 182: |11111111|111100  V: 94 */ 94,
	/* 183: |11111111|111101  V: 125 */ 125,
	/* 184: |11111111|11111  X: 188 */ 4,
	/* 185: |11111111|111110  X: 187 */ 2,
	/* 186: |11111111|1111100  V: 60 */ 60,
	/* 187: |11111111|1111101  V: 96 */ 96,
	/* 188: |11111111|111111  X: 190 */ 2,
	/* 189: |11111111|1111110  V: 123 */ 123,
	/* 190: |11111111|1111111  X: 220 */ 30,
	/* 191: |11111111|11111110|  X: 201 */ 10,
	/* 192: |11111111|11111110|0  X: 196 */ 4,
	/* 193: |11111111|11111110|00  X: 195 */ 2,
	/* 194: |11111111|11111110|000  V: 92 */ 92,
	/* 195: |11111111|11111110|001  V: 195 */ 195,
	/* 196: |11111111|11111110|01  X: 198 */ 2,
	/* 197: |11111111|11111110|010  V: 208 */ 208,
	/* 198: |11111111|11111110|011  X: 200 */ 2,
	/* 199: |11111111|11111110|0110  V: 128 */ 128,
	/* 200: |11111111|11111110|0111  V: 130 */ 130,
	/* 201: |11111111|11111110|1  X: 209 */ 8,
	/* 202: |11111111|11111110|10  X: 206 */ 4,
	/* 203: |11111111|11111110|100  X: 205 */ 2,
	/* 204: |11111111|11111110|1000  V: 131 */ 131,
	/* 205: |11111111|11111110|1001  V: 162 */ 162,
	/* 206: |11111111|11111110|101  X: 208 */ 2,
	/* 207: |11111111|11111110|1010  V: 184 */ 184,
	/* 208: |11111111|11111110|1011  V: 194 */ 194,
	/* 209: |11111111|11111110|11  X: 213 */ 4,
	/* 210: |11111111|11111110|110  X: 212 */ 2,
	/* 211: |11111111|11111110|1100  V: 224 */ 224,
	/* 212: |11111111|11111110|1101  V: 226 */ 226,
	/* 213: |11111111|11111110|111  X: 217 */ 4,
	/* 214: |11111111|11111110|1110  X: 216 */ 2,
	/* 215: |11111111|11111110|11100  V: 153 */ 153,
	/* 216: |11111111|11111110|11101  V: 161 */ 161,
	/* 217: |11111111|11111110|1111  X: 219 */ 2,
	/* 218: |11111111|11111110|11110  V: 167 */ 167,
	/* 219: |11111111|11111110|11111  V: 172 */ 172,
	/* 220: |11111111|11111111|  X: 266 */ 46,
	/* 221: |11111111|11111111|0  X: 237 */ 16,
	/* 222: |11111111|11111111|00  X: 230 */ 8,
	/* 223: |11111111|11111111|000  X: 227 */ 4,
	/* 224: |11111111|11111111|0000  X: 226 */ 2,
	/* 225: |11111111|11111111|00000  V: 176 */ 176,
	/* 226: |11111111|11111111|00001  V: 177 */ 177,
	/* 227: |11111111|11111111|0001  X: 229 */ 2,
	/* 228: |11111111|11111111|00010  V: 179 */ 179,
	/* 229: |11111111|11111111|00011  V: 209 */ 209,
	/* 230: |11111111|11111111|001  X: 234 */ 4,
	/* 231: |11111111|11111111|0010  X: 233 */ 2,
	/* 232: |11111111|11111111|00100  V: 216 */ 216,
	/* 233: |11111111|11111111|00101  V: 217 */ 217,
	/* 234: |11111111|11111111|0011  X: 236 */ 2,
	/* 235: |11111111|11111111|00110  V: 227 */ 227,
	/* 236: |11111111|11111111|00111  V: 229 */ 229,
	/* 237: |11111111|11111111|01  X: 251 */ 14,
	/* 238: |11111111|11111111|010  X: 244 */ 6,
	/* 239: |11111111|11111111|0100  X: 241 */ 2,
	/* 240: |11111111|11111111|01000  V: 230 */ 230,
	/* 241: |11111111|11111111|01001  X: 243 */ 2,
	/* 242: |11111111|11111111|010010  V: 129 */ 129,
	/* 243: |11111111|11111111|010011  V: 132 */ 132,
	/* 244: |11111111|11111111|0101  X: 248 */ 4,
	/* 245: |11111111|11111111|01010  X: 247 */ 2,
	/* 246: |11111111|11111111|010100  V: 133 */ 133,
	/* 247: |11111111|11111111|010101  V: 134 */ 134,
	/* 248: |11111111|11111111|01011  X: 250 */ 2,
	/* 249: |11111111|11111111|010110  V: 136 */ 136,
	/* 250: |11111111|11111111|010111  V: 146 */ 146,
	/* 251: |11111111|11111111|011  X: 259 */ 8,
	/* 252: |11111111|11111111|0110  X: 256 */ 4,
	/* 253: |11111111|11111111|01100  X: 255 */ 2,
	/* 254: |11111111|11111111|011000  V: 154 */ 154,
	/* 255: |11111111|11111111|011001  V: 156 */ 156,
	/* 256: |11111111|11111111|01101  X: 257 */ 2,
	/* 257: |11111111|11111111|011010  V: 160 */ 160,
	/* 258: |11111111|11111111|011011  V: 163 */ 163,
	/* 259: |11111111|11111111|0111  X: 263 */ 4,
	/* 260: |11111111|11111111|01110  X: 262 */ 2,
	/* 261: |11111111|11111111|011100  V: 164 */ 164,
	/* 262: |11111111|11111111|011101  V: 169 */ 169,
	/* 263: |11111111|11111111|01111  X: 265 */ 2,
	/* 264: |11111111|11111111|011110  V: 170 */ 170,
	/* 265: |11111111|11111111|011111  V: 173 */ 173,
	/* 266: |11111111|11111111|1  X: 306 */ 40,
	/* 267: |11111111|11111111|10  X: 283 */ 16,
	/* 268: |11111111|11111111|100  X: 276 */ 8,
	/* 269: |11111111|11111111|1000  X: 273 */ 4,
	/* 270: |11111111|11111111|10000  X: 272 */ 2,
	/* 271: |11111111|11111111|100000  V: 178 */ 178,
	/* 272: |11111111|11111111|100001  V: 181 */ 181,
	/* 273: |11111111|11111111|10001  X: 275 */ 2,
	/* 274: |11111111|11111111|100010  V: 185 */ 185,
	/* 275: |11111111|11111111|100011  V: 186 */ 186,
	/* 276: |11111111|11111111|1001  X: 280 */ 4,
	/* 277: |11111111|11111111|10010  X: 279 */ 2,
	/* 278: |11111111|11111111|100100  V: 187 */ 187,
	/* 279: |11111111|11111111|100101  V: 189 */ 189,
	/* 280: |11111111|11111111|10011  X: 282 */ 2,
	/* 281: |11111111|11111111|100110  V: 190 */ 190,
	/* 282: |11111111|11111111|100111  V: 196 */ 196,
	/* 283: |11111111|11111111|101  X: 291 */ 8,
	/* 284: |11111111|11111111|1010  X: 288 */ 4,
	/* 285: |11111111|11111111|10100  X: 287 */ 2,
	/* 286: |11111111|11111111|101000  V: 198 */ 198,
	/* 287: |11111111|11111111|101001  V: 228 */ 228,
	/* 288: |11111111|11111111|10101  X: 290 */ 2,
	/* 289: |11111111|11111111|101010  V: 232 */ 232,
	/* 290: |11111111|11111111|101011  V: 233 */ 233,
	/* 291: |11111111|11111111|1011  X: 299 */ 8,
	/* 292: |11111111|11111111|10110  X: 296 */ 4,
	/* 293: |11111111|11111111|101100  X: 295 */ 2,
	/* 294: |11111111|11111111|1011000  V: 1 */ 1,
	/* 295: |11111111|11111111|1011001  V: 135 */ 135,
	/* 296: |11111111|11111111|101101  X: 298 */ 2,
	/* 297: |11111111|11111111|1011010  V: 137 */ 137,
	/* 298: |11111111|11111111|1011011  V: 138 */ 138,
	/* 299: |11111111|11111111|10111  X: 303 */ 4,
	/* 300: |11111111|11111111|101110  X: 302 */ 2,
	/* 301: |11111111|11111111|1011100  V: 139 */ 139,
	/* 302: |11111111|11111111|1011101  V: 140 */ 140,
	/* 303: |11111111|11111111|101111  X: 305 */ 2,
	/* 304: |11111111|11111111|1011110  V: 141 */ 141,
	/* 305: |11111111|11111111|1011111  V: 143 */ 143,
	/* 306: |11111111|11111111|11  X: 338 */ 32,
	/* 307: |11111111|11111111|110  X: 323 */ 16,
	/* 308: |11111111|11111111|1100  X: 316 */ 8,
	/* 309: |11111111|11111111|11000  X: 313 */ 4,
	/* 310: |11111111|11111111|110000  X: 312 */ 2,
	/* 311: |11111111|11111111|1100000  V: 147 */ 147,
	/* 312: |11111111|11111111|1100001  V: 149 */ 149,
	/* 313: |11111111|11111111|110001  X: 315 */ 2,
	/* 314: |11111111|11111111|1100010  V: 150 */ 150,
	/* 315: |11111111|11111111|1100011  V: 151 */ 151,
	/* 316: |11111111|11111111|11001  X: 320 */ 4,
	/* 317: |11111111|11111111|110010  X: 319 */ 2,
	/* 318: |11111111|11111111|1100100  V: 152 */ 152,
	/* 319: |11111111|11111111|1100101  V: 155 */ 155,
	/* 320: |11111111|11111111|110011  X: 322 */ 2,
	/* 321: |11111111|11111111|1100110  V: 157 */ 157,
	/* 322: |11111111|11111111|1100111  V: 158 */ 158,
	/* 323: |11111111|11111111|1101  X: 331 */ 8,
	/* 324: |11111111|11111111|11010  X: 328 */ 4,
	/* 325: |11111111|11111111|110100  X: 327 */ 2,
	/* 326: |11111111|11111111|1101000  V: 165 */ 165,
	/* 327: |11111111|11111111|1101001  V: 166 */ 166,
	/* 328: |11111111|11111111|110101  X: 330 */ 2,
	/* 329: |11111111|11111111|1101010  V: 168 */ 168,
	/* 330: |11111111|11111111|1101011  V: 174 */ 174,
	/* 331: |11111111|11111111|11011  X: 335 */ 4,
	/* 332: |11111111|11111111|110110  X: 334 */ 2,
	/* 333: |11111111|11111111|1101100  V: 175 */ 175,
	/* 334: |11111111|11111111|1101101  V: 180 */ 180,
	/* 335: |11111111|11111111|110111  X: 337 */ 2,
	/* 336: |11111111|11111111|1101110  V: 182 */ 182,
	/* 337: |11111111|11111111|1101111  V: 183 */ 183,
	/* 338: |11111111|11111111|111  X: 360 */ 22,
	/* 339: |11111111|11111111|1110  X: 347 */ 8,
	/* 340: |11111111|11111111|11100  X: 344 */ 4,
	/* 341: |11111111|11111111|111000  X: 343 */ 2,
	/* 342: |11111111|11111111|1110000  V: 188 */ 188,
	/* 343: |11111111|11111111|1110001  V: 191 */ 191,
	/* 344: |11111111|11111111|111001  X: 346 */ 2,
	/* 345: |11111111|11111111|1110010  V: 197 */ 197,
	/* 346: |11111111|11111111|1110011  V: 231 */ 231,
	/* 347: |11111111|11111111|11101  X: 353 */ 6,
	/* 348: |11111111|11111111|111010  X: 350 */ 2,
	/* 349: |11111111|11111111|1110100  V: 239 */ 239,
	/* 350: |11111111|11111111|1110101  X: 352 */ 2,
	/* 351: |11111111|11111111|11101010  V: 9 */ 9,
	/* 352: |11111111|11111111|11101011  V: 142 */ 142,
	/* 353: |11111111|11111111|111011  X: 357 */ 4,
	/* 354: |11111111|11111111|1110110  X: 356 */ 2,
	/* 355: |11111111|11111111|11101100  V: 144 */ 144,
	/* 356: |11111111|11111111|11101101  V: 145 */ 145,
	/* 357: |11111111|11111111|1110111  X: 359 */ 2,
	/* 358: |11111111|11111111|11101110  V: 148 */ 148,
	/* 359: |11111111|11111111|11101111  V: 159 */ 159,
	/* 360: |11111111|11111111|1111  X: 380 */ 20,
	/* 361: |11111111|11111111|11110  X: 369 */ 8,
	/* 362: |11111111|11111111|111100  X: 366 */ 4,
	/* 363: |11111111|11111111|1111000  X: 365 */ 2,
	/* 364: |11111111|11111111|11110000  V: 171 */ 171,
	/* 365: |11111111|11111111|11110001  V: 206 */ 206,
	/* 366: |11111111|11111111|1111001  X: 368 */ 2,
	/* 367: |11111111|11111111|11110010  V: 215 */ 215,
	/* 368: |11111111|11111111|11110011  V: 225 */ 225,
	/* 369: |11111111|11111111|111101  X: 373 */ 4,
	/* 370: |11111111|11111111|1111010  X: 372 */ 2,
	/* 371: |11111111|11111111|11110100  V: 236 */ 236,
	/* 372: |11111111|11111111|11110101  V: 237 */ 237,
	/* 373: |11111111|11111111|1111011  X: 377 */ 4,
	/* 374: |11111111|11111111|11110110|  X: 376 */ 2,
	/* 375: |11111111|11111111|11110110|0  V: 199 */ 199,
	/* 376: |11111111|11111111|11110110|1  V: 207 */ 207,
	/* 377: |11111111|11111111|11110111|  X: 379 */ 2,
	/* 378: |11111111|11111111|11110111|0  V: 234 */ 234,
	/* 379: |11111111|11111111|11110111|1  V: 235 */ 235,
	/* 380: |11111111|11111111|11111  X: 414 */ 34,
	/* 381: |11111111|11111111|111110  X: 397 */ 16,
	/* 382: |11111111|11111111|1111100  X: 390 */ 8,
	/* 383: |11111111|11111111|11111000|  X: 387 */ 4,
	/* 384: |11111111|11111111|11111000|0  X: 386 */ 2,
	/* 385: |11111111|11111111|11111000|00  V: 192 */ 192,
	/* 386: |11111111|11111111|11111000|01  V: 193 */ 193,
	/* 387: |11111111|11111111|11111000|1  X: 389 */ 2,
	/* 388: |11111111|11111111|11111000|10  V: 200 */ 200,
	/* 389: |11111111|11111111|11111000|11  V: 201 */ 201,
	/* 390: |11111111|11111111|11111001|  X: 394 */ 4,
	/* 391: |11111111|11111111|11111001|0  X: 393 */ 2,
	/* 392: |11111111|11111111|11111001|00  V: 202 */ 202,
	/* 393: |11111111|11111111|11111001|01  V: 205 */ 205,
	/* 394: |11111111|11111111|11111001|1  X: 396 */ 2,
	/* 395: |11111111|11111111|11111001|10  V: 210 */ 210,
	/* 396: |11111111|11111111|11111001|11  V: 213 */ 213,
	/* 397: |11111111|11111111|1111101  X: 405 */ 8,
	/* 398: |11111111|11111111|11111010|  X: 402 */ 4,
	/* 399: |11111111|11111111|11111010|0  X: 401 */ 2,
	/* 400: |11111111|11111111|11111010|00  V: 218 */ 218,
	/* 401: |11111111|11111111|11111010|01  V: 219 */ 219,
	/* 402: |11111111|11111111|11111010|1  X: 404 */ 2,
	/* 403: |11111111|11111111|11111010|10  V: 238 */ 238,
	/* 404: |11111111|11111111|11111010|11  V: 240 */ 240,
	/* 405: |11111111|11111111|11111011|  X: 409 */ 4,
	/* 406: |11111111|11111111|11111011|0  X: 408 */ 2,
	/* 407: |11111111|11111111|11111011|00  V: 242 */ 242,
	/* 408: |11111111|11111111|11111011|01  V: 243 */ 243,
	/* 409: |11111111|11111111|11111011|1  X: 411 */ 2,
	/* 410: |11111111|11111111|11111011|10  V: 255 */ 255,
	/* 411: |11111111|11111111|11111011|11  X: 413 */ 2,
	/* 412: |11111111|11111111|11111011|110  V: 203 */ 203,
	/* 413: |11111111|11111111|11111011|111  V: 204 */ 204,
	/* 414: |11111111|11111111|111111  X: 446 */ 32,
	/* 415: |11111111|11111111|1111110  X: 431 */ 16,
	/* 416: |11111111|11111111|11111100|  X: 424 */ 8,
	/* 417: |11111111|11111111|11111100|0  X: 421 */ 4,
	/* 418: |11111111|11111111|11111100|00  X: 420 */ 2,
	/* 419: |11111111|11111111|11111100|000  V: 211 */ 211,
	/* 420: |11111111|11111111|11111100|001  V: 212 */ 212,
	/* 421: |11111111|11111111|11111100|01  X: 423 */ 2,
	/* 422: |11111111|11111111|11111100|010  V: 214 */ 214,
	/* 423: |11111111|11111111|11111100|011  V: 221 */ 221,
	/* 424: |11111111|11111111|11111100|1  X: 428 */ 4,
	/* 425: |11111111|11111111|11111100|10  X: 427 */ 2,
	/* 426: |11111111|11111111|11111100|100  V: 222 */ 222,
	/* 427: |11111111|11111111|11111100|101  V: 223 */ 223,
	/* 428: |11111111|11111111|11111100|11  X: 430 */ 2,
	/* 429: |11111111|11111111|11111100|110  V: 241 */ 241,
	/* 430: |11111111|11111111|11111100|111  V: 244 */ 244,
	/* 431: |11111111|11111111|11111101|  X: 439 */ 8,
	/* 432: |11111111|11111111|11111101|0  X: 436 */ 4,
	/* 433: |11111111|11111111|11111101|00  X: 435 */ 2,
	/* 434: |11111111|11111111|11111101|000  V: 245 */ 245,
	/* 435: |11111111|11111111|11111101|001  V: 246 */ 246,
	/* 436: |11111111|11111111|11111101|01  X: 438 */ 2,
	/* 437: |11111111|11111111|11111101|010  V: 247 */ 247,
	/* 438: |11111111|11111111|11111101|011  V: 248 */ 248,
	/* 439: |11111111|11111111|11111101|1  X: 443 */ 4,
	/* 440: |11111111|11111111|11111101|10  X: 442 */ 2,
	/* 441: |11111111|11111111|11111101|100  V: 250 */ 250,
	/* 442: |11111111|11111111|11111101|101  V: 251 */ 251,
	/* 443: |11111111|11111111|11111101|11  X: 445 */ 2,
	/* 444: |11111111|11111111|11111101|110  V: 252 */ 252,
	/* 445: |11111111|11111111|11111101|111  V: 253 */ 253,
	/* 446: |11111111|11111111|1111111  X: 476 */ 30,
	/* 447: |11111111|11111111|11111110|  X: 461 */ 14,
	/* 448: |11111111|11111111|11111110|0  X: 454 */ 6,
	/* 449: |11111111|11111111|11111110|00  X: 451 */ 2,
	/* 450: |11111111|11111111|11111110|000  V: 254 */ 254,
	/* 451: |11111111|11111111|11111110|001  X: 453 */ 2,
	/* 452: |11111111|11111111|11111110|0010  V: 2 */ 2,
	/* 453: |11111111|11111111|11111110|0011  V: 3 */ 3,
	/* 454: |11111111|11111111|11111110|01  X: 458 */ 4,
	/* 455: |11111111|11111111|11111110|010  X: 457 */ 2,
	/* 456: |11111111|11111111|11111110|0100  V: 4 */ 4,
	/* 457: |11111111|11111111|11111110|0101  V: 5 */ 5,
	/* 458: |11111111|11111111|11111110|011  X: 460 */ 2,
	/* 459: |11111111|11111111|11111110|0110  V: 6 */ 6,
	/* 460: |11111111|11111111|11111110|0111  V: 7 */ 7,
	/* 461: |11111111|11111111|11111110|1  X: 469 */ 8,
	/* 462: |11111111|11111111|11111110|10  X: 466 */ 4,
	/* 463: |11111111|11111111|11111110|100  X: 465 */ 2,
	/* 464: |11111111|11111111|11111110|1000  V: 8 */ 8,
	/* 465: |11111111|11111111|11111110|1001  V: 11 */ 11,
	/* 466: |11111111|11111111|11111110|101  X: 468 */ 2,
	/* 467: |11111111|11111111|11111110|1010  V: 12 */ 12,
	/* 468: |11111111|11111111|11111110|1011  V: 14 */ 14,
	/* 469: |11111111|11111111|11111110|11  X: 473 */ 4,
	/* 470: |11111111|11111111|11111110|110  X: 472 */ 2,
	/* 471: |11111111|11111111|11111110|1100  V: 15 */ 15,
	/* 472: |11111111|11111111|11111110|1101  V: 16 */ 16,
	/* 473: |11111111|11111111|11111110|111  X: 475 */ 2,
	/* 474: |11111111|11111111|11111110|1110  V: 17 */ 17,
	/* 475: |11111111|11111111|11111110|1111  V: 18 */ 18,
	/* 476: |11111111|11111111|11111111|  X: 492 */ 16,
	/* 477: |11111111|11111111|11111111|0  X: 485 */ 8,
	/* 478: |11111111|11111111|11111111|00  X: 482 */ 4,
	/* 479: |11111111|11111111|11111111|000  X: 481 */ 2,
	/* 480: |11111111|11111111|11111111|0000  V: 19 */ 19,
	/* 481: |11111111|11111111|11111111|0001  V: 20 */ 20,
	/* 482: |11111111|11111111|11111111|001  X: 484 */ 2,
	/* 483: |11111111|11111111|11111111|0010  V: 21 */ 21,
	/* 484: |11111111|11111111|11111111|0011  V: 23 */ 23,
	/* 485: |11111111|11111111|11111111|01  X: 489 */ 4,
	/* 486: |11111111|11111111|11111111|010  X: 488 */ 2,
	/* 487: |11111111|11111111|11111111|0100  V: 24 */ 24,
	/* 488: |11111111|11111111|11111111|0101  V: 25 */ 25,
	/* 489: |11111111|11111111|11111111|011  X: 491 */ 2,
	/* 490: |11111111|11111111|11111111|0110  V: 26 */ 26,
	/* 491: |11111111|11111111|11111111|0111  V: 27 */ 27,
	/* 492: |11111111|11111111|11111111|1  X: 500 */ 8,
	/* 493: |11111111|11111111|11111111|10  X: 497 */ 4,
	/* 494: |11111111|11111111|11111111|100  X: 496 */ 2,
	/* 495: |11111111|11111111|11111111|1000  V: 28 */ 28,
	/* 496: |11111111|11111111|11111111|1001  V: 29 */ 29,
	/* 497: |11111111|11111111|11111111|101  X: 499 */ 2,
	/* 498: |11111111|11111111|11111111|1010  V: 30 */ 30,
	/* 499: |11111111|11111111|11111111|1011  V: 31 */ 31,
	/* 500: |11111111|11111111|11111111|11  X: 504 */ 4,
	/* 501: |11111111|11111111|11111111|110  X: 503 */ 2,
	/* 502: |11111111|11111111|11111111|1100  V: 127 */ 127,
	/* 503: |11111111|11111111|11111111|1101  V: 220 */ 220,
	/* 504: |11111111|11111111|11111111|111  X: 506 */ 2,
	/* 505: |11111111|11111111|11111111|1110  V: 249 */ 249,
	/* 506: |11111111|11111111|11111111|1111  X: 510 */ 4,
	/* 507: |11111111|11111111|11111111|11110  X: 509 */ 2,
	/* 508: |11111111|11111111|11111111|111100  V: 10 */ 10,
	/* 509: |11111111|11111111|11111111|111101  V: 13 */ 13,
	/* 510: |11111111|11111111|11111111|11111  X: 512 */ 2,
	/* 511: |11111111|11111111|11111111|111110  V: 22 */ 22};

int qpack_huffman_decode(const uint8_t *bytes, const uint8_t *bytes_max, uint8_t *decoded, size_t max_decoded,
						 size_t *nb_decoded)
{
	int ret = 0;
	uint64_t val_in = 0;
	int bits_in = 0;
	size_t decoded_index = 0;
	int index = 0;
	int was_all_ones = 1;

	while (1) {
		int bit;
		int index_64 = index >> 3;
		int b_index = 7 - (index & 7);

		while (bits_in < 57 && bytes < bytes_max) {
			uint64_t added = *bytes++;
			int shift = 64 - bits_in - 8;
			added <<= shift;
			val_in |= added;
			bits_in += 8;
		}

		if ((_qpack_huffman_bit[index_64] >> b_index) & 1) {
			if (bits_in <= 0) {
				break;
			}
			bit = (val_in >> 63) & 1;
			val_in <<= 1;
			bits_in--;
			if (bit) {
				index += _qpack_huffman_val[index];
			} else {
				index++;
				was_all_ones = 0;
			}
			if (index >= 512) {
				break;
			}
		} else if (decoded_index < max_decoded) {
			decoded[decoded_index++] = _qpack_huffman_val[index];
			index = 0;
			was_all_ones = 1;
		} else {
			was_all_ones = 1;
			break;
		}
	}

	if (!was_all_ones) {
		ret = -1;
	}

	*nb_decoded = decoded_index;

	return ret;
}

/* clang-format off */
static struct qpack_header_field _http3_static_header_table[] = {
    {.name = ":authority", .value = ""},
    {.name = ":path", .value = "/"},
    {.name = "age", .value = "0"},
    {.name = "content-disposition", .value = ""},
    {.name = "content-length", .value = "0"},
    {.name = "cookie", .value = ""},
    {.name = "date", .value = ""},
    {.name = "etag", .value = ""},
    {.name = "if-modified-since", .value = ""},
    {.name = "if-none-match", .value = ""},
    {.name = "last-modified", .value = ""},
    {.name = "link", .value = ""},
    {.name = "location", .value = ""},
    {.name = "referer", .value = ""},
    {.name = "set-cookie", .value = ""},
    {.name = ":method", .value = "CONNECT"},
    {.name = ":method", .value = "DELETE"},
    {.name = ":method", .value = "GET"},
    {.name = ":method", .value = "HEAD"},
    {.name = ":method", .value = "OPTIONS"},
    {.name = ":method", .value = "POST"},
    {.name = ":method", .value = "PUT"},
    {.name = ":scheme", .value = "http"},
    {.name = ":scheme", .value = "https"},
    {.name = ":status", .value = "103"},
    {.name = ":status", .value = "200"},
    {.name = ":status", .value = "304"},
    {.name = ":status", .value = "404"},
    {.name = ":status", .value = "503"},
    {.name = "accept", .value = "*/*"},
    {.name = "accept", .value = "application/dns-message"},
    {.name = "accept-encoding", .value = "gzip, .value = deflate, .value = br"},
    {.name = "accept-ranges", .value = "bytes"},
    {.name = "access-control-allow-headers", .value = "cache-control"},
    {.name = "access-control-allow-headers", .value = "content-type"},
    {.name = "access-control-allow-origin", .value = "*"},
    {.name = "cache-control", .value = "max-age=0"},
    {.name = "cache-control", .value = "max-age=2592000"},
    {.name = "cache-control", .value = "max-age=604800"},
    {.name = "cache-control", .value = "no-cache"},
    {.name = "cache-control", .value = "no-store"},
    {.name = "cache-control", .value = "public, .value = max-age=31536000"},
    {.name = "content-encoding", .value = "br"},
    {.name = "content-encoding", .value = "gzip"},
    {.name = "content-type", .value = "application/dns-message"},
    {.name = "content-type", .value = "application/javascript"},
    {.name = "content-type", .value = "application/json"},
    {.name = "content-type", .value = "application/x-www-form-urlencoded"},
    {.name = "content-type", .value = "image/gif"},
    {.name = "content-type", .value = "image/jpeg"},
    {.name = "content-type", .value = "image/png"},
    {.name = "content-type", .value = "text/css"},
    {.name = "content-type", .value = "text/html; charset=utf-8"},
    {.name = "content-type", .value = "text/plain"},
    {.name = "content-type", .value = "text/plain;charset=utf-8"},
    {.name = "range", .value = "bytes=0-"},
    {.name = "strict-transport-security", .value = "max-age=31536000"},
    {.name = "strict-transport-security", .value = "max-age=31536000; includesubdomains"},
    {.name = "strict-transport-security", .value = "max-age=31536000; includesubdomains; preload"},
    {.name = "vary", .value = "accept-encoding"},
    {.name = "vary", .value = "origin"},
    {.name = "x-content-type-options", .value = "nosniff"},
    {.name = "x-xss-protection", .value = "1; mode=block"},
    {.name = ":status", .value = "100"},
    {.name = ":status", .value = "204"},
    {.name = ":status", .value = "206"},
    {.name = ":status", .value = "302"},
    {.name = ":status", .value = "400"},
    {.name = ":status", .value = "403"},
    {.name = ":status", .value = "421"},
    {.name = ":status", .value = "425"},
    {.name = ":status", .value = "500"},
    {.name = "accept-language", .value = ""},
    {.name = "access-control-allow-credentials", .value = "FALSE"},
    {.name = "access-control-allow-credentials", .value = "TRUE"},
    {.name = "access-control-allow-headers", .value = "*"},
    {.name = "access-control-allow-methods", .value = "get"},
    {.name = "access-control-allow-methods", .value = "get, .value = post, .value = options"},
    {.name = "access-control-allow-methods", .value = "options"},
    {.name = "access-control-expose-headers", .value = "content-length"},
    {.name = "access-control-request-headers", .value = "content-type"},
    {.name = "access-control-request-method", .value = "get"},
    {.name = "access-control-request-method", .value = "post"},
    {.name = "alt-svc", .value = "clear"},
    {.name = "authorization"},
    {.name = "content-security-policy", .value = "script-src 'none'; object-src 'none'; base-uri 'none'"},
    {.name = "early-data", .value = "1"},
    {.name = "expect-ct", .value = ""},
    {.name = "forwarded", .value = ""},
    {.name = "if-range", .value = ""},
    {.name = "origin", .value = ""},
    {.name = "purpose", .value = "prefetch"},
    {.name = "server", .value = ""},
    {.name = "timing-allow-origin", .value = "*"},
    {.name = "upgrade-insecure-requests", .value = "1"},
    {.name = "user-agent", .value = ""},
    {.name = "x-forwarded-for", .value = ""},
    {.name = "x-frame-options", .value = "deny"},
    {.name = "x-frame-options", .value = "sameorigin"},
 };
 static int _http3_static_header_table_len = sizeof(_http3_static_header_table) / sizeof(struct qpack_header_field);
/* clang-format on */

struct qpack_header_field *qpack_get_static_header_field(int index)
{
	if (index < 0 || index >= _http3_static_header_table_len) {
		return NULL;
	}
	return &_http3_static_header_table[index];
}