// Copyright (c) HashiCorp, Inc.
// SPDX-License-Identifier: BUSL-1.1

package structs

type HostVolumeState struct {
	ID        string
	HostPath  string
	CreateReq *ClientHostVolumeCreateRequest
}

type ClientHostVolumeCreateRequest struct {
	// ID is a UUID-like string generated by the server.
	ID string

	// Name is the name that group.volume will use to identify the volume
	// source. Not expected to be unique.
	Name string

	// PluginID is the name of the host volume plugin on the client that will be
	// used for creating the volume. If omitted, the client will use its default
	// built-in plugin.
	PluginID string

	// Namespace is the Nomad namespace for the volume.
	// It's in the client RPC to be included in plugin execution environment.
	Namespace string

	// NodeID is the node where the volume is placed. It's included in the
	// client RPC request so that the server can route the request to the
	// correct node.
	NodeID string

	// Because storage may allow only specific intervals of size, we accept a
	// min and max and return the actual capacity when the volume is created or
	// updated on the client
	RequestedCapacityMinBytes int64
	RequestedCapacityMaxBytes int64

	// Parameters are an opaque map of parameters for the host volume plugin.
	Parameters map[string]string
}

type ClientHostVolumeCreateResponse struct {
	VolumeName string
	VolumeID   string

	// HostPath is the host path where the volume's mount point was created. We
	// send this back to the server to make debugging easier.
	HostPath string

	// Capacity is the size in bytes that was actually provisioned by the host
	// volume plugin.
	CapacityBytes int64
}

type ClientHostVolumeRegisterRequest struct {
	// ID is a UUID-like string generated by the server.
	ID string

	// Name is the name that group.volume will use to identify the volume
	// source. Not expected to be unique cluster-wide, but must be unique per
	// node.
	Name string

	// NodeID is the node where the volume is placed. It's included in the
	// client RPC request so that the server can route the request to the
	// correct node.
	NodeID string

	// HostPath is the host path where the volume's mount point was created
	// out-of-band.
	HostPath string

	// Capacity is the size in bytes that was provisioned out-of-band.
	CapacityBytes int64

	// Parameters are an opaque map of parameters for the host volume plugin.
	Parameters map[string]string
}

type ClientHostVolumeRegisterResponse struct{}

type ClientHostVolumeDeleteRequest struct {
	// ID is a UUID-like string generated by the server.
	ID string

	Name string

	// PluginID is the name of the host volume plugin on the client that will be
	// used for deleting the volume. If omitted, the client will use its default
	// built-in plugin.
	PluginID string

	// Namespace is the Nomad namespace for the volume.
	// It's in the client RPC to be included in plugin execution environment.
	Namespace string

	// NodeID is the node where the volume is placed. It's included in the
	// client RPC request so that the server can route the request to the
	// correct node.
	NodeID string

	// HostPath is the host path where the volume's mount point was created.
	// We send this from the server to allow verification by plugins.
	HostPath string

	// Parameters are an opaque map of parameters for the host volume plugin.
	Parameters map[string]string
}

type ClientHostVolumeDeleteResponse struct {
	VolumeName string
	VolumeID   string
}
