mod item_row;
mod item_row_dnd;
mod models;

use self::item_row::ItemRow;
pub use self::models::{FeedListItem, FeedListItemID, FeedListTree};
use super::SideBar;
use crate::app::App;
use crate::gobject_models::{GFeedListItem, GSidebarSelection};
use diffus::Diffable;
use diffus::edit::{Edit, collection};
use gio::{ListModel, ListStore};
use glib::{Object, SignalHandlerId, subclass::*};
use gtk4::{
    Accessible, Buildable, CompositeTemplate, ConstraintTarget, CustomFilter, FilterChange, FilterListModel, ListItem,
    SingleSelection, TreeListModel, TreeListRow, Widget, prelude::*,
};
use libadwaita::{Bin, subclass::prelude::*};
use once_cell::sync::Lazy;
use std::cell::RefCell;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate)]
    #[template(file = "data/resources/ui_templates/sidebar/feed_list.blp")]
    pub struct FeedList {
        #[template_child]
        pub selection: TemplateChild<SingleSelection>,
        #[template_child]
        pub filter_model: TemplateChild<FilterListModel>,
        #[template_child]
        pub filter: TemplateChild<CustomFilter>,
        #[template_child]
        pub list_store: TemplateChild<ListStore>,

        pub tree: RefCell<FeedListTree>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for FeedList {
        const NAME: &'static str = "FeedList";
        type ParentType = Bin;
        type Type = super::FeedList;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    impl ObjectImpl for FeedList {
        fn constructed(&self) {
            let tree_model = TreeListModel::new(self.list_store.clone(), false, false, |item| {
                item.downcast_ref::<GFeedListItem>()
                    .and_then(GFeedListItem::children)
                    .and_upcast::<ListModel>()
            });

            self.filter.set_filter_func(Self::filter_callback);
            self.filter_model.set_model(Some(&tree_model));
        }

        fn signals() -> &'static [Signal] {
            static SIGNALS: Lazy<Vec<Signal>> = Lazy::new(|| {
                vec![
                    Signal::builder("activated")
                        .action()
                        .param_types([super::FeedList::static_type()])
                        .build(),
                    Signal::builder("selection-changed")
                        .param_types([super::FeedList::static_type(), GSidebarSelection::static_type()])
                        .build(),
                ]
            });

            SIGNALS.as_ref()
        }
    }

    impl WidgetImpl for FeedList {}

    impl BinImpl for FeedList {}

    #[gtk4::template_callbacks]
    impl FeedList {
        #[template_callback]
        fn factory_setup(&self, obj: &Object) {
            let Some(list_item) = obj.downcast_ref::<ListItem>() else {
                return;
            };

            let row = ItemRow::new();
            list_item.set_child(Some(&row));
        }

        #[template_callback]
        fn factory_bind(&self, obj: &Object) {
            let Some(list_item) = obj.downcast_ref::<ListItem>() else {
                return;
            };
            let Some(tree_list_row) = list_item.item().and_downcast::<TreeListRow>() else {
                return;
            };
            let Some(item_row) = list_item.child().and_downcast::<ItemRow>() else {
                return;
            };
            let Some(model) = tree_list_row.item().and_downcast::<GFeedListItem>() else {
                return;
            };

            item_row.bind_model(&model, &tree_list_row);
        }

        #[template_callback]
        fn on_listview_activate(&self, _pos: u32) {
            self.obj().activate();
        }

        #[template_callback]
        fn on_selection_changed(&self, _pos: u32, _n_items: u32) {
            let obj = self.obj();
            let pos = self.selection.selected();
            let selection = self
                .selection
                .item(pos)
                .and_downcast_ref::<TreeListRow>()
                .and_then(TreeListRow::item)
                .and_downcast::<GFeedListItem>()
                .map(|item| GSidebarSelection::from_feed_list(item, pos));
            if let Some(selection) = selection {
                obj.emit_by_name::<()>("selection-changed", &[&*obj, &selection])
            }
        }

        fn filter_callback(obj: &Object) -> bool {
            let tree_list_row = obj
                .downcast_ref::<TreeListRow>()
                .expect("The object needs to be of type `TreeListRow`.");

            let item_gobject = tree_list_row
                .item()
                .expect("No item in TreeListRow")
                .downcast::<GFeedListItem>()
                .expect("The object needs to be of type `GFeedListItem`.");

            let item_count = item_gobject.item_count();

            // Only allow even numbers
            !App::default().settings().feed_list().only_show_relevant() || item_count > 0
        }
    }
}

glib::wrapper! {
    pub struct FeedList(ObjectSubclass<imp::FeedList>)
        @extends Widget, Bin,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for FeedList {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}

impl FeedList {
    pub fn instance() -> Self {
        SideBar::instance().imp().feed_list.get()
    }

    pub fn update(&self, new_tree: FeedListTree) {
        let imp = self.imp();

        let mut new_tree = new_tree;
        let mut old_tree = FeedListTree::new();

        std::mem::swap(&mut old_tree, &mut *imp.tree.borrow_mut());

        let diff = old_tree.top_level.diff(&new_tree.top_level);

        Self::process_diff(&diff, &imp.list_store);

        std::mem::swap(&mut new_tree, &mut *imp.tree.borrow_mut());
        imp.filter.changed(FilterChange::Different);
    }

    fn process_diff(diff: &Edit<Vec<FeedListItem>>, list_store: &ListStore) {
        let mut pos = 0;

        match diff {
            Edit::Copy(_list) => { /* no difference */ }
            Edit::Change(diff) => {
                for edit in diff {
                    match edit {
                        collection::Edit::Copy(_item) => {
                            // nothing changed
                            pos += 1;
                        }
                        &collection::Edit::Insert(item) => {
                            let item_gobject: GFeedListItem = item.into();
                            list_store.insert(pos, &item_gobject);
                            pos += 1;
                        }
                        collection::Edit::Remove(_item) => {
                            list_store.remove(pos);
                        }
                        collection::Edit::Change(diff) => {
                            let item = list_store.item(pos).and_downcast::<GFeedListItem>();
                            if let Some(item_gobject) = item {
                                if let Some(new_label) = diff.label {
                                    item_gobject.set_label(new_label.to_string());
                                }
                                if let Some(new_count) = diff.item_count {
                                    item_gobject.set_item_count(new_count);
                                }
                                if let Some(error_message) = diff.error_message {
                                    item_gobject.set_error_message(error_message.to_string());
                                }
                                if let Some(child_list_store) = item_gobject.children() {
                                    Self::process_diff(&diff.child_diff, &child_list_store);
                                }
                            }
                            pos += 1;
                        }
                    }
                }
            }
        };
    }

    pub fn expand_collapse_selected_category(&self) {
        let imp = self.imp();

        if let Some(row) = imp.selection.selected_item()
            && let Ok(tree_list_row) = row.downcast::<TreeListRow>()
        {
            let is_expanded = tree_list_row.is_expanded();
            tree_list_row.set_expanded(!is_expanded);
        }
    }

    pub fn get_expanded_categories(&self) -> Vec<u32> {
        let mut expanded_categories = Vec::new();
        let mut index = 0;
        while let Some(tree_list_row) = self.imp().selection.item(index).and_downcast_ref::<TreeListRow>() {
            if tree_list_row.is_expanded() {
                expanded_categories.push(index);
            }

            index += 1;
        }

        expanded_categories
    }

    pub fn restore_expanded_categories(&self, mut expanded_categories: Vec<u32>) {
        let imp = self.imp();
        expanded_categories.sort();

        for category_index in expanded_categories {
            if let Some(tree_list_row) = imp.selection.item(category_index).and_downcast_ref::<TreeListRow>() {
                tree_list_row.set_expanded(true);
            }
        }
    }

    pub fn clear_selection(&self) {
        self.imp().selection.set_selected(gtk4::INVALID_LIST_POSITION);
    }

    pub fn first_item(&self) -> Option<GSidebarSelection> {
        self.imp()
            .selection
            .item(0)
            .and_downcast_ref::<TreeListRow>()
            .and_then(TreeListRow::item)
            .and_downcast::<GFeedListItem>()
            .map(|item| GSidebarSelection::from_feed_list(item, 0))
    }

    pub fn last_item(&self) -> Option<GSidebarSelection> {
        let imp = self.imp();
        let count = imp.selection.n_items();

        if count > 0 {
            let pos = count - 1;

            imp.selection
                .item(pos)
                .and_downcast_ref::<TreeListRow>()
                .and_then(TreeListRow::item)
                .and_downcast::<GFeedListItem>()
                .map(|item| GSidebarSelection::from_feed_list(item, pos))
        } else {
            None
        }
    }

    pub fn next_item(&self) -> Option<GSidebarSelection> {
        let imp = self.imp();
        let selected_pos = imp.selection.selected();

        if selected_pos == gtk4::INVALID_LIST_POSITION {
            None
        } else {
            let next_pos = selected_pos + 1;
            imp.selection.set_selected(selected_pos + 1);
            imp.selection
                .item(selected_pos + 1)
                .and_downcast_ref::<TreeListRow>()
                .and_then(TreeListRow::item)
                .and_downcast::<GFeedListItem>()
                .map(|item| GSidebarSelection::from_feed_list(item, next_pos))
        }
    }

    pub fn prev_item(&self) -> Option<GSidebarSelection> {
        let imp = self.imp();
        let selected_pos = imp.selection.selected();

        if selected_pos == gtk4::INVALID_LIST_POSITION || selected_pos == 0 {
            None
        } else {
            let prev_pos = selected_pos - 1;
            imp.selection.set_selected(prev_pos);
            imp.selection
                .item(prev_pos)
                .and_downcast_ref::<TreeListRow>()
                .and_then(TreeListRow::item)
                .and_downcast::<GFeedListItem>()
                .map(|item| GSidebarSelection::from_feed_list(item, prev_pos))
        }
    }

    pub fn set_selection(&self, pos: u32) {
        self.imp().selection.set_selected(pos);
    }

    pub fn activate(&self) {
        self.emit_by_name::<()>("activated", &[&self.clone()]);
    }

    pub fn connect_activated<F: Fn(&Self) + 'static>(&self, f: F) -> SignalHandlerId {
        self.connect_local("activated", false, move |args| {
            let feed_list = args[1].get::<Self>().expect("The value needs to be of type `FeedList`");
            f(&feed_list);
            None
        })
    }
}
