use gio::{ActionEntry, prelude::*};
use glib::{Variant, VariantType};

use crate::app::App;
use crate::article_list::{OpenArticleInBrowser, ReadUpdate};
use crate::content_page::{ArticleListColumn, ArticleViewColumn};
use crate::gobject_models::GRead;
use crate::main_window::MainWindow;
use crate::sidebar::FeedList;

pub struct MiscActions;

impl MiscActions {
    pub fn setup() {
        // -------------------------
        // open selected article in browser
        // -------------------------
        let open_selected_article = ActionEntry::builder("open-selected-article-in-browser")
            .activate(|_win, _action, _parameter| Self::open_selected_article_in_browser())
            .build();

        // -------------------------
        // open article in browser
        // -------------------------
        let open_article = ActionEntry::builder("open-article-in-browser")
            .parameter_type(Some(&OpenArticleInBrowser::static_variant_type()))
            .activate(|_win, _action, parameter| Self::open_article_in_browser(parameter))
            .build();

        // -------------------------
        // open uri in browser
        // -------------------------
        let open_uri = ActionEntry::builder("open-uri-in-browser")
            .parameter_type(VariantType::new("s").as_deref().ok())
            .activate(|_win, _action, parameter| Self::open_uri(parameter))
            .build();

        // ---------------------------
        // ignore tls error
        // ---------------------------
        let ignore_tls_errors = ActionEntry::builder("ignore-tls-errors")
            .activate(|_win, _action, _parameter| Self::ignore_tls_errors())
            .build();

        // ---------------------------
        // focus search action
        // ---------------------------
        let focus_search = ActionEntry::builder("focus-search")
            .activate(|_win, _action, _parameter| ArticleListColumn::instance().focus_search())
            .build();

        // ---------------------------
        // toggle category expanded
        // ---------------------------
        let toggle_category_expanded = ActionEntry::builder("toggle-category-expanded")
            .activate(|_win, _action, _parameter| FeedList::instance().expand_collapse_selected_category())
            .build();

        // ---------------------------
        // tag popover
        // ---------------------------
        let tag_article = ActionEntry::builder("tag-popover")
            .activate(|_win, _action, _parameter| ArticleViewColumn::instance().popup_tag_popover())
            .build();

        MainWindow::instance().add_action_entries([
            open_selected_article,
            open_article,
            open_uri,
            ignore_tls_errors,
            focus_search,
            toggle_category_expanded,
            tag_article,
        ]);
    }

    fn open_uri(parameter: Option<&Variant>) {
        let Some(uri) = parameter.and_then(Variant::str) else {
            tracing::warn!("open uri: no parameter");
            return;
        };

        App::open_url_in_default_browser(uri);
    }

    fn open_article_in_browser(parameter: Option<&Variant>) {
        let Some(open_msg) = parameter.and_then(OpenArticleInBrowser::from_variant) else {
            return;
        };

        App::open_url_in_default_browser(&open_msg.url);

        if open_msg.read == GRead::Unread {
            let update = ReadUpdate {
                article_id: open_msg.article_id,
                read: GRead::Read,
            };
            MainWindow::activate_action("set-article-read", Some(&update.to_variant()));
        }
    }

    fn open_selected_article_in_browser() {
        let Some(article_model) = ArticleViewColumn::instance().article() else {
            tracing::warn!("Open selected article in browser: No article Selected.");
            return;
        };

        let Some(url) = article_model.url() else {
            tracing::warn!("Open selected article in browser: No url available.");
            return;
        };

        App::open_url_in_default_browser(&url);
    }

    fn ignore_tls_errors() {
        let res1 = App::default().settings().set_accept_invalid_certs(true);
        let res2 = App::default().settings().set_accept_invalid_hostnames(true);

        if res1.is_err() || res2.is_err() {
            tracing::error!("Error writing settings");
        }
    }
}
