use crate::app::App;
use crate::article_list::{MarkUpdate, ReadUpdate};
use crate::article_view::{ArticleLoader, ArticleView};
use crate::gobject_models::{GArticle, GArticleID, GMarked, GRead};
use crate::i18n::i18n;
use crate::infrastructure::TokioRuntime;
use crate::main_window::MainWindow;
use crate::settings::Keybindings;
use crate::share::share_popover::SharePopover;
use crate::tag_popover::TagPopover;
use crate::util::constants;
use gio::{SimpleAction, SimpleActionGroup};
use glib::{Object, Properties, SignalHandlerId, clone, subclass::prelude::*, subclass::*};
use gtk4::{
    Accessible, Box, Buildable, CompositeTemplate, ConstraintTarget, GestureClick, MenuButton, NamedAction,
    PopoverMenu, Shortcut, Stack, ToggleButton, Widget, prelude::*, subclass::prelude::*,
};
use news_flash::models::{ArticleID, PluginCapabilities};
use std::cell::{Cell, RefCell};

use super::ContentPage;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::ArticleViewColumn)]
    #[template(file = "data/resources/ui_templates/article_view/column.blp")]
    pub struct ArticleViewColumn {
        #[template_child]
        pub article_view: TemplateChild<ArticleView>,
        #[template_child]
        pub scrape_content_button: TemplateChild<ToggleButton>,
        #[template_child]
        pub share_button: TemplateChild<MenuButton>,
        #[template_child]
        pub scrape_content_stack: TemplateChild<Stack>,
        #[template_child]
        pub tag_button: TemplateChild<MenuButton>,
        #[template_child]
        pub tag_button_click: TemplateChild<GestureClick>,
        #[template_child]
        pub more_actions_button: TemplateChild<MenuButton>,
        #[template_child]
        pub more_actions_stack: TemplateChild<Stack>,
        #[template_child]
        pub tag_popover: TemplateChild<TagPopover>,
        #[template_child]
        pub share_popover: TemplateChild<SharePopover>,

        #[template_child]
        pub mark_article_read_button: TemplateChild<ToggleButton>,
        #[template_child]
        pub mark_article_button: TemplateChild<ToggleButton>,

        #[template_child]
        pub zoom_in_shortcut: TemplateChild<Shortcut>,
        #[template_child]
        pub zoom_out_shortcut: TemplateChild<Shortcut>,
        #[template_child]
        pub zoom_reset_shortcut: TemplateChild<Shortcut>,

        #[property(get, set, nullable)]
        pub article: RefCell<Option<GArticle>>,

        #[property(get, set, name = "have-article-and-online")]
        pub have_article_and_online: Cell<bool>,

        #[property(get, set, name = "can-tag")]
        pub can_tag: Cell<bool>,

        #[property(get, set, name = "block-next-view-switch")]
        pub block_next_view_switch: Cell<bool>,

        pub scrape_content_button_signal_id: RefCell<Option<SignalHandlerId>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ArticleViewColumn {
        const NAME: &'static str = "ArticleViewColumn";
        type ParentType = gtk4::Box;
        type Type = super::ArticleViewColumn;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for ArticleViewColumn {
        fn constructed(&self) {
            self.insert_menu_controls();
            self.bind_article_and_online();
            self.bind_scrape_content_button();
            self.bind_more_actions_stack();
            self.bind_can_tag();

            // Setup dynamic tooltip bindings for keybindings
            let keybindings_article_list = App::default().settings().keybindings().article_list();
            let keybindings_article_view = App::default().settings().keybindings().article_view();

            Self::bind_keybinding_tooltip(
                keybindings_article_list.upcast_ref(),
                "read",
                self.mark_article_read_button.upcast_ref(),
                i18n("Toggle Read"),
            );
            Self::bind_keybinding_tooltip(
                keybindings_article_list.upcast_ref(),
                "mark",
                self.mark_article_button.upcast_ref(),
                i18n("Toggle Starred"),
            );
            Self::bind_keybinding_tooltip(
                keybindings_article_view.upcast_ref(),
                "scrap-content",
                self.scrape_content_button.upcast_ref(),
                i18n("Try to Show Full Content"),
            );
            Self::bind_keybinding_tooltip(
                keybindings_article_view.upcast_ref(),
                "tag",
                self.tag_button.upcast_ref(),
                i18n("Tag Article"),
            );

            let zoom_in_action = SimpleAction::new("zoom-in", None);
            zoom_in_action.connect_activate(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |_action, _parameter| {
                    let view = imp.article_view.get();
                    let zoom = view.zoom();
                    if zoom < constants::ARTICLE_ZOOM_UPPER {
                        view.set_zoom(zoom + 0.25);
                    }
                }
            ));

            let zoom_out_action = SimpleAction::new("zoom-out", None);
            zoom_out_action.connect_activate(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |_action, _parameter| {
                    let view = imp.article_view.get();
                    let zoom = view.zoom();
                    if zoom > constants::ARTICLE_ZOOM_LOWER {
                        view.set_zoom(zoom - 0.25);
                    }
                }
            ));

            let reset_zoom_action = SimpleAction::new("reset-zoom", None);
            reset_zoom_action.connect_activate(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |_action, _parameter| {
                    imp.article_view.set_zoom(1.0);
                }
            ));

            let increase_fontsize_action = SimpleAction::new("increase-fontsize", None);
            increase_fontsize_action.connect_activate(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |_action, _parameter| {
                    let view = imp.article_view.get();
                    let font_size = view.font_size();
                    if font_size < constants::FONT_SIZE_UPPER {
                        view.set_font_size(font_size + 1);
                    }
                }
            ));

            let decrease_fontsize_action = SimpleAction::new("decrease-fontsize", None);
            decrease_fontsize_action.connect_activate(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |_action, _parameter| {
                    let view = imp.article_view.get();
                    let font_size = view.font_size();
                    if font_size > constants::FONT_SIZE_LOWER {
                        view.set_font_size(font_size - 1);
                    }
                }
            ));

            let reset_fontsize_action = SimpleAction::new("reset-fontsize", None);
            reset_fontsize_action.connect_activate(clone!(
                #[weak(rename_to = imp)]
                self,
                #[upgrade_or_panic]
                move |_action, _parameter| {
                    imp.article_view.set_font_size(constants::DEFAULT_FONT_SIZE);
                }
            ));

            let obj = self.obj();
            let action_group = SimpleActionGroup::new();
            action_group.add_action(&zoom_in_action);
            action_group.add_action(&zoom_out_action);
            action_group.add_action(&reset_zoom_action);
            action_group.add_action(&increase_fontsize_action);
            action_group.add_action(&decrease_fontsize_action);
            action_group.add_action(&reset_fontsize_action);
            obj.insert_action_group("articleview", Some(&action_group));

            self.zoom_in_shortcut
                .set_action(Some(NamedAction::new("articleview.zoom-in")));
            self.zoom_out_shortcut
                .set_action(Some(NamedAction::new("articleview.zoom-out")));
            self.zoom_reset_shortcut
                .set_action(Some(NamedAction::new("articleview.reset-zoom")));
        }
    }

    impl WidgetImpl for ArticleViewColumn {}

    impl BoxImpl for ArticleViewColumn {}

    #[gtk4::template_callbacks]
    impl ArticleViewColumn {
        #[template_callback]
        fn have_article(&self, article: Option<GArticle>) -> bool {
            article.is_some()
        }

        #[template_callback]
        fn is_unread(&self, article: Option<GArticle>) -> bool {
            article.map(|a| a.read() == GRead::Unread).unwrap_or(false)
        }

        #[template_callback]
        fn is_marked(&self, article: Option<GArticle>) -> bool {
            article.map(|a| a.marked() == GMarked::Marked).unwrap_or(false)
        }

        #[template_callback]
        fn have_enclosures(&self, article: Option<GArticle>) -> bool {
            article.map(|a| !a.enclosures().is_empty()).unwrap_or(false)
        }

        #[template_callback]
        fn on_marked_clicked(&self) {
            let Some(article) = &*self.article.borrow() else {
                return;
            };

            let update = MarkUpdate {
                article_id: article.article_id(),
                marked: article.marked().invert(),
            };

            MainWindow::activate_action("set-article-marked", Some(&update.to_variant()));
        }

        #[template_callback]
        fn on_read_clicked(&self) {
            let Some(article) = &*self.article.borrow() else {
                return;
            };

            let update = ReadUpdate {
                article_id: article.article_id(),
                read: article.read().invert(),
            };

            MainWindow::activate_action("set-article-read", Some(&update.to_variant()));
        }

        #[template_callback]
        fn active_to_stack(&self, active: bool) -> &'static str {
            Self::bool_to_scrape_page(active)
        }

        fn bind_keybinding_tooltip(source: &Object, source_property: &str, target: &Object, tooltip: String) {
            source
                .bind_property(source_property, target, "tooltip-text")
                .transform_to(move |_binding, key_binding: Option<String>| {
                    let combined_tooltip = key_binding
                        .as_deref()
                        .and_then(Keybindings::parse_shortcut_string)
                        .map(|key| format!("{tooltip} ({key})"))
                        .unwrap_or(tooltip.clone());
                    Some(combined_tooltip)
                })
                .sync_create()
                .build();
        }

        fn bool_to_scrape_page(active: bool) -> &'static str {
            if active { "active" } else { "inactive" }
        }

        fn bool_to_actions_page(active: bool) -> &'static str {
            if active { "spinner" } else { "button" }
        }

        fn on_scrape_toggled(&self) {
            let active = self.scrape_content_button.is_active();

            if active {
                MainWindow::activate_action("scrape-content", None);
            } else {
                self.article_view.set_prefer_scraped_content(false);
                self.article_view.redraw_article();
            }
        }

        fn bind_scrape_content_button(&self) {
            App::default()
                .bind_property("is-scraping-content", &*self.scrape_content_stack, "visible-child-name")
                .transform_to(|_binding, active: bool| Some(Self::bool_to_scrape_page(active)))
                .build();

            let signal_id = self.scrape_content_button.connect_toggled(clone!(
                #[weak(rename_to = imp)]
                self,
                move |_button| imp.on_scrape_toggled()
            ));
            self.scrape_content_button_signal_id.replace(Some(signal_id));
        }

        fn bind_more_actions_stack(&self) {
            App::default()
                .bind_property("is-exporting-article", &*self.more_actions_stack, "visible-child-name")
                .transform_to(|_binding, active: bool| Some(Self::bool_to_actions_page(active)))
                .build();
        }

        fn bind_article_and_online(&self) {
            let obj = self.obj();

            obj.connect_article_notify(|obj| {
                let have_article = obj.article().is_some();
                let value = if App::default().is_offline() {
                    false
                } else {
                    have_article
                };
                obj.set_have_article_and_online(value);
            });

            App::default().connect_is_offline_notify(clone!(
                #[weak]
                obj,
                move |app| {
                    let have_article = obj.article().is_some();
                    let value = if app.is_offline() { false } else { have_article };
                    obj.set_have_article_and_online(value);
                }
            ));
        }

        fn bind_can_tag(&self) {
            let obj = self.obj();

            obj.connect_article_notify(|obj| {
                let have_article_and_online = obj.have_article_and_online();
                let support_tags = App::default()
                    .features()
                    .as_ref()
                    .contains(PluginCapabilities::SUPPORT_TAGS);
                obj.set_can_tag(have_article_and_online && support_tags);
            });

            App::default().connect_features_notify(clone!(
                #[weak]
                obj,
                move |app| {
                    let have_article_and_online = obj.have_article_and_online();
                    let support_tags = app.features().as_ref().contains(PluginCapabilities::SUPPORT_TAGS);
                    obj.set_can_tag(have_article_and_online && support_tags);
                }
            ));
        }

        fn insert_menu_controls(&self) {
            let Some(popover) = self.more_actions_button.popover().and_downcast::<PopoverMenu>() else {
                return;
            };

            popover.connect_closed(|_popover| {
                if let Err(error) = App::default().settings().write() {
                    ContentPage::instance()
                        .simple_message(&crate::i18n::i18n(&format!("Failed to write settings file: {error}")));
                }
            });

            let articleview_settings = App::default().settings().article_view();

            let zoom_control = crate::article_popover::ZoomControl::default();
            self.article_view
                .bind_property("zoom", &zoom_control, "percentage")
                .transform_to(|_binding, b: f64| Some(format!("{:.0} %", b * 100.0)))
                .build();

            let font_style = crate::article_popover::FontStyle::default();
            articleview_settings
                .bind_property("font-style", &font_style, "font-style")
                .bidirectional()
                .sync_create()
                .build();
            font_style.connect_font_style_notify(clone!(
                #[weak(rename_to = article_view)]
                self.article_view,
                move |_| article_view.redraw_article()
            ));

            let font_size = crate::article_popover::FontSizeControl::default();
            articleview_settings
                .bind_property("font-size", &font_size, "font-size")
                .sync_create()
                .build();

            let content_width = crate::article_popover::Slider::default();
            content_width.set_title("Content Width");
            content_width.set_icon_name("content-width-symbolic");
            content_width.add_marks(&[300.0, 450.0, 650.0, 900.0, 1200.0]);
            articleview_settings
                .bind_property("content-width", &content_width, "value")
                .bidirectional()
                .sync_create()
                .build();

            let line_height = crate::article_popover::Slider::default();
            line_height.set_title("Line Height");
            line_height.set_icon_name("line-height-symbolic");
            line_height.add_marks(&[1.4, 1.8, 2.2, 2.6]);
            articleview_settings
                .bind_property("line-height", &line_height, "value")
                .bidirectional()
                .sync_create()
                .build();

            popover.add_child(&font_style, "font-style");
            popover.add_child(&font_size, "font-size");
            popover.add_child(&content_width, "content-width");
            popover.add_child(&line_height, "line-height");
            popover.add_child(&zoom_control, "zoom");
        }
    }
}

glib::wrapper! {
    pub struct ArticleViewColumn(ObjectSubclass<imp::ArticleViewColumn>)
        @extends Widget, Box,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for ArticleViewColumn {
    fn default() -> Self {
        glib::Object::new()
    }
}

impl ArticleViewColumn {
    pub fn instance() -> Self {
        ContentPage::instance().imp().articleview_column.get()
    }

    pub fn show_article(&self, article_id: GArticleID, overwrite_read: bool) {
        TokioRuntime::execute_with_callback(
            move || async move {
                let news_flash = App::news_flash();
                let news_flash_guard = news_flash.read().await;
                let news_flash = news_flash_guard.as_ref()?;
                ArticleLoader::load(&article_id, news_flash)
            },
            clone!(
                #[weak(rename_to = this)]
                self,
                move |res: Option<ArticleLoader>| {
                    let Some(loader) = res else {
                        return;
                    };

                    let imp = this.imp();
                    let new_article = loader.build();

                    if overwrite_read {
                        new_article.set_read(GRead::Read);
                    }

                    // block scrape content button toggled signal
                    if let Some(handler_id) = imp.scrape_content_button_signal_id.borrow().as_ref() {
                        imp.scrape_content_button.block_signal(handler_id);
                    }

                    this.set_article(Some(new_article));

                    // unblock scrape content button toggled signal
                    if let Some(handler_id) = imp.scrape_content_button_signal_id.borrow().as_ref() {
                        imp.scrape_content_button.unblock_signal(handler_id);
                    }

                    if !MainWindow::instance().is_fullscreen() {
                        let show_article_view =
                            ContentPage::instance().is_article_view_hidden() && !this.block_next_view_switch();
                        tracing::debug!(%show_article_view);
                        ContentPage::instance().show_article_view(show_article_view);
                    }

                    this.set_block_next_view_switch(false);
                },
            ),
        );
    }

    pub fn reload(&self) {
        let Some(article) = self.article() else {
            return;
        };

        let article_id = article.article_id();

        TokioRuntime::execute_with_callback(
            move || async move {
                let news_flash = App::news_flash();
                let news_flash_guard = news_flash.read().await;
                let news_flash = news_flash_guard.as_ref()?;
                ArticleLoader::load(&article_id, news_flash)
            },
            clone!(
                #[weak(rename_to = this)]
                self,
                move |res: Option<ArticleLoader>| {
                    let Some(loader) = res else {
                        return;
                    };

                    let imp = this.imp();
                    let new_article = loader.build();

                    imp.article_view.set_article(Some(new_article));
                },
            ),
        );
    }

    pub fn refresh_article_metadata_from_db(&self) {
        let Some(article) = self.article() else {
            return;
        };

        let article_id: ArticleID = article.article_id().into();

        TokioRuntime::execute_with_callback(
            move || async move {
                let news_flash = App::news_flash();
                let news_flash_guard = news_flash.read().await;
                let news_flash = news_flash_guard.as_ref()?;
                news_flash.get_article(&article_id).ok()
            },
            move |res| {
                let Some(loaded_article) = res else {
                    return;
                };

                let read: GRead = loaded_article.unread.into();
                let marked: GMarked = loaded_article.marked.into();

                article.set_read(read);
                article.set_marked(marked);
            },
        );
    }

    pub fn popup_tag_popover(&self) {
        self.imp().tag_button.popup();
    }
}
