use crate::app::App;
use crate::content_page::ContentPage;
use crate::i18n::{i18n, i18n_f};
use crate::infrastructure::TokioRuntime;
use bytesize::ByteSize;
use gdk4::Texture;
use glib::{Object, Properties, SignalHandlerId, clone, prelude::*, subclass::*};
use gtk4::{Accessible, Buildable, CompositeTemplate, ConstraintTarget, ContentFit, Widget, subclass::prelude::*};
use libadwaita::{Bin, subclass::prelude::*};
use news_flash::models::Image;
use news_flash::{Progress, error::NewsFlashError, models::ArticleID};
use once_cell::sync::Lazy;
use std::cell::{Cell, RefCell};

mod imp {
    use super::*;

    #[derive(Debug, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::ImageWidget)]
    #[template(file = "data/resources/ui_templates/media/image_widget.blp")]
    pub struct ImageWidget {
        #[property(get, set, nullable)]
        pub texture: RefCell<Option<Texture>>,

        #[property(get, set, name = "progress-text")]
        pub progress_text: RefCell<String>,

        #[property(get, set, builder(ContentFit::Contain))]
        pub fit: Cell<ContentFit>,
    }

    impl Default for ImageWidget {
        fn default() -> Self {
            Self {
                texture: RefCell::new(None),
                progress_text: RefCell::new(String::new()),
                fit: Cell::new(ContentFit::Contain),
            }
        }
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ImageWidget {
        const NAME: &'static str = "ImageWidget";
        type ParentType = Bin;
        type Type = super::ImageWidget;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for ImageWidget {
        fn signals() -> &'static [Signal] {
            static SIGNALS: Lazy<Vec<Signal>> = Lazy::new(|| {
                vec![
                    Signal::builder("error").param_types([String::static_type()]).build(),
                    Signal::builder("tiny").build(),
                ]
            });
            SIGNALS.as_ref()
        }
    }

    impl WidgetImpl for ImageWidget {}

    impl BinImpl for ImageWidget {}

    #[gtk4::template_callbacks]
    impl ImageWidget {
        #[template_callback]
        fn visible_child_name(&self, texture: Option<Texture>) -> &'static str {
            if texture.is_some() { "picture" } else { "spinner" }
        }

        #[template_callback]
        fn is_progress_label_visible(&self, text: String) -> bool {
            !text.is_empty()
        }
    }
}

glib::wrapper! {
    pub struct ImageWidget(ObjectSubclass<imp::ImageWidget>)
        @extends Widget, Bin,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for ImageWidget {
    fn default() -> Self {
        Object::new()
    }
}

impl ImageWidget {
    pub fn load(&self, article_id: &ArticleID, url: &str) {
        let (sender, mut receiver) = tokio::sync::mpsc::channel::<Progress>(2);

        glib::MainContext::default().spawn_local(clone!(
            #[weak(rename_to = widget)]
            self,
            async move {
                while let Some(progress) = receiver.recv().await {
                    let downloaded = ByteSize(progress.downloaded as u64);
                    let total = ByteSize(progress.total_size as u64);
                    let progress_text = format!("{downloaded} of {total}");
                    widget.set_progress_text(progress_text);
                }
            }
        ));

        let url = url.replace("&amp;", "&").to_string();
        let article_id = article_id.clone();
        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                news_flash
                    .get_image(&article_id, url.as_str(), &App::client(), Some(sender))
                    .await
            },
            clone!(
                #[weak(rename_to = widget)]
                self,
                move |res: Result<Image, NewsFlashError>| match res {
                    Ok(image) => {
                        if let (Some(width), Some(height)) = (image.width, image.height)
                            && (width < 200 || height < 200)
                        {
                            widget.emmit_tiny();
                        }

                        let bytes = glib::Bytes::from_owned(image.data);
                        let texture = Texture::from_bytes(&bytes);

                        if let Err(error) = texture {
                            let message = i18n_f("Failed to decode image '{}'", &[&error.to_string()]);
                            widget.emmit_error(&message);
                            return;
                        };

                        widget.set_texture(texture.ok());
                    }
                    Err(error) => {
                        let message = i18n("Failed to download image");
                        ContentPage::instance().newsflash_error(&message, error);
                        widget.emmit_error(&message);
                    }
                }
            ),
        );
    }

    fn emmit_error(&self, error: &str) {
        self.emit_by_name::<()>("error", &[&error])
    }

    pub fn connect_error<F: Fn(String) + 'static>(&self, f: F) -> SignalHandlerId {
        self.connect_local("error", false, move |args| {
            let error = args[1].get::<String>().expect("The value needs to be of type `String`");
            f(error);
            None
        })
    }

    fn emmit_tiny(&self) {
        tracing::debug!("image is tiny");
        self.emit_by_name::<()>("tiny", &[])
    }

    pub fn connect_tiny<F: Fn() + 'static>(&self, f: F) -> SignalHandlerId {
        self.connect_local("tiny", false, move |_args| {
            f();
            None
        })
    }
}
