/*
 * RewriteContext.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2003 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: cstevens.
 * Portions created by cstevens are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, drach, guym, suhler.
 *
 * Version:  2.5
 * Created by cstevens on 99/09/29
 * Last modified by suhler on 03/08/01 16:19:12
 */

package sunlabs.brazil.template;

import java.util.Vector;
import sunlabs.brazil.handler.HtmlRewriter;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.Format;

/**
 * A variant containing instance variables that may be referenced by
 * rewriting filters.  Every implementation of the template class
 * may define methods of the form:
 * <code>tag_<i>xxx</i></code>
 * or
 * <code>tag_slash_<i>xxx</i></code>
 * which will get called when the corrosponding HTML entity
 * <code>&lt;xxx ...&gt;</code>
 * or
 * <code>&lt;/xxx ...&gt;</code>
 * is found in the content being filtered.
 * <p>
 * An instance of this class is passed to each <code>tag</code>
 * method, permitting introspection of the current filtering context.
 *
 * @version %V% RewriteContext.java
 */

public class RewriteContext extends HtmlRewriter {
    /**
     * The server object, as passed to a handler's init method.
     */
    public Server server;		// The server, containing properties.
    /**
     * The prefix to use for locating keys in the server or request properties
     * objects.  This is a dynamic value that changes during template
     * processing.  Before each template is processed, it is set to the
     * prefix associated with the template class that the current tag belongs
     * to.
     */
    public String prefix;
    /**
     * This is the prefix defined by the invoker of a set of templates.  Its
     * value corrosponds to the class that invokes a set of templates.
     */
    public String templatePrefix;
    /**
     * The Request object, as passed to a handler's respond method.
     */
    public Request request;		// The current request
    /**
     * A unique <b>session id</b>, if available.
     */
    public String sessionId;		// The session ID (if any)
    /**
     * Application specific placeholder.
     */
    Object[] args;			// app. specific extra stuff

    TemplateRunner runner;
    Vector templates;

    /* Private flags enumeration for RewriteContext state   */
    /* Note that the BSLTemplate has defined the 0,1,2 bits */
    /* for these flags  - Hex values 0x1, 0x2, 0x4          */

    private int stateFlags; 

    /* Private state variable to track the depth of our recursive descent */

    private int nestingLevel; 

    /* Our constructor */

    public
    RewriteContext(Server server, String prefix, Request request,
	    String content, String sessionId, TemplateRunner runner,
	    Vector templates) {
	super(content);

	this.server = server;
	this.prefix = prefix;
	this.templatePrefix = prefix;
	this.request = request;
	this.sessionId = sessionId;
	this.runner = runner;
	this.templates = templates;
	this.args = new Object[] {this};

        stateFlags = 0;
        nestingLevel = 0;
    }

    /**
     * Cause this RewriteContext to abort its processing.
     */

    public void abort() {
        this.lex.replace("");
        setRewriteState(BSLTemplate.ABORT);
    }

    /**
     * Set the given state of this RewriteContext.
     */
     
    public boolean
    setRewriteState(int state) {
        boolean nesting = (nestingLevel >= 1);

        if (nesting) {
          stateFlags |= state;  
        } 

        return nesting;
    }

    /**
     * Unset the given state of this RewriteContext.
     */
     
    public void 
    unsetRewriteState(int state) {
        stateFlags &= ~state;
    }

    /**
     * Check whether this RewriteContext is in the specified state.
     */

    public boolean
    checkRewriteState(int state) {
        return (stateFlags & state) != 0;
    }

    /**
     * Increment the nesting level counter.
     */

    public void 
    incrNestingLevel() {
        nestingLevel++;
    }

    /**
     * Decrement the nesting level counter.
     */

    public void 
    decrNestingLevel() {
        if (nestingLevel != 0) {
          nestingLevel--;
        } else {
          System.err.println("Someone tried to decrement the nesting level past zero!");
        }
    }

    /**
     * Return the current nesting level counter.
     */

    public int
    getNestingLevel() {
        return nestingLevel;
    }

    /**
     * Invoke a template on this token, if any template is interested in this
     * token.  The template may consume more than just this token, if it
     * wants.
     */

    public void
    process() {
	runner.process(this);
    }

    /* Convenience methods */

    /**
     * Return the number of HTML tags seen to this point.
     *
     */

    public int
    tagsSeen() {
	return runner.tagsSeen();
    }

    /**
     * overwrite "get" to automatically do ${...} substitutions
     * The default is "true".  Setting the default to "false" and
     * recompiling is more backward compatible (but less useful).
     */

    public String
    get(String name) {
	return get(name, true);
    }

    /**
     * Get an attribute value, and optionally perform ${...} substitutions.
     */

    public String
    get(String name, boolean subst) {
	if (subst) {
	    return Format.subst(request.props, super.get(name));
	} else {
	    return(super.get(name));
	}
    }

    /**
     * Get a tag attribute, with a default value.
     * The name is looked-up as an attribute in the current tag.  If
     * it is not found, the configuration property by the same name is used.
     * If that is not found, <code>dflt</code> is used instead.
     *
     * @param name	The name of the attribute to look up
     * @param dflt	The default attribute value (may be null)
     * @return		The value of "name"
     */

    public String
    get(String name, String dflt) {
	String result = get(name);
	if (result == null) {
	    result = request.props.getProperty(prefix + name, dflt);
	}
	return result;
    }

    /**
     * Determine the value of a boolean attribute in the current tag.
     * ${...} substitution is performed on the value.
     * @param name	The name of the  boolean attribute
     * @return          false if the value is:
     *                    null, "", "0", "no", "off", or "false"
     *                  true otherwise.
     *                  "attribute=" is false, but "attribute" with no
     *                  value is true;
     */

    public boolean
    isTrue(String name) {
        String value = get(name);
        if (value==null) {
            return false;
        }
        if (value.length() == 0) {
            return (getArgs().toLowerCase().indexOf(name + "=") == -1);
        }
        value = value.trim().toLowerCase();
	if (Format.isFalse(value)) {
            return false;
        } else {
            return true;
        }
    }

    /**
     * See if the current token is the closing tag for the given string.
     * Account for tag prefix, if any.
     * @param tag	tag whose match ( e.g. /[prefix]tag is to be found
     * @return		true if the current tag closes "tag"
     */

    public boolean isClosingFor(String tag) {
	return isClosingFor(tag, true);
    }
    /**
     * See if the current token is the closing tag for the given string.
     * Account for tag prefix, if any.
     * @param tag	tag whose match ( e.g. /[prefix]tag is to be found
     * @param close	if set, "tag" matches  "/[prefix]tag", otherwise
     *			tag matches "[prefix]tag".
     * @return		true if the current tag closes "tag"
     */

    public boolean isClosingFor(String tag, boolean close) {
	String currentTag = getTag();
	if (currentTag == null || (close && !currentTag.startsWith("/")) ||
		!currentTag.endsWith(tag)) {
	    return false;
	}
	return currentTag.equals((close ? "/" : "") + getTagPrefix() + tag);
    }

    /**
     * Add a closing tag to the list of tags that mark un-interpreted text
     * Deal with the tag prefix, if any
     */

    public void addClosingTag(String tag) {
	String add = getTagPrefix() + tag;
	if (lex.getClosingTags().indexOf(add) == -1) {
	    lex.getClosingTags().addElement(add);
	}
    }

    /**
     * get the tag prefix associated with the current tag.  
     */

    String getTagPrefix() {
	return server.props.getProperty(prefix + TemplateRunner.TAG_PREFIX,
		    server.props.getProperty(templatePrefix +
		    TemplateRunner.TAG_PREFIX, ""));
    }
}
