package org.apache.dvsl;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.context.Context;
import org.apache.velocity.runtime.log.LogChute;
import org.apache.velocity.runtime.log.LogChuteSystem;
import org.apache.velocity.runtime.log.LogSystem;

import org.dom4j.Document;
import org.dom4j.io.SAXReader;

/**
 *  Main DVSL class - use this as the helper class for apps
 *
 *  @author <a href="mailto:geirm@apache.org">Geir Magnusson Jr.</a>
 *  @author <a href="mailto:billb@progress.com">Bill Burton.</a>
 */
public class DVSL
{
    private static String TOOL_PROP_PREFIX = "toolbox.tool.";
    private static String STRING_PROP_PREFIX = "toolbox.string.";
    private static String INTEGER_PROP_PREFIX = "toolbox.integer.";
    private static String TOOLBOX_NAME = "toolbox.contextname.";

    private VelocityEngine ve = null;
    private Document currentDocument = null;
    private Writer currentWriter = null;
    private Context toolContext;
    private Context userContext;
    private Context styleContext;
    private DVSLContext baseContext = new DVSLContext();
    private Transformer transformer;
    private ClassLoader classLoader;
    private boolean ready = false;

    private Map velConfig = null;
    private File logFile;
    private LogChute logger;

    private Map appVals = new HashMap();

    private TemplateHandler templateHandler = new TemplateHandler();

    boolean validate = false;

    public DVSL()
    {
        classLoader = DVSL.class.getClassLoader();
    }

    /**
     *  <p>
     *  lets the user specify a filename for logging.
     *  </p>
     */
    public void setLogFile(File logFile)
    {
        this.logFile = logFile;

        if (velConfig == null)
        {
            velConfig = new HashMap();
        }

        velConfig.put(VelocityEngine.RUNTIME_LOG, logFile.getAbsolutePath());
    }

    /**
     *  <p>
     *  lets the user specify a class instance for logging.
     *  </p>
     * @deprecated use setLogChute instead
     */
    public void setLogSystem(LogSystem logger)
    {
        this.logger = new LogAdapter(logger);

        if (velConfig == null)
        {
            velConfig = new HashMap();
        }

        velConfig.put(VelocityEngine.RUNTIME_LOG_LOGSYSTEM, logger);
    }

    /**
     *  <p>
     *  lets the user specify a class instance for logging.
     *  </p>
     */
    public void setLogChute(LogChute logger)
    {
        this.logger = logger;

        if (velConfig == null)
        {
            velConfig = new HashMap();
        }

        velConfig.put(VelocityEngine.RUNTIME_LOG_LOGSYSTEM, logger);
    }


    /**
     *  <p>
     *  lets the user pass a java.util.Properties containing
     *  properties for the configuration of the VelocityEngine
     *  used by DVSL
     *  </p>
     */
    public void setVelocityConfig(Map map)
    {
        if (velConfig != null)
        {
            map.putAll(velConfig);
        }

        velConfig = map;
    }

    /**
     *  <p>
     *  Sets the user context.  The user context is
     *  a Velocity Context containing user-supplied
     *  objects and data that are to be made available
     *  in the template
     *  </p>
     *
     *  @param ctx User context of data
     */
    public void setUserContext(Context ctx)
    {
        ready = false;
        userContext = ctx;
    }

    /**
     * <p>
     *  Uses a validating parser on all input documents
     * </p>
     *
     * @param validate
     */

    public void setValidatingParser(boolean validate)
    {
        this.validate = validate;
    }

    /**
     *  <p>
     *  Specify a classloader for loading the Toolbox classes.  Setting to null
     *  resets to the default ClassLoader.
     *  </p>
     *
     *  @param classLoader ClassLoader or null for default ClassLoader
     */
    public void setClassLoader(ClassLoader classLoader)
    {
        if (classLoader == null)
        {
            this.classLoader = this.getClass().getClassLoader();
        }
        else
        {
            this.classLoader = classLoader;
        }
    }

    /**
     *  <p>
     *  Loads the toolbox from the input Properties.
     *  </p>
     *
     *  <p>
     *  Currently supports specification of the Toolbox
     *  name in the context, creating classes, and string
     *  and integer values.  Ex :
     *  </p>
     *
     *  <pre>
     *  toolbox.contextname = floyd
     *  toolbox.tool.footool = Footool
     *  toolbox.string.mystring = Hello there!
     *  toolbox.integer.myint = 7
     *  toolbox.string.sourcebase = ./xdocs/
     *  </pre>
     *
     *  <p>
     *  So in template, this toolbox and it's values would
     *  be accessed as :
     *  </p>
     *  <pre>
     *    $context.floyd.footool.getFoo()
     *    $context.floyd.mystring
     *    $context.floyd.myint
     *  </pre>
     */
    public void setToolbox(Properties p)
        throws ClassNotFoundException, InstantiationException, IllegalAccessException
    {
        ready = false;

        /*
         *  for each key that looks like
         *     toolbox.tool.<token> = class
         */

        Map toolbox = new HashMap();

        String toolboxname = "toolbox";

        for (Enumeration e = p.propertyNames(); e.hasMoreElements(); )
        {
            String key = (String) e.nextElement();

            String value = p.getProperty(key);

            if (key.startsWith(TOOL_PROP_PREFIX))
            {
                String toolname = key.substring(TOOL_PROP_PREFIX.length());

                Object o = Class.forName(value, true, classLoader).newInstance();

                toolbox.put(toolname, o);
            }
            else if (key.startsWith(INTEGER_PROP_PREFIX))
            {
                String toolname = key.substring(INTEGER_PROP_PREFIX.length());

                int i = 0;

                try
                {
                    i = Integer.parseInt(value);
                }
                catch(Exception ee)
                {
                }

                toolbox.put(toolname, new Integer(i));
            }
            else if (key.startsWith(STRING_PROP_PREFIX))
            {
                String toolname = key.substring(STRING_PROP_PREFIX.length());
                toolbox.put(toolname, value);
            }
            else if (key.startsWith(TOOLBOX_NAME))
            {
                toolboxname = value;
            }
        }

        toolContext = new VelocityContext();

        toolContext.put(toolboxname, toolbox);
    }

    /**
     *  Convenience function.  See...
     */
    public void setStylesheet(String stylesheet)
        throws Exception
    {
        setStylesheet(new File(stylesheet), null);
    }

    /**
     *  Convenience function.  See...
     */
    public void setStylesheet(File stylesheet)
        throws Exception
    {
        setStylesheet(stylesheet, null);
    }

    /**
     *  Convenience function.  See...
     */
    public void setStylesheet(File stylesheet, String stylesheetEncoding)
        throws Exception
    {
        Reader fr = null;

        try
        {
            if (stylesheetEncoding != null)
            {
                fr = new InputStreamReader(
                    new FileInputStream(stylesheet), stylesheetEncoding);
            }
            else
            {
                fr = new FileReader(stylesheet);
            }

            setStylesheet(fr);
        }
        finally
        {
            if (fr != null)
                fr.close();
        }
    }


    /**
     *  <p>
     *  Sets the stylesheet for this transformation set
     *  </p>
     *
     *  <p>
     *  Note that don't need this for each document you want
     *  to transform.  Just do it once, and transform away...
     *  </p>
     *
     *  @param styleReader Reader with stylesheet char stream
     */
    public void setStylesheet(Reader styleReader)
        throws Exception
    {
        ready = false;
        /*
         *  now initialize Velocity - we need to do that
         *  on change of stylesheet
         */

        ve = new VelocityEngine();

        /*
         * if there are user properties, set those first - carefully
         */

        if (velConfig != null)
        {
            configureVelocityEngine(ve, velConfig);
        }

        /*
         *  register our template() directive
         */

        ve.setProperty("userdirective", "org.apache.dvsl.directive.MatchDirective");
        ve.init();

        /*
         *  add our template accumulator
         */

        ve.setApplicationAttribute("org.apache.dvsl.TemplateHandler", templateHandler);

        /*
         *  load and render the stylesheet
         *
         *  this sets stylesheet specific context
         *  values
         */

        StringWriter junkWriter = new StringWriter();

        styleContext = new VelocityContext();
        ve.evaluate(styleContext, junkWriter, "DVSL:stylesheet", styleReader);

        /*
         *  now run the base template through for the rules
         */

        InputStream is = this.getClass().getClassLoader()
              .getResourceAsStream("org/apache/dvsl/resource/defaultroot.dvsl");

        if (is == null)
        {
            System.out.println("DEFAULT TRANSFORM RULES NOT FOUND ");
        }
        else
        {
            ve.evaluate(new VelocityContext(),
                    junkWriter, "defaultroot.dvsl", new InputStreamReader(is));
            is.close();
        }

        /*
         *  need a new transformer, as it depends on the
         *  velocity engine
         */

        transformer = new Transformer(ve, templateHandler,  baseContext, appVals,
                            validate);
    }

    /**
     *  <p>
     *   Adds the allowed properties from the Properties to the
     *   velocity engine
     *  </p>
     *  <p>
     *  So far, we support, in RuntimeConstant parlance :
     *  </p>
     *  <ul>
     *  <li>VM_LIBRARY</li>
     *  <li>FILE_RESOURCE_LOADER_PATH</li>
     *  <li>RUNTIME_LOG</li>
     *  <li>RUNTIME_LOG_LOGSYSTEM</li>
     *  <li>RUNTIME_LOG_LOGSYSTEM_CLASS</li>
     *  </ul>
     *
     *  <p>
     *  If you are going to use this, ensure you do it *before* setting
     *  the stylesheet, as that creates the VelocityEngine
     *  </p>
     */
    private void configureVelocityEngine(VelocityEngine ve, Map map)
    {
        if (ve == null || map == null)
        {
            return;
        }

        /*
         * for now,  keep it simple
         */

        Object val = map.get(VelocityEngine.VM_LIBRARY);

        if (val instanceof String)
        {
            ve.setProperty(VelocityEngine.VM_LIBRARY, (String) val);
        }

        /*
         *  assumes that for now, we are using the file loader
         */
        val = map.get(VelocityEngine.FILE_RESOURCE_LOADER_PATH);

        if (val instanceof String)
        {
            ve.setProperty(VelocityEngine.FILE_RESOURCE_LOADER_PATH, (String) val);
        }

        /*
         *  No real assumptions - pass what you want
         */
        val = map.get(VelocityEngine.RUNTIME_LOG);

        if (val instanceof String)
        {
            ve.setProperty(VelocityEngine.RUNTIME_LOG,  val);
        }

        val = map.get(VelocityEngine.RUNTIME_LOG_LOGSYSTEM);

        if (val != null)
        {
            ve.setProperty(VelocityEngine.RUNTIME_LOG_LOGSYSTEM,  val);
        }

        val = map.get(VelocityEngine.RUNTIME_LOG_LOGSYSTEM_CLASS);

        if (val instanceof String)
        {
            ve.setProperty(VelocityEngine.RUNTIME_LOG_LOGSYSTEM_CLASS,  val);
        }

    }

    /**
     *  sets up all the context goodies
     */
    protected void makeReady()
    {
        /*
         *  put all the contexts together
         */

        baseContext.clearContexts();

        baseContext.addContext(userContext);
        baseContext.addContext(toolContext);
        baseContext.setStyleContext(styleContext);

        ready = true;
     }

    /**
     *  does the transformation of the inputstream into
     *  the output writer
     */
    protected long xform(Reader reader, Writer writer)
        throws Exception
    {
        if (!ready)
            makeReady();

        return transformer.transform(reader, writer);
    }

    protected long xform(Document dom4jdoc, Writer writer)
        throws Exception
    {
        if (!ready)
            makeReady();

        return transformer.transform(dom4jdoc, writer);
    }

    public long transform(File f, Writer writer)
        throws Exception
    {
        InputStream is = null;
        try
        {
            is = new FileInputStream(f);
            return transform(is, writer);
        }
        finally
        {
            if (is != null)
            {
                is.close();
            }
        }
    }

    public long transform(Reader reader, Writer writer)
        throws Exception
    {
        return xform(reader, writer);
    }

    public long transform(InputStream is, Writer writer)
        throws Exception
    {
        SAXReader reader = new SAXReader();
        return xform(reader.read(is), writer);
    }

    /**
     * Transforms the given dom4j Document into the writer.
     *
     * @param dom4jdoc dom4j Document object
     * @param writer Writer for output
     */
    public long transform(Document dom4jdoc, Writer writer)
        throws Exception
    {
        return xform(dom4jdoc, writer);
    }

    public long transform(String infile, Writer writer)
        throws Exception
    {
        return transform(new File(infile), writer);
    }


    /**
     *  Gets the application value for the specified key
     *
     *  @param key key to use to retrieve value
     *  @return value if found, null otherwise
     */
    public Object getAppValue(Object key)
    {
        return appVals.get(key);
    }

    /**
     *  Sets the application value for the specified key
     *
     *  @param key key to use to store value
     *  @param value value to be stored
     *  @return old value if any, null otherwise
     */
    public Object putAppValue(Object key, Object value)
    {
        return appVals.put(key, value);
    }

    /**
     *  <p>
     *  Allows command-line access.
     *  </p>
     *  <p>
     *  Usage :  java -jar dvsl.jar -STYLE stylesheeet [-IN infile] [-OUT outfile] [-TOOL toolboxname]
     *  </p>
     */
    public static void main(String[] args)
        throws Exception
    {

        DVSL dvsl = new DVSL();

        Reader in = new InputStreamReader(System.in);
        String infile = null;
        String style = null;
        String outfile = null;

        Writer out = new OutputStreamWriter(System.out);
        String toolfile = null;

        for(int i = 0; i < args.length; i++)
        {
            if (args[i].equals("-IN"))
                infile = args[++i];
            else if (args[i].equals("-OUT"))
                outfile = args[++i];
           else if (args[i].equals("-STYLE"))
                style = args[++i];
           else if (args[i].equals("-TOOL"))
                toolfile = args[++i];
        }

        if (style == null)
        {
            System.out.println("usage :need to specify a stylesheet. ");
            System.out.println("java -jar dvsl.jar -STYLE stylesheeet [-IN infile] [-OUT outfile] [-TOOL toolboxname]");
            return;
        }

        if (style != null)
            dvsl.setStylesheet(style);

        if (toolfile != null)
        {
            Properties p = new Properties();

            InputStream fis = new FileInputStream(toolfile);

            p.load(fis);

            dvsl.setToolbox(p);
        }

        if (infile != null)
            in = new FileReader(infile);

        if (outfile != null)
            out = new FileWriter(outfile);

        long time = dvsl.transform(in, out);

        out.flush();

    }

    /*
     * inner class to wrap a LogSystem into a LogChute
     */
    protected static class LogAdapter extends LogChuteSystem {
        protected LogAdapter(LogSystem logSystem) {
            super(logSystem);
        }
    }
}

