/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.seata.server.session;

import org.apache.seata.common.store.SessionMode;
import org.apache.seata.core.exception.TransactionException;
import org.apache.seata.core.model.BranchStatus;
import org.apache.seata.core.model.BranchType;
import org.apache.seata.core.model.GlobalStatus;
import org.apache.seata.server.BaseSpringBootTest;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.springframework.context.ApplicationContext;

import java.io.IOException;
import java.util.stream.Stream;

import static org.apache.seata.common.DefaultValues.DEFAULT_TX_GROUP;

/**
 * The type Global session test.
 *
 * @since 2019 /1/23
 */
public class GlobalSessionTest extends BaseSpringBootTest {

    @BeforeAll
    public static void init(ApplicationContext context) {
        SessionHolder.init(SessionMode.FILE);
    }

    @AfterAll
    public static void destroy() {
        SessionHolder.destroy();
    }

    /**
     * Can be committed async test.
     *
     * @param globalSession the global session
     */
    @ParameterizedTest
    @MethodSource("branchSessionTCCProvider")
    public void canBeCommittedAsyncTest(GlobalSession globalSession) {
        Assertions.assertFalse(globalSession.canBeCommittedAsync());
    }

    /**
     * Begin test.
     *
     * @param globalSession the global session
     * @throws Exception the exception
     */
    @ParameterizedTest
    @MethodSource("globalSessionProvider")
    public void beginTest(GlobalSession globalSession) throws Exception {
        globalSession.begin();
    }

    /**
     * Begin test.
     *
     * @param globalSession the global session
     * @throws Exception the exception
     */
    @ParameterizedTest
    @MethodSource("globalSessionProvider")
    public void checkSizeTest(GlobalSession globalSession) throws Exception {
        Assertions.assertDoesNotThrow(globalSession::checkSize);
        int size = 520;
        StringBuilder sb = new StringBuilder(size);
        for (int i = 0; i < size; i++) {
            sb.append('a');
        }
        String str = sb.toString();
        globalSession.setApplicationData(str);
        Assertions.assertThrows(TransactionException.class, globalSession::checkSize);
        globalSession.setApplicationData(null);
        globalSession.setXid(str);
        Assertions.assertThrows(TransactionException.class, globalSession::checkSize);
        GlobalSession globalSession1 = new GlobalSession(null, str, null, 60000, true);
        Assertions.assertThrows(TransactionException.class, globalSession1::checkSize);
    }

    /**
     * Change status test.
     *
     * @param globalSession the global session
     * @throws Exception the exception
     */
    @ParameterizedTest
    @MethodSource("globalSessionProvider")
    public void changeStatusTest(GlobalSession globalSession) throws Exception {
        globalSession.changeGlobalStatus(GlobalStatus.Committed);
    }

    /**
     * Change branch status test.
     *
     * @param globalSession the global session
     * @param branchSession the branch session
     * @throws Exception the exception
     */
    @ParameterizedTest
    @MethodSource("branchSessionProvider")
    public void changeBranchStatusTest(GlobalSession globalSession, BranchSession branchSession) throws Exception {
        globalSession.changeBranchStatus(branchSession, BranchStatus.PhaseTwo_Committed);
    }

    /**
     * Close test.
     *
     * @param globalSession the global session
     * @throws Exception the exception
     */
    @ParameterizedTest
    @MethodSource("globalSessionProvider")
    public void closeTest(GlobalSession globalSession) throws Exception {
        globalSession.close();
    }

    /**
     * End test.
     *
     * @param globalSession the global session
     * @throws Exception the exception
     */
    @ParameterizedTest
    @MethodSource("globalSessionProvider")
    public void endTest(GlobalSession globalSession) throws Exception {
        globalSession.end();
    }

    /**
     * Add branch test.
     *
     * @param globalSession the global session
     * @param branchSession the branch session
     * @throws Exception the exception
     */
    @ParameterizedTest
    @MethodSource("branchSessionProvider")
    public void addBranchTest(GlobalSession globalSession, BranchSession branchSession) throws Exception {
        globalSession.addBranch(branchSession);
    }

    /**
     * Remove branch test.
     *
     * @param globalSession the global session
     * @param branchSession the branch session
     * @throws Exception the exception
     */
    @ParameterizedTest
    @MethodSource("branchSessionProvider")
    public void removeBranchTest(GlobalSession globalSession, BranchSession branchSession) throws Exception {
        globalSession.addBranch(branchSession);
        globalSession.removeAndUnlockBranch(branchSession);
    }

    /**
     * Codec test.
     *
     * @param globalSession the global session
     */
    @ParameterizedTest
    @MethodSource("globalSessionProvider")
    public void codecTest(GlobalSession globalSession) {
        byte[] result = globalSession.encode();
        Assertions.assertNotNull(result);
        GlobalSession expected = new GlobalSession();
        expected.decode(result);
        Assertions.assertEquals(expected.getTransactionId(), globalSession.getTransactionId());
        Assertions.assertEquals(expected.getTimeout(), globalSession.getTimeout());
        Assertions.assertEquals(expected.getApplicationId(), globalSession.getApplicationId());
        Assertions.assertEquals(expected.getTransactionServiceGroup(), globalSession.getTransactionServiceGroup());
        Assertions.assertEquals(expected.getTransactionName(), globalSession.getTransactionName());
        Assertions.assertTrue(expected.isActive());
    }

    /**
     * Global session provider object [ ] [ ].
     *
     * @return the object [ ] [ ]
     */
    static Stream<Arguments> globalSessionProvider() throws IOException {
        GlobalSession globalSession = new GlobalSession("demo-app", DEFAULT_TX_GROUP, "test", 6000);
        globalSession.setActive(true);
        return Stream.of(Arguments.of(globalSession));
    }

    /**
     * Branch session provider object [ ] [ ].
     *
     * @return the object [ ] [ ]
     */
    static Stream<Arguments> branchSessionProvider() {
        GlobalSession globalSession = new GlobalSession("demo-app", DEFAULT_TX_GROUP, "test", 6000);
        BranchSession branchSession = new BranchSession();
        branchSession.setXid(globalSession.getXid());
        branchSession.setTransactionId(globalSession.getTransactionId());
        branchSession.setBranchId(1L);
        branchSession.setResourceGroupId(DEFAULT_TX_GROUP);
        branchSession.setResourceId("tb_1");
        branchSession.setLockKey("t_1");
        branchSession.setBranchType(BranchType.AT);
        branchSession.setApplicationData("{\"data\":\"test\"}");
        globalSession.add(branchSession);
        return Stream.of(Arguments.of(globalSession, branchSession));
    }

    /**
     * Branch session mt provider object [ ] [ ].
     *
     * @return the object [ ] [ ]
     */
    static Stream<Arguments> branchSessionTCCProvider() {
        GlobalSession globalSession = new GlobalSession("demo-app", DEFAULT_TX_GROUP, "test", 6000);
        BranchSession branchSession = new BranchSession();
        branchSession.setXid(globalSession.getXid());
        branchSession.setTransactionId(globalSession.getTransactionId());
        branchSession.setBranchId(1L);
        branchSession.setResourceGroupId(DEFAULT_TX_GROUP);
        branchSession.setResourceId("tb_1");
        branchSession.setLockKey("t_1");
        branchSession.setBranchType(BranchType.TCC);
        branchSession.setApplicationData("{\"data\":\"test\"}");
        globalSession.add(branchSession);
        return Stream.of(Arguments.of(globalSession));
    }
}
