/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.cxf.systest.sts.secure_conv;

import java.net.URL;

import javax.xml.namespace.QName;
import javax.xml.ws.Service;

import org.apache.cxf.systest.sts.deployment.DoubleItServer;
import org.apache.cxf.systest.sts.deployment.STSServer;
import org.apache.cxf.testutil.common.AbstractBusClientServerTestBase;
import org.example.contract.doubleit.DoubleItPortType;

import org.junit.BeforeClass;

import static org.junit.Assert.assertTrue;

/**
 * In this test case, a CXF client requests a SecurityContextToken from an STS, and then uses the
 * corresponding secret to secure a service request. The service endpoint must contact the STS to validate
 * the received SCT and get the secret required to decrypt/verify the client request (via a SAML2 Assertion).
 */
public class SecureConversationTest extends AbstractBusClientServerTestBase {

    static final String STSPORT = allocatePort(STSServer.class);

    private static final String NAMESPACE = "http://www.example.org/contract/DoubleIt";
    private static final QName SERVICE_QNAME = new QName(NAMESPACE, "DoubleItService");

    private static final String PORT = allocatePort(DoubleItServer.class);
    private static final String PORT2 = allocatePort(DoubleItServer.class, 2);

    @BeforeClass
    public static void startServers() throws Exception {
        assertTrue(launchServer(new DoubleItServer(
            SecureConversationTest.class.getResource("cxf-service.xml")
        )));
        assertTrue(launchServer(new STSServer(
            SecureConversationTest.class.getResource("cxf-sts.xml"))));
    }

    @org.junit.Test
    public void testSecureConversation() throws Exception {
        createBus(getClass().getResource("cxf-client.xml").toString());

        URL wsdl = SecureConversationTest.class.getResource("DoubleIt.wsdl");
        Service service = Service.create(wsdl, SERVICE_QNAME);
        QName portQName = new QName(NAMESPACE, "DoubleItTransportSecureConvPort");
        DoubleItPortType transportPort =
            service.getPort(portQName, DoubleItPortType.class);
        updateAddressPort(transportPort, PORT);

        doubleIt(transportPort, 25);
    }

    @org.junit.Test
    public void testSecureConversationSymmetric() throws Exception {
        createBus(getClass().getResource("cxf-client.xml").toString());

        URL wsdl = SecureConversationTest.class.getResource("DoubleIt.wsdl");
        Service service = Service.create(wsdl, SERVICE_QNAME);
        QName portQName = new QName(NAMESPACE, "DoubleItSymmetricSecureConvPort");
        DoubleItPortType symmetricPort =
            service.getPort(portQName, DoubleItPortType.class);
        updateAddressPort(symmetricPort, PORT2);

        doubleIt(symmetricPort, 30);
    }

    @org.junit.Test
    public void testSecureConversationSupporting() throws Exception {
        createBus(getClass().getResource("cxf-client.xml").toString());

        URL wsdl = SecureConversationTest.class.getResource("DoubleIt.wsdl");
        Service service = Service.create(wsdl, SERVICE_QNAME);
        QName portQName = new QName(NAMESPACE, "DoubleItTransportSupportingPort");
        DoubleItPortType transportPort =
            service.getPort(portQName, DoubleItPortType.class);
        updateAddressPort(transportPort, PORT);

        doubleIt(transportPort, 25);
    }

    private static void doubleIt(DoubleItPortType port, int numToDouble) {
        int resp = port.doubleIt(numToDouble);
        assertTrue(resp == 2 * numToDouble);
    }

}
