/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable.indexsummary;

import com.google.common.annotations.VisibleForTesting;
import java.io.DataInput;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.PartitionPosition;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.Downsampling;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.util.DataInputPlus;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.Memory;
import org.apache.cassandra.io.util.MemoryOutputStream;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.concurrent.Ref;
import org.apache.cassandra.utils.concurrent.WrappedSharedCloseable;
import org.apache.cassandra.utils.memory.MemoryUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexSummary
extends WrappedSharedCloseable {
    private static final Logger logger = LoggerFactory.getLogger(IndexSummary.class);
    public static final IndexSummarySerializer serializer = new IndexSummarySerializer();
    private final int minIndexInterval;
    private final IPartitioner partitioner;
    private final int sizeAtFullSampling;
    private final Memory offsets;
    private final int offsetCount;
    private final Memory entries;
    private final long entriesLength;
    private final int samplingLevel;

    public IndexSummary(IPartitioner partitioner, Memory offsets, int offsetCount, Memory entries, long entriesLength, int sizeAtFullSampling, int minIndexInterval, int samplingLevel) {
        super(new Memory[]{offsets, entries});
        assert (offsets.getInt(0L) == 0);
        this.partitioner = partitioner;
        this.minIndexInterval = minIndexInterval;
        this.offsetCount = offsetCount;
        this.entriesLength = entriesLength;
        this.sizeAtFullSampling = sizeAtFullSampling;
        this.offsets = offsets;
        this.entries = entries;
        this.samplingLevel = samplingLevel;
        assert (samplingLevel > 0);
    }

    private IndexSummary(IndexSummary copy) {
        super(copy);
        this.partitioner = copy.partitioner;
        this.minIndexInterval = copy.minIndexInterval;
        this.offsetCount = copy.offsetCount;
        this.entriesLength = copy.entriesLength;
        this.sizeAtFullSampling = copy.sizeAtFullSampling;
        this.offsets = copy.offsets;
        this.entries = copy.entries;
        this.samplingLevel = copy.samplingLevel;
    }

    public int binarySearch(PartitionPosition key) {
        ByteBuffer hollow = MemoryUtil.getHollowDirectByteBuffer().order(ByteOrder.BIG_ENDIAN);
        int low = 0;
        int mid = this.offsetCount;
        int high = mid - 1;
        int result = -1;
        while (low <= high) {
            mid = low + high >> 1;
            this.fillTemporaryKey(mid, hollow);
            result = -DecoratedKey.compareTo(this.partitioner, hollow, key);
            if (result > 0) {
                low = mid + 1;
                continue;
            }
            if (result == 0) {
                return mid;
            }
            high = mid - 1;
        }
        return -mid - (result < 0 ? 1 : 2);
    }

    public int getPositionInSummary(int index) {
        return this.offsets.getInt(index << 2);
    }

    public byte[] getKey(int index) {
        long start = this.getPositionInSummary(index);
        int keySize = (int)(this.calculateEnd(index) - start - 8L);
        byte[] key = new byte[keySize];
        this.entries.getBytes(start, key, 0, keySize);
        return key;
    }

    private void fillTemporaryKey(int index, ByteBuffer buffer) {
        long start = this.getPositionInSummary(index);
        int keySize = (int)(this.calculateEnd(index) - start - 8L);
        this.entries.setByteBuffer(buffer, start, keySize);
    }

    @Override
    public void addTo(Ref.IdentityCollection identities) {
        super.addTo(identities);
        identities.add(this.offsets);
        identities.add(this.entries);
    }

    public long getPosition(int index) {
        return this.entries.getLong(this.calculateEnd(index) - 8L);
    }

    public long getEndInSummary(int index) {
        return this.calculateEnd(index);
    }

    private long calculateEnd(int index) {
        return index == this.offsetCount - 1 ? this.entriesLength : (long)this.getPositionInSummary(index + 1);
    }

    public int getMinIndexInterval() {
        return this.minIndexInterval;
    }

    public double getEffectiveIndexInterval() {
        return 128.0 / (double)this.samplingLevel * (double)this.minIndexInterval;
    }

    public long getEstimatedKeyCount() {
        return ((long)this.getMaxNumberOfEntries() + 1L) * (long)this.minIndexInterval;
    }

    public int size() {
        return this.offsetCount;
    }

    public int getSamplingLevel() {
        return this.samplingLevel;
    }

    public int getMaxNumberOfEntries() {
        return this.sizeAtFullSampling;
    }

    long getEntriesLength() {
        return this.entriesLength;
    }

    Memory getOffsets() {
        return this.offsets;
    }

    Memory getEntries() {
        return this.entries;
    }

    public long getOffHeapSize() {
        return (long)(this.offsetCount * 4) + this.entriesLength;
    }

    public int getEffectiveIndexIntervalAfterIndex(int index) {
        return Downsampling.getEffectiveIndexIntervalAfterIndex(index, this.samplingLevel, this.minIndexInterval);
    }

    public List<SSTableReader.IndexesBounds> getSampleIndexesForRanges(Collection<Range<Token>> ranges) {
        ArrayList<SSTableReader.IndexesBounds> positions = new ArrayList<SSTableReader.IndexesBounds>();
        for (Range range : Range.normalize(ranges)) {
            int right;
            Token.KeyBound leftPosition = ((Token)range.left).maxKeyBound();
            Token.KeyBound rightPosition = ((Token)range.right).maxKeyBound();
            int left = this.binarySearch(leftPosition);
            left = left < 0 ? (left + 1) * -1 : ++left;
            if (left == this.size()) continue;
            int n = right = Range.isWrapAround((Token)range.left, (Token)range.right) ? this.size() - 1 : this.binarySearch(rightPosition);
            if (right < 0) {
                if ((right = (right + 1) * -1) == 0) continue;
                --right;
            }
            if (left > right) continue;
            positions.add(new SSTableReader.IndexesBounds(left, right));
        }
        return positions;
    }

    public Iterable<byte[]> getKeySamples(Range<Token> range) {
        final List<SSTableReader.IndexesBounds> indexRanges = this.getSampleIndexesForRanges(Collections.singletonList(range));
        if (indexRanges.isEmpty()) {
            return Collections.emptyList();
        }
        return () -> new Iterator<byte[]>(){
            private SSTableReader.IndexesBounds current;
            private int idx;
            private Iterator rangeIter;
            {
                this.rangeIter = indexRanges.iterator();
            }

            @Override
            public boolean hasNext() {
                if (this.current == null || this.idx > this.current.upperPosition) {
                    if (this.rangeIter.hasNext()) {
                        this.current = (SSTableReader.IndexesBounds)this.rangeIter.next();
                        this.idx = this.current.lowerPosition;
                        return true;
                    }
                    return false;
                }
                return true;
            }

            @Override
            public byte[] next() {
                return IndexSummary.this.getKey(this.idx++);
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    public long getScanPosition(PartitionPosition key) {
        return this.getScanPositionFromBinarySearchResult(this.binarySearch(key));
    }

    @VisibleForTesting
    public long getScanPositionFromBinarySearchResult(int binarySearchResult) {
        if (binarySearchResult == -1) {
            return 0L;
        }
        return this.getPosition(IndexSummary.getIndexFromBinarySearchResult(binarySearchResult));
    }

    public static int getIndexFromBinarySearchResult(int binarySearchResult) {
        if (binarySearchResult < 0) {
            int greaterThan = (binarySearchResult + 1) * -1;
            if (greaterThan == 0) {
                return -1;
            }
            return greaterThan - 1;
        }
        return binarySearchResult;
    }

    @Override
    public IndexSummary sharedCopy() {
        return new IndexSummary(this);
    }

    public static class IndexSummarySerializer {
        public void serialize(IndexSummary t2, DataOutputPlus out) throws IOException {
            out.writeInt(t2.minIndexInterval);
            out.writeInt(t2.offsetCount);
            out.writeLong(t2.getOffHeapSize());
            out.writeInt(t2.samplingLevel);
            out.writeInt(t2.sizeAtFullSampling);
            int baseOffset = t2.offsetCount * 4;
            for (int i = 0; i < t2.offsetCount; ++i) {
                int offset = t2.offsets.getInt(i * 4) + baseOffset;
                offset = Integer.reverseBytes(offset);
                out.writeInt(offset);
            }
            out.write(t2.entries, 0L, t2.entriesLength);
        }

        public <T extends InputStream> IndexSummary deserialize(T in, IPartitioner partitioner, int expectedMinIndexInterval, int maxIndexInterval) throws IOException {
            int offset;
            int offsetReversed;
            int minIndexInterval = ((DataInput)((Object)in)).readInt();
            if (minIndexInterval != expectedMinIndexInterval) {
                throw new IOException(String.format("Cannot read index summary because min_index_interval changed from %d to %d.", minIndexInterval, expectedMinIndexInterval));
            }
            int offsetCount = ((DataInput)((Object)in)).readInt();
            long offheapSize = ((DataInput)((Object)in)).readLong();
            int samplingLevel = ((DataInput)((Object)in)).readInt();
            int fullSamplingSummarySize = ((DataInput)((Object)in)).readInt();
            int effectiveIndexInterval = (int)Math.ceil(128.0 / (double)samplingLevel * (double)minIndexInterval);
            if (effectiveIndexInterval > maxIndexInterval) {
                throw new IOException(String.format("Rebuilding index summary because the effective index interval (%d) is higher than the current max index interval (%d)", effectiveIndexInterval, maxIndexInterval));
            }
            Memory offsets = Memory.allocate(offsetCount * 4);
            Memory entries = Memory.allocate(offheapSize - offsets.size());
            try {
                FBUtilities.copy(in, new MemoryOutputStream(offsets), offsets.size());
                FBUtilities.copy(in, new MemoryOutputStream(entries), entries.size());
            }
            catch (IOException ioe) {
                offsets.free();
                entries.free();
                throw ioe;
            }
            if (offsets.size() > 0L && ((offsetReversed = Integer.reverseBytes(offset = offsets.getInt(0L))) > 0 && offset > offsetReversed || (long)offset - offsets.size() < 0L)) {
                throw new IOException(String.format("Rebuilding index summary because offset value (%d) at position: %d is Big Endian while Little Endian is expected", offset, 0));
            }
            int i = 0;
            while ((long)i < offsets.size()) {
                offsets.setInt(i, (int)((long)offsets.getInt(i) - offsets.size()));
                i += 4;
            }
            return new IndexSummary(partitioner, offsets, offsetCount, entries, entries.size(), fullSamplingSummarySize, minIndexInterval, samplingLevel);
        }

        public Pair<DecoratedKey, DecoratedKey> deserializeFirstLastKey(DataInputPlus.DataInputStreamPlus in, IPartitioner partitioner) throws IOException {
            in.skipBytes(4);
            int offsetCount = in.readInt();
            long offheapSize = in.readLong();
            in.skipBytes(8);
            in.skipBytes(offsetCount * 4);
            in.skipBytes((int)(offheapSize - (long)(offsetCount * 4)));
            DecoratedKey first = partitioner.decorateKey(ByteBufferUtil.readWithLength(in));
            DecoratedKey last = partitioner.decorateKey(ByteBufferUtil.readWithLength(in));
            return Pair.create(first, last);
        }
    }
}

