/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.config;

import com.google.common.primitives.Ints;
import java.util.Arrays;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public abstract class DataStorageSpec {
    private static final Pattern UNITS_PATTERN = Pattern.compile("^(\\d+)(GiB|MiB|KiB|B)$");
    private final long quantity;
    private final DataStorageUnit unit;

    private DataStorageSpec(long quantity, DataStorageUnit unit, DataStorageUnit minUnit, long max, String value) {
        this.quantity = quantity;
        this.unit = unit;
        DataStorageSpec.validateMinUnit(unit, minUnit, value);
        DataStorageSpec.validateQuantity(quantity, unit, minUnit, max);
    }

    private DataStorageSpec(String value, DataStorageUnit minUnit) {
        Matcher matcher = UNITS_PATTERN.matcher(value);
        if (!matcher.find()) {
            throw new IllegalArgumentException("Invalid data storage: " + value + " Accepted units:" + DataStorageSpec.acceptedUnits(minUnit) + " where case matters and only non-negative values are accepted");
        }
        this.quantity = Long.parseLong(matcher.group(1));
        this.unit = DataStorageUnit.fromSymbol(matcher.group(2));
    }

    private DataStorageSpec(String value, DataStorageUnit minUnit, long max) {
        this(value, minUnit);
        DataStorageSpec.validateMinUnit(this.unit, minUnit, value);
        DataStorageSpec.validateQuantity(value, this.quantity(), this.unit(), minUnit, max);
    }

    private static void validateMinUnit(DataStorageUnit sourceUnit, DataStorageUnit minUnit, String value) {
        if (sourceUnit.compareTo(minUnit) < 0) {
            throw new IllegalArgumentException(String.format("Invalid data storage: %s Accepted units:%s", value, DataStorageSpec.acceptedUnits(minUnit)));
        }
    }

    private static String acceptedUnits(DataStorageUnit minUnit) {
        DataStorageUnit[] units = DataStorageUnit.values();
        return Arrays.toString((Object[])Arrays.copyOfRange(units, minUnit.ordinal(), units.length));
    }

    private static void validateQuantity(String value, long quantity, DataStorageUnit sourceUnit, DataStorageUnit minUnit, long max) {
        if (minUnit.convert(quantity, sourceUnit) >= max) {
            throw new IllegalArgumentException("Invalid data storage: " + value + ". It shouldn't be more than " + (max - 1L) + " in " + minUnit.name().toLowerCase());
        }
    }

    private static void validateQuantity(long quantity, DataStorageUnit sourceUnit, DataStorageUnit minUnit, long max) {
        if (quantity < 0L) {
            throw new IllegalArgumentException("Invalid data storage: value must be non-negative");
        }
        if (minUnit.convert(quantity, sourceUnit) >= max) {
            throw new IllegalArgumentException(String.format("Invalid data storage: %d %s. It shouldn't be more than %d in %s", quantity, sourceUnit.name().toLowerCase(), max - 1L, minUnit.name().toLowerCase()));
        }
    }

    public long quantity() {
        return this.quantity;
    }

    public DataStorageUnit unit() {
        return this.unit;
    }

    public int hashCode() {
        return Objects.hash(this.unit.toKibibytes(this.quantity));
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof DataStorageSpec)) {
            return false;
        }
        DataStorageSpec other = (DataStorageSpec)obj;
        if (this.unit == other.unit) {
            return this.quantity == other.quantity;
        }
        return this.unit.convert(other.quantity, other.unit) == this.quantity && other.unit.convert(this.quantity, this.unit) == other.quantity;
    }

    public String toString() {
        return this.quantity + this.unit.symbol;
    }

    public static class DataStorageUnit
    extends Enum<DataStorageUnit> {
        public static final /* enum */ DataStorageUnit BYTES = new DataStorageUnit("B"){

            @Override
            public long toBytes(long d) {
                return d;
            }

            @Override
            public long toKibibytes(long d) {
                return d / 1024L;
            }

            @Override
            public long toMebibytes(long d) {
                return d / 0x100000L;
            }

            @Override
            public long toGibibytes(long d) {
                return d / 0x40000000L;
            }

            @Override
            public long convert(long source, DataStorageUnit sourceUnit) {
                return sourceUnit.toBytes(source);
            }
        };
        public static final /* enum */ DataStorageUnit KIBIBYTES = new DataStorageUnit("KiB"){

            @Override
            public long toBytes(long d) {
                return 2.x(d, 1024L, 0x1FFFFFFFFFFFFFL);
            }

            @Override
            public long toKibibytes(long d) {
                return d;
            }

            @Override
            public long toMebibytes(long d) {
                return d / 1024L;
            }

            @Override
            public long toGibibytes(long d) {
                return d / 0x100000L;
            }

            @Override
            public long convert(long source, DataStorageUnit sourceUnit) {
                return sourceUnit.toKibibytes(source);
            }
        };
        public static final /* enum */ DataStorageUnit MEBIBYTES = new DataStorageUnit("MiB"){

            @Override
            public long toBytes(long d) {
                return 3.x(d, 0x100000L, 0x7FFFFFFFFFFL);
            }

            @Override
            public long toKibibytes(long d) {
                return 3.x(d, 1024L, 0x1FFFFFFFFFFFFFL);
            }

            @Override
            public long toMebibytes(long d) {
                return d;
            }

            @Override
            public long toGibibytes(long d) {
                return d / 1024L;
            }

            @Override
            public long convert(long source, DataStorageUnit sourceUnit) {
                return sourceUnit.toMebibytes(source);
            }
        };
        public static final /* enum */ DataStorageUnit GIBIBYTES = new DataStorageUnit("GiB"){

            @Override
            public long toBytes(long d) {
                return 4.x(d, 0x40000000L, 0x1FFFFFFFFL);
            }

            @Override
            public long toKibibytes(long d) {
                return 4.x(d, 0x100000L, 0x7FFFFFFFFFFL);
            }

            @Override
            public long toMebibytes(long d) {
                return 4.x(d, 1024L, 0x1FFFFFFFFFFFFFL);
            }

            @Override
            public long toGibibytes(long d) {
                return d;
            }

            @Override
            public long convert(long source, DataStorageUnit sourceUnit) {
                return sourceUnit.toGibibytes(source);
            }
        };
        static final long MAX = Long.MAX_VALUE;
        private final String symbol;
        private static final /* synthetic */ DataStorageUnit[] $VALUES;

        public static DataStorageUnit[] values() {
            return (DataStorageUnit[])$VALUES.clone();
        }

        public static DataStorageUnit valueOf(String name) {
            return Enum.valueOf(DataStorageUnit.class, name);
        }

        static long x(long d, long m4, long over) {
            assert (over > 0L && over < 0x7FFFFFFFFFFFFFFEL && over == Long.MAX_VALUE / m4);
            if (d > over) {
                return Long.MAX_VALUE;
            }
            return Math.multiplyExact(d, m4);
        }

        public static DataStorageUnit fromSymbol(String symbol) {
            for (DataStorageUnit value : DataStorageUnit.values()) {
                if (!value.symbol.equalsIgnoreCase(symbol)) continue;
                return value;
            }
            throw new IllegalArgumentException(String.format("Unsupported data storage unit: %s. Supported units are: %s", symbol, Arrays.stream(DataStorageUnit.values()).map(u -> u.symbol).collect(Collectors.joining(", "))));
        }

        private DataStorageUnit(String symbol) {
            this.symbol = symbol;
        }

        public long toBytes(long d) {
            throw new AbstractMethodError();
        }

        public long toKibibytes(long d) {
            throw new AbstractMethodError();
        }

        public long toMebibytes(long d) {
            throw new AbstractMethodError();
        }

        public long toGibibytes(long d) {
            throw new AbstractMethodError();
        }

        public long convert(long source, DataStorageUnit sourceUnit) {
            throw new AbstractMethodError();
        }

        static {
            $VALUES = new DataStorageUnit[]{BYTES, KIBIBYTES, MEBIBYTES, GIBIBYTES};
        }
    }

    public static final class IntMebibytesBound
    extends DataStorageSpec {
        public IntMebibytesBound(String value) {
            super(value, DataStorageUnit.MEBIBYTES, Integer.MAX_VALUE);
        }

        public IntMebibytesBound(long quantity, DataStorageUnit unit) {
            super(quantity, unit, DataStorageUnit.MEBIBYTES, Integer.MAX_VALUE, quantity + unit.symbol);
        }

        public IntMebibytesBound(long mebibytes) {
            this(mebibytes, DataStorageUnit.MEBIBYTES);
        }

        public int toBytes() {
            return Ints.saturatedCast(this.unit().toBytes(this.quantity()));
        }

        public int toKibibytes() {
            return Ints.saturatedCast(this.unit().toKibibytes(this.quantity()));
        }

        public int toMebibytes() {
            return Ints.saturatedCast(this.unit().toMebibytes(this.quantity()));
        }

        public long toBytesInLong() {
            return this.unit().toBytes(this.quantity());
        }
    }

    public static final class LongMebibytesBound
    extends DataStorageSpec {
        public LongMebibytesBound(String value) {
            super(value, DataStorageUnit.MEBIBYTES, Long.MAX_VALUE);
        }

        public LongMebibytesBound(long quantity, DataStorageUnit unit) {
            super(quantity, unit, DataStorageUnit.MEBIBYTES, Long.MAX_VALUE, quantity + unit.symbol);
        }

        public LongMebibytesBound(long mebibytes) {
            this(mebibytes, DataStorageUnit.MEBIBYTES);
        }

        public long toBytes() {
            return this.unit().toBytes(this.quantity());
        }

        public long toKibibytes() {
            return this.unit().toKibibytes(this.quantity());
        }

        public long toMebibytes() {
            return this.unit().toMebibytes(this.quantity());
        }
    }

    public static final class IntKibibytesBound
    extends DataStorageSpec {
        public IntKibibytesBound(String value) {
            super(value, DataStorageUnit.KIBIBYTES, Integer.MAX_VALUE);
        }

        public IntKibibytesBound(long quantity, DataStorageUnit unit) {
            super(quantity, unit, DataStorageUnit.KIBIBYTES, Integer.MAX_VALUE, quantity + unit.symbol);
        }

        public IntKibibytesBound(long kibibytes) {
            this(kibibytes, DataStorageUnit.KIBIBYTES);
        }

        public int toBytes() {
            return Ints.saturatedCast(this.unit().toBytes(this.quantity()));
        }

        public int toKibibytes() {
            return Ints.saturatedCast(this.unit().toKibibytes(this.quantity()));
        }

        public long toBytesInLong() {
            return this.unit().toBytes(this.quantity());
        }
    }

    public static final class IntBytesBound
    extends DataStorageSpec {
        public IntBytesBound(String value) {
            super(value, DataStorageUnit.BYTES, Integer.MAX_VALUE);
        }

        public IntBytesBound(long quantity, DataStorageUnit unit) {
            super(quantity, unit, DataStorageUnit.BYTES, Integer.MAX_VALUE, quantity + unit.symbol);
        }

        public IntBytesBound(long bytes) {
            this(bytes, DataStorageUnit.BYTES);
        }

        public int toBytes() {
            return Ints.saturatedCast(this.unit().toBytes(this.quantity()));
        }
    }

    public static final class LongBytesBound
    extends DataStorageSpec {
        public LongBytesBound(String value) {
            super(value, DataStorageUnit.BYTES, Long.MAX_VALUE);
        }

        public LongBytesBound(long quantity, DataStorageUnit unit) {
            super(quantity, unit, DataStorageUnit.BYTES, Long.MAX_VALUE, quantity + unit.symbol);
        }

        public LongBytesBound(long bytes) {
            this(bytes, DataStorageUnit.BYTES);
        }

        public long toBytes() {
            return this.unit().toBytes(this.quantity());
        }

        public int toMebibytesInt() {
            return Ints.saturatedCast(this.unit().toMebibytes(this.quantity()));
        }
    }
}

